# tests.platform.event_logging.test_linux
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Test the event logging on linux."""

import sys

from twisted.trial.unittest import TestCase

import ubuntuone.platform.event_logging.linux as event_logging


class ZeitgeistDetectionTestCase(TestCase):
    """Test the is_zeitgeist_installed function."""

    def patch_module(self, module_name, fake_module):
        """Monkey patch a module for the duration of the test."""
        UNDEFINED = object()
        real_module = sys.modules.get(module_name, UNDEFINED)

        def restore():
            """Restore the real_module."""
            if real_module is UNDEFINED:
                del(sys.modules[module_name])
            else:
                sys.modules[module_name] = real_module

        self.addCleanup(restore)
        sys.modules[module_name] = fake_module

    def test_zg_installed_returns_true(self):
        """When zg is installed, it returns true."""
        self.patch_module("zeitgeist", object())
        self.patch_module("zeitgeist.mimetypes", object())
        result = event_logging.is_zeitgeist_installed()
        self.assertEqual(result, True)

    def test_zg_not_installed_returns_false(self):
        """When zg is not installed, it returns false."""
        self.patch_module("zeitgeist", None)
        result = event_logging.is_zeitgeist_installed()
        self.assertEqual(result, False)

    def test_old_zg_installed_returns_false(self):
        """When an old zg is installed, it returns false."""
        self.patch_module("zeitgeist", object())
        self.patch_module("zeitgeist.mimetypes", None)
        result = event_logging.is_zeitgeist_installed()
        self.assertEqual(result, False)


class GetListenerTestCase(TestCase):
    """The zg listener is created."""

    def test_zeitgeist_installed_returns_listener(self):
        """get_listener returns a listener if ZG installed."""
        self.patch(event_logging, "is_zeitgeist_installed", lambda: True)
        listener = event_logging.get_listener(None, None)
        self.assertNotEqual(listener, None)

    def test_zeitgeist_not_installed_returns_none(self):
        """get_listener returns None if ZG not installed."""
        self.patch(event_logging, "is_zeitgeist_installed", lambda: False)
        listener = event_logging.get_listener(None, None)
        self.assertEqual(listener, None)
