# -*- coding: utf-8 -*-
#
# Author: Manuel de la Pena<manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""IPC tests on windows."""

from mocker import MockerTestCase

from unittest import TestCase

from ubuntuone.platform.windows.dbus_interface import (
    Config,
    Events,
    Folders,
    FileSystem,
    PublicFiles,
    Shares,
    SignalBroadcaster,
    Status,
    SyncDaemon
)

class PerspectiveBrokerTestCase(TestCase):
    """Base test case for the IPC used on Windows."""

    def setUp(self):
        """Setup tests."""
        super(PerspectiveBrokerTestCase, self).setUp()
        self.config = Config(None, None)
        self.status = Status(None, None, None)
        self.events = Events(None)
        self.sync = SyncDaemon(None, None, None, None)
        self.shares = Shares(None, None, None)
        self.folders = Folders(None, None)
        self.public_files = PublicFiles(None, None)
        self.fs = FileSystem(None, None)

class TestSignalBroadcaster(MockerTestCase):
    """Test the signal brocaster code."""
    
    def setUp(self):
        super(TestSignalBroadcaster, self).setUp()
        self.client = self.mocker.mock()
        self.broad_caster = SignalBroadcaster()
        self.broad_caster.clients.append(self.client)

    def test_remote_register_to_signals(self):
        """Assert that the client was added."""
        self.mocker.replay()
        self.broad_caster.remote_register_to_signals(self.client)
        self.assertTrue(self.client in self.broad_caster.clients)

    def test_emit_signal(self):
        """Assert that the cleitn method was called."""
        first = 1
        second = 2
        word = 'word'
        signal_name = 'on_test'
        self.client.callRemote(signal_name, first, second, word=word)
        self.mocker.replay()
        self.broad_caster.emit_signal(signal_name, first, second, word=word)


class TestStatusEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Tests that the emit methods are correct."""
    
    def setUp(self):
        """Setup tests."""       
        super(TestStatusEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.status.syncdaemon_status = self.mocker.mock()
        self.status.emit_signal = self.signal_method

    def test_emit_content_queue_changed(self):
        """Emit ContentQueueChanged."""
        self.signal_method('on_content_queue_changed')
        self.mocker.replay()
        self.status.emit_content_queue_changed()

    def test_emit_invalid_name(self):
        """Emit InvalidName."""
        dirname = 'dirname'
        filename = 'filename'
        self.signal_method('on_invalid_name', unicode(dirname), str(filename))
        self.mocker.replay()
        self.status.emit_invalid_name(dirname, filename)

    def test_emit_broken_node(self):
        """Emit BrokenNode."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        mdid = 'mdid'
        path = 'path'
        self.signal_method('on_broken_node', volume_id, node_id, mdid,
                           path.decode('utf8'))
        self.mocker.replay()
        self.status.emit_broken_node(volume_id, node_id, mdid, path)

    def test_emit_status_changed(self):
        """Emit StatusChanged."""
        status = 'status'
        self.status.syncdaemon_status.current_status()
        self.mocker.result(status)
        self.signal_method('on_status_changed', status)
        self.mocker.replay()
        self.status.emit_status_changed(status)

    def test_emit_download_started(self):
        """Emit DownloadStarted."""
        download = 'download'
        self.signal_method('on_download_started', download)
        self.mocker.replay()
        self.status.emit_download_started(download)

    def test_emit_download_file_progress(self):
        """Emit DownloadFileProgress."""
        download = 'download'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_download_file_progress', download, string_info)
        self.mocker.replay()
        self.status.emit_download_file_progress(download, test=2, name=3)

    def test_emit_download_finished(self):
        """Emit DownloadFinished."""
        download = 'download'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_download_finished', download, string_info)
        self.mocker.replay()
        self.status.emit_download_finished(download, test=2, name=3)

    def test_emit_upload_started(self):
        """Emit UploadStarted."""
        upload = 'upload'
        self.signal_method('on_upload_started', upload)
        self.mocker.replay()
        self.status.emit_upload_started(upload)

    def test_emit_upload_file_progress(self):
        """Emit UploadFileProgress."""
        upload = 'upload'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_upload_file_progress', upload, string_info)
        self.mocker.replay()
        self.status.emit_upload_file_progress(upload, test=2, name=3)

    def test_emit_upload_finished(self):
        """Emit UploadFinished."""
        upload = 'upload'
        string_info = {'test':'2', 'name':'3'}
        self.signal_method('on_upload_finished', upload, string_info)
        self.mocker.replay()
        self.status.emit_upload_finished(upload, test=2, name=3)

    def test_emit_account_changed(self):
        """Emit AccountChanged."""
        bytes = '5'
        account_info = self.mocker.mock()
        account_info.purchased_bytes
        self.mocker.result(bytes)
        info_dict = {'purchased_bytes' : unicode('5')}
        self.signal_method('on_account_changed', info_dict)
        self.mocker.replay()
        self.status.emit_account_changed(account_info)

    def test_emit_metaqueue_changed(self):
        """Emit MetaQueueChanged."""
        self.signal_method('on_metaqueue_changed')
        self.mocker.replay()
        self.status.emit_metaqueue_changed()


class TestEventsEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""       
        super(TestEventsEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.events.emit_signal = self.signal_method

    def test_emit_event(self):
        """Test the Event signal."""
        items = {1:2}
        event = self.mocker.mock()
        event.iteritems()
        self.mocker.result(items.iteritems())
        self.signal_method('on_event', {'1':'2'})
        self.mocker.replay()
        self.events.emit_event(event)


class TestSyncDaemonEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""
    
    def setUp(self):
        """Setup tests."""
        super(TestSyncDaemonEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.sync.emit_signal = self.signal_method

    def test_emit_root_mismatch(self):
        """Emit RootMismatch signal."""
        root_id = 'root_id'
        new_root_id = 'new_root_id'
        self.signal_method('on_root_mismatch', root_id, new_root_id)
        self.mocker.replay()
        self.sync.emit_root_mismatch(root_id, new_root_id)

    def test_emit_quota_exceeded(self):
        """Emit QuotaExceeded signal."""
        volume_dict = 'volume_dict'
        self.signal_method('on_quota_exceeded', volume_dict)
        self.mocker.replay()
        self.sync.emit_quota_exceeded(volume_dict)


class TestSharesEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""       
        super(TestSharesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.shares.emit_signal = self.signal_method
        self.shares.syncdaemon_shares = self.mocker.mock()
        self.get_share_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_share_dict')

    def test_emit_share_changed_deleted(self):
        """Test emit share changed on a delete event."""
        message = 'deleted'
        share = 'share'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_deleted', share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_changed_changed(self):
        """Test emit share changed on a changed event."""
        message = 'changed'
        share = 'share'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_changed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_changed(message, share)

    def test_emit_share_delete_error(self):
        """Emit ShareDeleteError signal."""
        share = 'share'
        error = 'error'
        share_dict = 'share_dict'
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_deleted_error', share_dict, error)
        self.mocker.replay()
        self.shares.emit_share_delete_error(share, error)

    def test_emit_free_space(self):
        """Emit ShareChanged when free space changes """
        free_bytes = '0'
        share_dict = shares = {'1': 'share', 'free_bytes':'0' }
        share = 'share'
        share_id = '1'
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.shares.syncdaemon_shares.shares
        self.mocker.result(shares)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_changed', share_dict)
        self.mocker.replay()
        self.shares.emit_free_space(share_id, free_bytes)

    def test_emit_share_created(self):
        """Emit ShareCreated signal """
        share_info = 'info'
        self.signal_method('on_share_created', share_info)
        self.mocker.replay()
        self.shares.emit_share_created(share_info)

    def test_emit_share_create_error(self):
        """Emit ShareCreateError signal."""
        share_info = 'id'
        error = 'error'
        info = 'info'
        self.shares.syncdaemon_shares.get_create_error_share_info(share_info)
        self.mocker.result(info)
        self.signal_method('on_share_create_error', info, error)
        self.mocker.replay()
        self.shares.emit_share_create_error(share_info, error)

    def test_emit_share_answer_response(self):
        """Emits ShareAnswerResponse signal."""
        share_id = 'id'
        answer = 'yes'
        error = 'boom'
        answer_info = dict(volume_id=share_id, answer=answer, error=error)
        self.signal_method('on_share_answer_response', answer_info)
        self.mocker.replay()
        self.shares.emit_share_answer_response(share_id, answer, error)

    def test_emit_new_share(self):
        """Emits NewShare signal."""
        share_id = 'id'
        share = 'share'
        share_dict = {'share':'id'}
        self.shares.syncdaemon_shares.get_volume(share_id)
        self.mocker.result(share)
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_new_share', share_dict)
        self.mocker.replay()
        self.shares.emit_new_share(share_id)

    def test_emit_share_subscribed(self):
        """Emit the ShareSubscribed signal"""
        share = 'share'
        share_dict = {'share' : 'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict)
        self.signal_method('on_share_subscribed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_subscribed(share)

    def test_emit_share_subscribe_error(self):
        """Emit the ShareSubscribeError signal"""
        share_id = 'id'
        error = 'error'
        self.signal_method('on_share_subscribed_error',
                           {'id': share_id}, str(error))
        self.mocker.replay()
        self.shares.emit_share_subscribe_error(share_id, error)

    def test_emit_share_unsubscribed(self):
        """Emit the ShareUnSubscribed signal"""
        share = 'share'
        share_dict = {'share':'id'}
        self.get_share_dict(share)
        self.mocker.result(share_dict) 
        self.signal_method('on_share_unsubscribed', share_dict)
        self.mocker.replay()
        self.shares.emit_share_unsubscribed(share)

    def test_emit_share_unsubscribe_error(self):
        """Emit the ShareUnSubscribeError signal"""
        share_id = 'id'
        error = 'error'
        self.signal_method('on_share_unsubscribed_error',{'id': share_id}, str(error))
        self.mocker.replay()
        self.shares.emit_share_unsubscribe_error(share_id, error)


class TestFoldersEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestFoldersEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.folders.emit_signal = self.signal_method
        self.get_udf_dict = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.get_udf_dict')

    def test_emit_folder_deleted(self):
        """Emit the FolderCreated signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_deleted', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_deleted(folder)

    def test_emit_folder_delete_error(self):
        """Emit the FolderCreateError signal"""
        folder = 'folder'
        error = 'error'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_delete_error', udf_dict, str(error))
        self.mocker.replay()
        self.folders.emit_folder_delete_error(folder, error)

    def test_emit_folder_subscribed(self):
        """Emit the FolderSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_subscribed', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_subscribed(folder)

    def test_emit_folder_subscribe_error(self):
        """Emit the FolderSubscribeError signal"""
        folder_id = 'id'
        error = 'error'
        self.signal_method('on_folder_subscribe_error',
                           {'id':folder_id}, str(error))
        self.mocker.replay()
        self.folders.emit_folder_subscribe_error(folder_id, error)

    def test_emit_folder_unsubscribed(self):
        """Emit the FolderUnSubscribed signal"""
        folder = 'folder'
        udf_dict = {'udf':'id'}
        self.get_udf_dict(folder)
        self.mocker.result(udf_dict)
        self.signal_method('on_folder_unsubscribed', udf_dict)
        self.mocker.replay()
        self.folders.emit_folder_unsubscribed(folder)

    def test_emit_folder_unsubscribe_error(self):
        """Emit the FolderUnSubscribeError signal"""
        folder_id = 'id'
        error = 'error'
        self.signal_method('on_folder_unsubscribe_error',
                           {'id':folder_id}, str(error))
        self.mocker.replay()
        self.folders.emit_folder_unsubscribe_error(folder_id, error)


class TestPublicFilesEmitSignals(PerspectiveBrokerTestCase, MockerTestCase):
    """Test that the emit method have been correctly implemented."""

    def setUp(self):
        """Setup tests."""
        super(TestPublicFilesEmitSignals, self).setUp()
        self.signal_method = self.mocker.mock()
        self.public_files.emit_signal = self.signal_method
        self.public_files.syncdaemon_public_files = self.mocker.mock()
        self.bool_str = self.mocker.replace(
            'ubuntuone.syncdaemon.interaction_interfaces.bool_str')

    def test_emit_public_access_changed(self): 
        """Emit the PublicAccessChanged signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        path = 'path'
        is_public = True
        public_url = 'url' 
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.bool_str(is_public)
        self.mocker.result('True')
        self.signal_method('on_public_access_changed',
                           dict(share_id=share_id, node_id=node_id,
                                is_public='True', public_url=public_url,
                                path=path))
        self.mocker.replay()
        self.public_files.emit_public_access_changed(share_id, node_id, 
                                                     is_public, public_url)

    def test_emit_public_access_change_error(self):
        """Emit the PublicAccessChangeError signal."""
        share_id = 'share_id'
        node_id = 'node_id'
        error = 'error'
        path = 'path'
        self.public_files.syncdaemon_public_files.get_path(share_id, node_id)
        self.mocker.result(path)
        self.signal_method('on_public_access_change_error',
                           dict(share_id=share_id, node_id=node_id, path=path),
                           error)
        self.mocker.replay()
        self.public_files.emit_public_access_change_error(share_id, node_id,
                                                          error)

    def test_emit_public_files_list(self):
        """Emit the PublicFilesList signal."""
        volume_id = 'volume_id'
        node_id = 'node_id'
        public_url = 'url'
        path = 'path'
        public_files = [dict(volume_id=volume_id, node_id=node_id,
                             public_url=public_url)]
        files = [dict(volume_id=volume_id, node_id=node_id,
                      public_url=public_url, path=path)]
        self.public_files.syncdaemon_public_files.get_path(volume_id, node_id)
        self.mocker.result(path)
        self.signal_method('on_public_files_list',files)
        self.mocker.replay()
        self.public_files.emit_public_files_list(public_files)

    def test_emit_public_files_list_error(self):
        """Emit the PublicFilesListError signal."""
        error = 'error'
        self.signal_method('on_public_files_list_error', error)
        self.mocker.replay()
        self.public_files.emit_public_files_list_error(error)

