#
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Specific tests for the os_helper on Windows."""

import os

from win32api import GetUserName

from ntsecuritycon import (
    FILE_ALL_ACCESS,
    FILE_GENERIC_READ,
    FILE_GENERIC_WRITE,
)
from ubuntuone.platform.windows.os_helper import (
    _set_file_attributes,
    access,
    set_no_rights
)
from contrib.testing.testcase import BaseTwistedTestCase

class OSWrapperTests(BaseTwistedTestCase):
    """Test specific windows implementation details."""
    
    def setUp(self):
        """Setup for the tests."""
        super(OSWrapperTests, self).setUp()
        self.basedir = self.mktemp('test_root')
        self.testfile = os.path.join(self.basedir, "test_file")
        self.fd = open(self.testfile, 'w')

    def tearDown(self):
        """Clean the env for the next test."""
        self.fd.close();
        self.rmtree(self.basedir)
        super(OSWrapperTests, self).tearDown()

    def test_access_no_rights(self):
        """Test when the sid is not present."""
        # remove all the rights from the test file so that
        # we cannot read or write
        set_no_rights(self.testfile)
        self.assertFalse(access(self.testfile))

    def test_access_read_write_user(self):
        """Test when the user sid has rw rights."""
        # set the file to be read and write just by the user
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_read_write_everyone(self):
        """Test when the everyone sid has rw rights."""
        groups = {}
        groups['Everyone'] = FILE_GENERIC_READ | FILE_GENERIC_WRITE
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))
        
    def test_access_write_user_everyone_read(self):
        """Test when the user sid has w rights."""
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_WRITE
        groups['Everyone'] = FILE_GENERIC_READ
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_write_everyone_user_read(self):
        """Test when the everyone sid has w rights"""
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_READ 
        groups['Everyone'] = FILE_GENERIC_WRITE
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_write_user_everyone(self):
        """Test when everyone and user have w rights."""
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_WRITE
        groups['Everyone'] = FILE_GENERIC_WRITE
        _set_file_attributes(self.testfile, groups)
        self.assertFalse(access(self.testfile))

    def test_access_read_user(self):
        """Test when the sid has r rights."""
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_READ
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_read_everyone(self):
        """Test when everyone has r rights."""
        groups = {}
        groups['Everyone'] = FILE_GENERIC_READ
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))
    
    def test_access_read_user_everyone(self):
        """Test when user and everyone have r rights."""
        groups = {}
        groups[GetUserName()] = FILE_GENERIC_READ
        groups['Everyone'] = FILE_GENERIC_READ
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_full_user(self):
        """Test when the sid has full control."""
        groups = {}
        groups[GetUserName()] = FILE_ALL_ACCESS
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))

    def test_access_full_everyone(self):
        """test when everyone has full control."""
        groups = {}
        groups['Everyone'] = FILE_ALL_ACCESS
        _set_file_attributes(self.testfile, groups)
        self.assertTrue(access(self.testfile))
