# tests.platform.linux - linux platform tests
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Linux specific tests for the platform module."""

import errno
import logging
import os

import gio

from ubuntuone.devtools.handlers import MementoHandler

from contrib.testing.testcase import BaseTwistedTestCase
from ubuntuone.platform import (
    access,
    allow_writes,
    listdir,
    make_dir,
    move_to_trash,
    open_file,
    path_exists,
    remove_dir,
    remove_file,
    rename,
    set_dir_readonly,
    set_dir_readwrite,
    set_file_readonly,
    set_file_readwrite,
    stat_path,
)

class FakeGIOFile(object):
    """Fake File for gio."""

    _bad_trash_call = None

    def __init__(self, path):
        pass

    def trash(self):
        """Fake trash call."""
        return self._bad_trash_call


class OSWrapperTests(BaseTwistedTestCase):
    """Tests for os wrapper functions."""

    def setUp(self):
        """Set up."""
        BaseTwistedTestCase.setUp(self)
        self.basedir = self.mktemp('test_root')
        self.testfile = os.path.join(self.basedir, "test_file")
        open(self.testfile, 'w').close()

        self.handler = MementoHandler()
        self.handler.setLevel(logging.DEBUG)
        self._logger = logging.getLogger('ubuntuone.SyncDaemon')
        self._logger.addHandler(self.handler)

    def tearDown(self):
        """Tear down."""
        self._logger.removeHandler(self.handler)
        self.rmtree(self.basedir)
        BaseTwistedTestCase.tearDown(self)

    def test_set_dir_readonly(self):
        """Test for set_dir_readonly."""
        set_dir_readonly(self.basedir)
        self.assertRaises(OSError, os.mkdir, os.path.join(self.basedir, 'foo'))

    def test_set_dir_readwrite(self):
        """Test for set_dir_readwrite."""
        set_dir_readonly(self.basedir)
        self.assertRaises(OSError, os.mkdir, os.path.join(self.basedir, 'foo'))
        set_dir_readwrite(self.basedir)
        os.mkdir(os.path.join(self.basedir, 'foo'))

    def test_allow_writes(self):
        """Test for allow_writes."""
        set_dir_readonly(self.basedir)
        self.assertRaises(OSError, os.mkdir, os.path.join(self.basedir, 'foo'))
        with allow_writes(self.basedir):
            os.mkdir(os.path.join(self.basedir, 'foo'))

    def test_set_file_readonly(self):
        """Test for set_file_readonly."""
        set_file_readonly(self.testfile)
        self.assertRaises(IOError, open, self.testfile, 'w')

    def test_set_file_readwrite(self):
        """Test for set_file_readwrite."""
        set_file_readonly(self.testfile)
        self.assertRaises(IOError, open, self.testfile, 'w')
        set_file_readwrite(self.testfile)
        open(self.testfile, 'w')

    def test_path_file_exist_yes(self):
        """Test that the file exists."""
        self.assertTrue(path_exists(self.testfile))

    def test_path_file_exist_no(self):
        """Test that the file doesn't exist."""
        os.remove(self.testfile)
        self.assertFalse(path_exists(self.testfile))

    def test_path_dir_exist_yes(self):
        """Test that the dir exists."""
        self.assertTrue(path_exists(self.basedir))

    def test_path_dir_exist_no(self):
        """Test that the dir doesn't exist."""
        os.remove(self.testfile)
        self.assertFalse(path_exists(os.path.join(self.basedir, 'nodir')))

    def test_remove_file(self):
        """Test the remove file."""
        remove_file(self.testfile)
        self.assertFalse(os.path.exists(self.testfile))

    def test_remove_dir(self):
        """Test the remove dir."""
        testdir = os.path.join(self.basedir, 'foodir')
        os.mkdir(testdir)
        assert os.path.exists(testdir)
        remove_dir(testdir)
        self.assertFalse(path_exists(testdir))

    def test_make_dir_one(self):
        """Test the make dir with one dir."""
        testdir = os.path.join(self.basedir, 'foodir')
        assert not os.path.exists(testdir)
        make_dir(testdir)
        self.assertTrue(os.path.exists(testdir))

    def test_make_dir_already_there(self):
        """Test the make dir with one dir that exists."""
        self.assertRaises(OSError, make_dir, self.basedir)

    def test_make_dir_recursive_no(self):
        """Test the make dir with some dirs, not recursive explicit."""
        testdir = os.path.join(self.basedir, 'foo', 'bar')
        assert not os.path.exists(testdir)
        self.assertRaises(OSError, make_dir, testdir)

    def test_make_dir_recursive_yes(self):
        """Test the make dir with some dirs, recursive."""
        testdir = os.path.join(self.basedir, 'foo', 'bar')
        assert not os.path.exists(testdir)
        make_dir(testdir, recursive=True)
        self.assertTrue(os.path.exists(testdir))

    def test_open_file_not_there(self):
        """Open a file that does not exist."""
        self.assertRaises(IOError, open_file, os.path.join(self.basedir, 'no'))

    def test_open_file_gets_a_fileobject(self):
        """Open a file, and get a file object."""
        testfile = os.path.join(self.basedir, 'testfile')
        open(testfile, 'w').close()
        f = open_file(testfile)
        self.assertTrue(isinstance(f, file))

    def test_open_file_read(self):
        """Open a file, and read."""
        testfile = os.path.join(self.basedir, 'testfile')
        with open(testfile, 'w') as fh:
            fh.write("foo")
        f = open_file(testfile, 'r')
        self.assertTrue(f.read(), "foo")

    def test_open_file_write(self):
        """Open a file, and write."""
        testfile = os.path.join(self.basedir, 'testfile')
        with open_file(testfile, 'w') as fh:
            fh.write("foo")
        f = open(testfile)
        self.assertTrue(f.read(), "foo")

    def test_rename_not_there(self):
        """Rename something that does not exist."""
        self.assertRaises(OSError, rename,
                          os.path.join(self.basedir, 'no'), 'foo')

    def test_rename_file(self):
        """Rename a file."""
        testfile1 = os.path.join(self.basedir, 'testfile1')
        testfile2 = os.path.join(self.basedir, 'testfile2')
        open(testfile1, 'w').close()
        rename(testfile1, testfile2)
        self.assertFalse(os.path.exists(testfile1))
        self.assertTrue(os.path.exists(testfile2))

    def test_rename_dir(self):
        """Rename a dir."""
        testdir1 = os.path.join(self.basedir, 'testdir1')
        testdir2 = os.path.join(self.basedir, 'testdir2')
        os.mkdir(testdir1)
        rename(testdir1, testdir2)
        self.assertFalse(os.path.exists(testdir1))
        self.assertTrue(os.path.exists(testdir2))

    def test_listdir(self):
        """Return a list of the files in a dir."""
        open(os.path.join(self.basedir, 'foo'), 'w').close()
        open(os.path.join(self.basedir, 'bar'), 'w').close()
        l = listdir(self.basedir)
        self.assertEqual(sorted(l), ['bar', 'foo', 'test_file'])

    def test_access_rw(self):
        """Test access on a file with full permission."""
        self.assertTrue(access(self.testfile))

    def test_access_ro(self):
        """Test access on a file with read only permission."""
        os.chmod(self.testfile, 0o444)
        self.assertTrue(access(self.testfile))
        os.chmod(self.testfile, 0o664)

    def test_access_nothing(self):
        """Test access on a file with no permission at all."""
        os.chmod(self.testfile, 0o000)
        self.assertFalse(access(self.testfile))
        os.chmod(self.testfile, 0o664)

    def test_stat_normal(self):
        """Test on a normal file."""
        self.assertEqual(os.stat(self.testfile), stat_path(self.testfile))

    def test_stat_symlink(self):
        """Test that it doesn't follow symlinks.

        We compare the inode only (enough to see if it's returning info
        from the link or the linked), as we can not compare the full stat
        because the st_mode will be different.
        """
        link = os.path.join(self.basedir, 'foo')
        os.symlink(self.testfile, link)
        self.assertNotEqual(os.stat(link).st_ino, stat_path(link).st_ino)
        self.assertEqual(os.lstat(link).st_ino, stat_path(link).st_ino)

    def test_stat_no_path(self):
        """Test that it raises proper error when no file is there."""
        try:
            return stat_path(os.path.join(self.basedir, 'nofile'))
        except OSError, e:
            self.assertEqual(e.errno, errno.ENOENT)

    def test_path_exists_file_yes(self):
        """The file is there."""
        self.assertTrue(path_exists(self.testfile))

    def test_path_exists_file_no(self):
        """The file is not there."""
        os.remove(self.testfile)
        self.assertFalse(path_exists(self.testfile))

    def test_path_exists_dir_yes(self):
        """The dir is there."""
        self.assertTrue(path_exists(self.basedir))

    def test_path_exists_dir_no(self):
        """The dir is not there."""
        self.rmtree(self.basedir)
        self.assertFalse(path_exists(self.basedir))

    def test_movetotrash_file_ok(self):
        """Move a file to trash ok.

        Just check it was removed because can't monkeypatch gio.File.trash
        to see that that was actually called.  Note that the gio
        infrastructure is used is obvious after the other trash tests here.
        """
        path = os.path.join(self.basedir, 'foo')
        open(path, 'w').close()
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))

    def test_movetotrash_dir_ok(self):
        """Move a dir to trash ok.

        Just check it was removed because can't monkeypatch gio.File.trash
        to see that that was actually called.  Note that the gio
        infrastructure is used is obvious after the other trash tests here.
        """
        path = os.path.join(self.basedir, 'foo')
        os.mkdir(path)
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))

    def test_movetotrash_file_bad(self):
        """Something bad happen when moving to trash, removed anyway."""
        FakeGIOFile._bad_trash_call = False   # error
        self.patch(gio, "File", FakeGIOFile)
        path = os.path.join(self.basedir, 'foo')
        open(path, 'w').close()
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))
        self.assertTrue(self.handler.check_warning("Problems moving to trash!",
                                                   "Removing anyway", "foo"))

    def test_movetotrash_dir_bad(self):
        """Something bad happen when moving to trash, removed anyway."""
        FakeGIOFile._bad_trash_call = False   # error
        self.patch(gio, "File", FakeGIOFile)
        path = os.path.join(self.basedir, 'foo')
        os.mkdir(path)
        open(os.path.join(path, 'file inside directory'), 'w').close()
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))
        self.assertTrue(self.handler.check_warning("Problems moving to trash!",
                                                   "Removing anyway", "foo"))


    def test_movetotrash_file_systemnotcapable(self):
        """The system is not capable of moving into trash."""
        FakeGIOFile._bad_trash_call = gio.ERROR_NOT_SUPPORTED
        self.patch(gio, "File", FakeGIOFile)
        path = os.path.join(self.basedir, 'foo')
        open(path, 'w').close()
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))
        self.assertTrue(self.handler.check_warning("Problems moving to trash!",
                                                   "Removing anyway", "foo",
                                                   "ERROR_NOT_SUPPORTED"))

    def test_movetotrash_dir_systemnotcapable(self):
        """The system is not capable of moving into trash."""
        FakeGIOFile._bad_trash_call = gio.ERROR_NOT_SUPPORTED
        self.patch(gio, "File", FakeGIOFile)
        path = os.path.join(self.basedir, 'foo')
        os.mkdir(path)
        open(os.path.join(path, 'file inside directory'), 'w').close()
        move_to_trash(path)
        self.assertFalse(os.path.exists(path))
        self.assertTrue(self.handler.check_warning("Problems moving to trash!",
                                                   "Removing anyway", "foo",
                                                   "ERROR_NOT_SUPPORTED"))

    def test_movetotrash_notthere(self):
        """Try to move to trash something that is not there."""
        path = os.path.join(self.basedir, 'notthere')
        e = self.assertRaises(OSError, move_to_trash, path)
        self.assertEqual(e.errno, errno.ENOENT)
