/*
 * pam_xdg_support.c: PAM module to provide integration for XDG_RUNTIME_DIR
 * and related user session management services
 *
 * Copyright (C) 2012 Canonical Ltd.
 * Author: Steve Langasek <steve.langasek@canonical.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <pwd.h>
#include <syslog.h>

#define PAM_SM_SESSION
#include <security/pam_modules.h>
#include <security/pam_ext.h>
#include <security/pam_modutil.h>

#define XDG_RUNTIME_DIR_BASE "/run/user"

static int unlink_tree (pam_handle_t *pamh, const char *dir)
{
	char *command = NULL;

	/* Cheese out for now and just use rm -rf. */
	if (asprintf (&command, "rm -rf %s", dir) < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Allocation failure");
		return PAM_SESSION_ERR;
	}
	if (system (command) != 0)
	{
		pam_syslog (pamh, LOG_ERR, "Could not delete directory %s",
		            dir);
		return PAM_SESSION_ERR;
	}
}


PAM_EXTERN int
pam_sm_open_session (pam_handle_t *pamh, int flags,
                     int argc, const char **argv)
{
	struct stat statbuf;
	struct passwd *pwd;
	uid_t uid;
	int fd, res, session_count = 0;
	FILE *handle;
	const char *user;
	char *env = NULL, *lockfile = NULL, *rundir = NULL;
	const char *basedir = XDG_RUNTIME_DIR_BASE;

	/*
	 * FIXME: support making the directory base configurable via a module
	 * argument
	 */

	/* If we're not root, this is a no-op. */
	if (geteuid () != 0)
	{
		pam_syslog (pamh, LOG_INFO, "Called as non-root.");
		return PAM_IGNORE;
	}

	/* Verify that the parent directory already exists. */
	if (stat (basedir, &statbuf) != 0 || !S_ISDIR (statbuf.st_mode))
	{
		pam_syslog (pamh, LOG_ERR, "Directory does not exist: %s",
		            basedir);
		return PAM_IGNORE;
	}

	/* Get the username, needed for the path. */
	if (pam_get_user (pamh, &user, NULL) != PAM_SUCCESS)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Could not determine user for session");
		return PAM_SESSION_ERR;
	}

	/* Get the user's uid so we can set proper perms on the directory */
	pwd = pam_modutil_getpwnam (pamh, user);
	if (!pwd)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Could not find password entry for user %s", user);
		return PAM_SESSION_ERR;
	}
	uid = pwd->pw_uid;

	/* open our lock file */
	if (asprintf (&lockfile, "%s/.%s.lock", basedir, user) < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Allocation failure");
		return PAM_SESSION_ERR;
	}

	fd = open (lockfile, O_RDWR | O_CREAT, 0600);
	if (fd < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Could not open lockfile %s: %m",
		            lockfile);
		goto err;
	}

	while ((res = flock (fd, LOCK_EX)) < 0 && errno == EINTR);
	if (res < 0)
	{
		pam_syslog (pamh, LOG_NOTICE, "Could not get lock on %s: %m",
		            lockfile);
		goto err;
	}

	handle = fdopen (fd, "r+");
	if (!handle)
	{
		pam_syslog (pamh, LOG_ERR, "fdopen: %m");
		goto err;
	}

	/* Any errors here are treated as zero existing sessions.
	 * This may give undesirable results in some unusual error cases.
	 */
	while (fscanf (handle, "%u", &session_count) == EOF && errno == EINTR);

	/* Increment our refcount. */
	session_count++;

	/* create the directory if necessary */
	if (asprintf (&rundir, "%s/%s", basedir, user) < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Allocation failure");
		goto err;
	}
	if (mkdir (rundir, 0700) < 0 && errno != EEXIST)
	{
		pam_syslog (pamh, LOG_ERR, "Failed to create directory %s: %m",
		            rundir);
		goto err;
	}
	if (chown (rundir, uid, 0) < 0)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Failed to set owner of %s to %d: %m", rundir, uid);
		goto err;
	}

	free (rundir);
	rundir = NULL;

	/* update the refcount and close the lock file */
	rewind (handle);
	while (ftruncate (fd, 0) < 0 && errno == EINTR);
	if (fprintf (handle, "%u\n", session_count) < 0 || fflush (handle) != 0)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Failed to increment session counter %s", lockfile);
		/* We have the directory, so even though we hit a failure here
		 * we go ahead and use it rather than leaving XDG_RUNTIME_DIR
		 * unset.
		 */
	}

	close (fd);
	free (lockfile);
	lockfile = NULL;

	/* Don't decrement our refcount on error; close_session() will do
	 * that regardless, and we don't want to throw off the count.
	 */
	if (asprintf (&env, "XDG_RUNTIME_DIR=%s/%s", basedir, user) >= 0)
	{
		pam_putenv (pamh, env);
		free (env);
	}
	return PAM_SUCCESS;

err:
	free (rundir);
	free (lockfile);
	close (fd);
	return PAM_SESSION_ERR;

}


PAM_EXTERN int
pam_sm_close_session (pam_handle_t *pamh, int flags,
                      int argc, const char **argv)
{
	struct stat statbuf;
	const char *basedir = XDG_RUNTIME_DIR_BASE;
	const char *user;
	char *lockfile = NULL, *rundir = NULL;
	int fd, res, session_count = 1;
	FILE *handle;

	/* If we're not root, this is a no-op. */
	if (geteuid () != 0)
	{
		pam_syslog (pamh, LOG_INFO, "Called as non-root.");
		return PAM_IGNORE;
	}

	/* Verify that the parent directory already exists. */
	if (stat (basedir, &statbuf) != 0 || !S_ISDIR (statbuf.st_mode))
	{
		pam_syslog (pamh, LOG_ERR, "Directory does not exist: %s", basedir);
		return PAM_IGNORE;
	}

	/* Get the username, needed for the path. */
	if (pam_get_user (pamh, &user, NULL) != PAM_SUCCESS)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Could not determine user for session");
		return PAM_SESSION_ERR;
	}

	/* open our lock file */
	if (asprintf (&lockfile, "%s/.%s.lock", basedir, user) < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Allocation failure");
		goto err;
	}
	if (asprintf (&rundir, "%s/%s", basedir, user) < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Allocation failure");
		goto err;
	}

	/* On shutdown, if the file doesn't already exist, just remove
	 * the directory.
	 */
	fd = open (lockfile, O_RDWR);
	if (fd < 0)
	{
		pam_syslog (pamh, LOG_ERR, "Could not open lockfile %s: %m",
		            lockfile);
		unlink_tree (pamh, rundir);
		unlink (lockfile);
		goto err;
	}

	while ((res = flock (fd, LOCK_EX)) < 0 && errno == EINTR);
	if (res < 0)
	{
		pam_syslog (pamh, LOG_NOTICE, "Could not get lock on %s: %m",
		            lockfile);
		goto err;
	}

	handle = fdopen (fd, "r+");
	if (!handle)
	{
		pam_syslog (pamh, LOG_ERR, "fdopen: %m");
		goto err;
	}

	/* Any errors here are treated as zero existing sessions.
	 * This may give undesirable results in some unusual error cases.
	 */
	while (fscanf (handle, "%u", &session_count) == EOF && errno == EINTR);

	/* Decrement our refcount. */
	session_count--;

	if (session_count <= 0)
	{
		unlink_tree (pamh, rundir);
		unlink (lockfile);
		free (rundir);
		free (lockfile);
		close (fd);
		return PAM_SUCCESS;
	}

	/* update the refcount and close the lock file */
	rewind (handle);
	while (ftruncate (fd, 0) < 0 && errno == EINTR);
	if (fprintf (handle, "%u\n", session_count) < 0 || fflush (handle) != 0)
	{
		pam_syslog (pamh, LOG_ERR,
		            "Failed to decrement session counter %s", lockfile);
		goto err;
	}

	close (fd);
	free (rundir);
	free (lockfile);
	return PAM_SUCCESS;

err:
	free (rundir);
	free (lockfile);
	close (fd);
	return PAM_SESSION_ERR;
}
