/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CompilerBase_p.h"

#include <map>

#include "ConvertCenter_p.h"
#include "Debug.h"
#include "CompilationMessage.h"
#include "ScopedName.h"
#include "Function.h"
#include "GTLCore/Function_p.h"
#include "Value.h"
#include "Macros_p.h"
#include "ModuleData_p.h"

#include <cstdarg>
#include "CompilationMessages.h"
#include "CompilationMessages_p.h"
#include "OperatorOverloadCenter_p.h"

using namespace GTLCore;

struct CompilerBase::Private {
  GTLCore::String moduleName;
  CompilationMessages compilationMessages;
  std::map< GTLCore::ScopedName, std::list< GTLCore::Function* > > functions;
  std::list<GTLCore::Function*> functionsToDelete;
  ConvertCenter* convertCenter;
  OperatorOverloadCenter* operatorOverloadCenter;
  ModuleData* moduleData;
  llvm::Module* llvmModule;
};

CompilerBase::CompilerBase() : d(new Private)
{
  d->convertCenter = new ConvertCenter;
  d->operatorOverloadCenter = new OperatorOverloadCenter;
}
CompilerBase::~CompilerBase()
{
  delete d->convertCenter;
  delete d->operatorOverloadCenter;
  delete d;
}

void CompilerBase::appendError( const GTLCore::CompilationMessage& _msg)
{
  GTLCore::CompilationMessage msg = _msg;
  if( msg.fileName() == "")
  {
    msg.setFileName( d->moduleName );
  }
  GTL_DEBUG( msg.fileName() << " at " << msg.line() << ": " << msg.message() );
  d->compilationMessages.d->appendMessage(msg);
}

void CompilerBase::appendError( GTLCore::CompilationMessage* msg)
{
  GTL_ASSERT(msg);
  if( msg->fileName() == "")
  {
    GTL_ASSERT(d);
    msg->setFileName( d->moduleName );
  }
  GTL_DEBUG( msg->fileName() << " at " << msg->line() << ": " << msg->message() );
  d->compilationMessages.d->appendMessage(*msg);
  delete msg;
}

void CompilerBase::appendErrors( const std::list<GTLCore::CompilationMessage>& _msgs)
{
  foreach(CompilationMessage msg, _msgs)
  {
    d->compilationMessages.d->appendMessage(msg);
  }
}

const CompilationMessages& CompilerBase::errorMessages() const
{
  return d->compilationMessages;
}

void CompilerBase::setModuleName( const GTLCore::String& _moduleName)
{
  d->moduleName = _moduleName;
}
const GTLCore::String& CompilerBase::moduleName() const
{
  return d->moduleName;
}

const std::list<GTLCore::Function*>* CompilerBase::function( const GTLCore::ScopedName&  name)
{
  GTL_DEBUG( "Look for function: " << name );
  std::map< GTLCore::ScopedName, std::list< GTLCore::Function* > >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    if( name.nameSpace() == "" )
    {
      GTL_DEBUG(" Available functions");
      for( std::map< GTLCore::ScopedName, std::list< GTLCore::Function* > >::iterator it2 = d->functions.begin(); it2 != d->functions.end(); ++it2)
      {
        GTL_DEBUG( it2->first );
      }
      return 0;
    }
    GTLCore::ScopedName  globalName( "", name.name() );
    return function( globalName );
  } else {
    return &it->second; // TODO check for the best function to call
  }
}

bool CompilerBase::declareFunction( const GTLCore::ScopedName& name, GTLCore::Function* func)
{
  std::map< GTLCore::ScopedName, std::list< GTLCore::Function*> >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    std::list<  GTLCore::Function* > funcs;
    funcs.push_back(func);
    d->functions[name] = funcs;
  } else {
    foreach( GTLCore::Function* function, it->second)
    {
      if( function->returnType() == func->returnType() and function->parameters().size() == func->parameters().size() )
      {
        bool allthesame = true;
        for( std::size_t i = 0; i < function->parameters().size(); ++i)
        {
          const GTLCore::Parameter& param1 = function->parameters()[i];
          const GTLCore::Parameter& param2 = func->parameters()[i];
          if( param1.type() != param2.type())
          {
            allthesame = false;
            break;
          }
        }
        if(allthesame) return false;
      }
    }
    
    it->second.push_back(func);
    GTL_ASSERT(it->second.size() > 1 );
  }
    return true;
}

bool CompilerBase::declareConstant( const GTLCore::ScopedName& _name, const GTLCore::Type* _type, const GTLCore::Value& _value)
{
  return d->moduleData->appendConstant(_name, _type, _value);
}

ConvertCenter* CompilerBase::convertCenter()
{
  return d->convertCenter;
}

OperatorOverloadCenter* CompilerBase::operatorOverloadCenter()
{
  return d->operatorOverloadCenter;
}

std::list<GTLCore::Function*>& CompilerBase::functionsToDelete()
{
  return d->functionsToDelete;
}

void CompilerBase::createStdLibFunction(llvm::LLVMContext& _context, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, int count, ...)
{
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, count);
  for(int i = 0; i < count; ++i)
  {
    const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
    bool output = va_arg(argp, int);
    arguments.push_back(GTLCore::Parameter("", type, output, false, GTLCore::Value() ) );
  }
  va_end(argp);
  GTLCore::Function* function = GTLCore::Function::Private::createExternalFunction( d->moduleData, d->llvmModule, _context, _name, _symbolName, retType, arguments );
  bool success = declareFunction(GTLCore::ScopedName("", _name), function);
  GTL_ASSERT( success );
  UNUSED( success );
  functionsToDelete().push_back(function);
}

void CompilerBase::setModuleData( ModuleData* moduleData, llvm::Module* llvmModule )
{
  d->moduleData = moduleData;
  d->llvmModule = llvmModule;
}
