/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Expression.h"

#include <llvm/Constant.h>
#include <llvm/Constants.h>
#include <llvm/Function.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "GTLCore/LLVMBackend/CodeGenerator_p.h"
#include "GTLCore/LLVMBackend/ExpressionResult_p.h"
#include "GTLCore/VariableNG_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/Utils_p.h"
#include "GTLCore/Value.h"
#include "GTLCore/Macros_p.h"

#include "GTLCore/Parameter.h"
#include "GTLCore/Debug.h"
#include "GTLCore/Function.h"
#include "GTLCore/Function_p.h"

#include "AccessorExpression.h"
#include "GarbageCollectionStatement.h"
#include "ConstantCompoundExpression.h"

#include "../LLVMBackend/ExpressionGenerationContext_p.h"
#include "GenerationVisitor.h"
#include "../Color.h"
#include "../TypesManager.h"

using namespace GTLCore;
using namespace GTLCore::AST;

//------------------------------------------//
//-------------- Expression ----------------//
//------------------------------------------//

llvm::BasicBlock* Expression::generateStatement( LLVMBackend::GenerationContext& _context, llvm::BasicBlock* _bb) const
{
  LLVMBackend::ExpressionGenerationContext egc(_bb);
  generateValue( _context, egc);
  return egc.currentBasicBlock();
}

Expression* Expression::fromValue( const GTLCore::Value& _val, const GTLCore::Type* _declarationType)
{
  switch( _val.type()->dataType() )
  {
    case Type::BOOLEAN:
      return new AST::NumberExpression<bool>( _val.asBoolean() );
    case Type::INTEGER32:
      return new AST::NumberExpression<gtl_int32>( _val.asInt32() );
    case Type::FLOAT32:
      return new AST::NumberExpression<float>( _val.asFloat32() );
    case Type::VECTOR:
    case Type::ARRAY:
    {
      std::vector<Expression*> expressions;
      GTL_ASSERT( _val.asArray() );
      foreach( const Value& val, *_val.asArray())
      {
        expressions.push_back( fromValue( val ) );
      }
      return new AST::ConstantCompoundExpression( _val.type(), expressions );
    }
    case Type::STRUCTURE:
    {
      if (_val.type() == Type::Color)
      {
        GTL_ASSERT(_declarationType);
        GTL_ASSERT(_declarationType->structName() == "color");
        std::vector<Expression*> expressions;
        GTLCore::Color c = _val.asColor();
        expressions.push_back(new AST::NumberExpression<float>(c.red()));
        expressions.push_back(new AST::NumberExpression<float>(c.green()));
        expressions.push_back(new AST::NumberExpression<float>(c.blue()));
        expressions.push_back(new AST::NumberExpression<float>(c.alpha()));
        return new AST::ConstantCompoundExpression( _declarationType, expressions );
      } else {
        std::vector<Expression*> expressions;
        foreach(const Value& val, *_val.asArray())
        {
          expressions.push_back(fromValue(val));
        }
        
        return new AST::ConstantCompoundExpression( _val.type(), expressions );
        
      }
    }
    default:
      GTL_ABORT("Unimplmeneted: type is " << _val.type()->dataType() << _val);
  }
}

//------------------------------------------//
//------------- ProxyExpression ------------//
//------------------------------------------//

ProxyExpression::ProxyExpression( Expression* expr ) : m_clone( expr)
{
  GTL_ASSERT(expr);
}

LLVMBackend::ExpressionResult ProxyExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  return m_clone->generateValue( _gc, _egc);
}

ExpressionResultSP ProxyExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return m_clone->generateValue(_generationVisitor);
}

//------------------------------------------//
//------------- GlobalDataExpression ------------//
//------------------------------------------//

GlobalDataExpression::GlobalDataExpression( Expression* expression ) : m_expression( expression ), m_isreturn(false)
{ }

GlobalDataExpression::~GlobalDataExpression()
{
  delete m_expression;
}

LLVMBackend::ExpressionResult GlobalDataExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  LLVMBackend::ExpressionResult value = m_expression->generateValue( _gc, _egc );
  if( m_isreturn) return value;
  if( value.isConstant() )
  {
    return LLVMBackend::ExpressionResult( new llvm::GlobalVariable( *_gc.llvmModule(), value.constant()->getType(), true, llvm::GlobalValue::InternalLinkage, value.constant(), "" ), type() );
  } else {
    return value;
  }
}

ExpressionResultSP GlobalDataExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return m_expression->generateValue(_generationVisitor); // TODO it really sounds like GlobalDataExpression is a mistake and it should be up to the Coumpound to return such a value, as needed
}

//------------------------------------------//
//--------- FunctionCallExpression ---------//
//------------------------------------------//

FunctionCallExpression::~FunctionCallExpression()
{
  deleteAll( m_arguments );
}

const GTLCore::Type* FunctionCallExpression::type() const
{
  GTL_DEBUG( m_function->name() << " <=> " << *m_function->returnType() );
  return m_function->returnType();
}

LLVMBackend::ExpressionResult FunctionCallExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  LLVMBackend::ExpressionResult returnValue = _gc.codeGenerator()->callFunction( _gc, _egc, m_function, m_arguments );
  _gc.appendDelayedStatement( new PointerGarbageCollectionStatement( returnValue.value(), returnValue.type() ) );
  return returnValue;
}

ExpressionResultSP FunctionCallExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateFunctionCall( m_function, m_arguments, annotation() );
}

void FunctionCallExpression::markAsReturnExpression()
{
  // There is no easy way to know if an Array/Structure given as argument of a function
  // will or will not be returned by the function, hence the need to mark parameters
  // as part of a return expression
  for( std::list<Expression*>::iterator it = m_arguments.begin();
       it != m_arguments.end(); ++it)
  {
    (*it)->markAsReturnExpression();
  }
}

// NumberExpression

namespace GTLCore { // GCC or C++ or both sucks and it is required for the template to be defined
  namespace AST {

template<>
const GTLCore::Type*  NumberExpression<float>::type() const
{
  return GTLCore::Type::Float32;
}

template<>
ExpressionResultSP NumberExpression<float>::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateFloat32( m_val, annotation() );
}

template<>
LLVMBackend::ExpressionResult NumberExpression<float>::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& ) const
{
  return LLVMBackend::ExpressionResult(_gc.codeGenerator()->floatToConstant( _gc.llvmContext(), m_val ), type());
}

template<>
const GTLCore::Type*  NumberExpression<gtl_int32>::type() const
{
  return GTLCore::Type::Integer32;
}
template<>
LLVMBackend::ExpressionResult NumberExpression<gtl_int32>::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return LLVMBackend::ExpressionResult(_gc.codeGenerator()->integerToConstant( _gc.llvmContext(), m_val ), type());
}

template<>
ExpressionResultSP NumberExpression<gtl_int32>::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateInteger32( m_val, annotation() );
}

template<>
const GTLCore::Type*  NumberExpression<bool>::type() const
{
  return GTLCore::Type::Boolean;
}
template<>
LLVMBackend::ExpressionResult NumberExpression<bool>::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return LLVMBackend::ExpressionResult(_gc.codeGenerator()-> boolToConstant(_gc.llvmContext(), m_val ), type());
}

template<>
ExpressionResultSP NumberExpression<bool>::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateBoolean( m_val, annotation() );
}

  }
}

LLVMBackend::ExpressionResult StringExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
    return LLVMBackend::ExpressionResult(llvm::ConstantArray::get(_gc.llvmContext(), (const std::string&)m_string, true ), Type::Integer8 ); // return a null terminated string
}

ExpressionResultSP StringExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateString(m_string, annotation());
}
