/*
 *  Copyright (c) 2011 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ColorToPixelOverloadOperatorFactory_p.h"

#include <GTLCore/AST/BinaryExpression.h>
#include <GTLCore/LLVMBackend/ExpressionResult_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>

#include "Debug.h"
#include "CodeGenerator_p.h"
#include <GTLFragment/wrappers/ColorWrap_p.h>
#include <GTLCore/Type_p.h>
#include <llvm/Value.h>
#include <llvm/DerivedTypes.h>
#include <llvm/CallingConv.h>
#include <llvm/Module.h>
#include <GTLCore/LLVMBackend/ExpressionGenerationContext_p.h>
#include <GTLCore/LLVMBackend/CodeGenerator_p.h>
#include <llvm/Instructions.h>
#include <GTLCore/TypesManager.h>

using namespace OpenShiva;

class ColorToPixelOverloadOperator : public GTLCore::AST::BinaryExpression {
public:
  ColorToPixelOverloadOperator(Expression* lhs, Expression* rhs);
  virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
  virtual GTLCore::AST::ExpressionResultSP generateValue( GTLCore::AST::GenerationVisitor* _generationVisitor) const;
};

ColorToPixelOverloadOperator::ColorToPixelOverloadOperator(Expression* lhs, Expression* rhs) : BinaryExpression(lhs, rhs)
{
}

LLVMBackend::ExpressionResult ColorToPixelOverloadOperator::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  llvm::Value* pixelPtr = leftHandSide()->generateValue( _gc, _egc).value();
  llvm::Value* pixelDataPtr = OpenShiva::CodeGenerator::accessPixelDataPtr( _gc, _egc.currentBasicBlock(), pixelPtr);
  llvm::Value* converterDataPtr = OpenShiva::CodeGenerator::accessColorConverterPtr( _gc, _egc.currentBasicBlock(), pixelPtr);
  
  llvm::Value* color = rightHandSide()->generateValue( _gc, _egc).value();

  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(ColorWrap::POS_RED) ));
  llvm::Value* colorPtr = llvm::GetElementPtrInst::Create( color, indexes, "", _egc.currentBasicBlock() );

  //Parameters are AbstractColorConverter*, ptrToColor, ptrTopixeldata, number of channels
  std::vector<llvm::Value*> parameters;
  parameters.push_back( new llvm::LoadInst( converterDataPtr, "ColorToPixelOverloadOperator::generateValue", _egc.currentBasicBlock() ));
  parameters.push_back(colorPtr);
  parameters.push_back(pixelDataPtr);
  parameters.push_back(LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), llvm::cast<llvm::VectorType>(pixelDataPtr->getType()->getContainedType(0))->getNumElements() ) );

  llvm::Function* func = CodeGenerator::createWrapConvertColorToPixelFunction(_gc, pixelDataPtr->getType());
  GTL_COMPARE_FUNCTION_PARAMETERS( func, parameters );
  
  llvm::CallInst *CallFunc = llvm::CallInst::Create(func, parameters, "", _egc.currentBasicBlock());
  CallFunc->setTailCall(false);
  
  return LLVMBackend::ExpressionResult(pixelPtr, leftHandSide()->type());
}

GTLCore::AST::ExpressionResultSP ColorToPixelOverloadOperator::generateValue( GTLCore::AST::GenerationVisitor* _generationVisitor) const
{
  GTL_ABORT("Unimplemented");
  return 0;
}

ColorToPixelOverloadOperatorFactory::ColorToPixelOverloadOperatorFactory()
{
}

ColorToPixelOverloadOperatorFactory::~ColorToPixelOverloadOperatorFactory()
{
}

GTLCore::AST::BinaryExpression* ColorToPixelOverloadOperatorFactory::create( GTLCore::AST::Expression* value1, GTLCore::AST::Expression* value2, GTLCore::Operator::Type type ) const
{
  GTL_ASSERT(canOverload(value1->type(), value2->type(), type) );
  
  return new ColorToPixelOverloadOperator(value1, value2);
}

bool ColorToPixelOverloadOperatorFactory::canOverload( const GTLCore::Type* srcType, const GTLCore::Type* dstType, GTLCore::Operator::Type type ) const
{
  return (srcType->dataType() == GTLCore::Type::STRUCTURE and srcType->structName().startWith( "pixel" ) and
          dstType->dataType() == GTLCore::Type::STRUCTURE and dstType->structName().startWith( "color" ) );
}
