/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "TemplateParser_p.h"

#include <fstream>

#include <GTLCore/CompilationMessage.h>
#include <GTLCore/Macros_p.h>

#include "Debug.h"
#include "TemplateLexer_p.h"
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Path.h>
#endif
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/CompilationMessages_p.h>

using namespace OpenCTL;
using namespace OpenCTL::TemplateAST;

struct TemplateParser::Private {
  TemplateLexer* m_lexer;
  GTLCore::String m_fileName;
  GTLCore::Token token;
  GTLCore::CompilationMessages errorMessages;
  std::list<GTLCore::String> includeDirectories;
};

TemplateParser::TemplateParser( TemplateLexer* _lexer, const GTLCore::String& _fileName, const std::list<GTLCore::String>& _includeDirectories) : d(new Private)
{
  d->m_lexer = _lexer;
  d->m_fileName = _fileName;
  d->includeDirectories = _includeDirectories;
}

TemplateParser::~TemplateParser()
{
}

TemplateAST::Node* TemplateParser::parse()
{
  using GTLCore::Token;
  std::list<Node*> nodes;
  getNextToken() ;
  while( currentToken().type != GTLCore::Token::END_OF_FILE )
  {
    bool shouldGetNextToken = true;
    switch(currentToken().type)
    {
      case Token::STRING_CONSTANT:
        nodes.push_back( new TextNode(currentToken().string) );
        break;
      case Token::STARTBRACKET: // In the body, brackets are always considered as strings
        nodes.push_back( new TextNode("(") );
        break;
      case Token::ENDBRACKET: // In the body, brackets are always considered as strings
        nodes.push_back( new TextNode(")") );
        break;
      case Token::COMA: // In the body, coma are always considered as strings
        nodes.push_back( new TextNode(",") );
        break;
      case Token::EQUAL: // In the body, coma are always considered as strings
        nodes.push_back( new TextNode("=") );
        break;
      case Token::ALLCHANNELS:
        nodes.push_back( parseAllChannels( AllChannelsNode::AllChannel ) );
        break;
      case Token::COLORCHANNELS:
        nodes.push_back( parseAllChannels( AllChannelsNode::ColorChannel ) );
        break;
      case Token::ALPHACHANNEL:
        nodes.push_back( parseAllChannels( AllChannelsNode::AlphaChannel ) );
        break;
      case Token::ALPHA:
        nodes.push_back( new AlphaNode );
        break;
      case Token::ALPHAMAX:
        nodes.push_back( new AlphaMaxNode );
        break;
      case Token::ALPHAMIN:
        nodes.push_back( new AlphaMinNode );
        break;
      case Token::ALPHAUNIT:
        nodes.push_back( new AlphaUnitNode );
        break;
      case Token::MAX:
        getNextToken();
        nodes.push_back( new NamedMaxNode( parseName(false) ) );
        break;
      case Token::MIN:
        getNextToken();
        nodes.push_back( new NamedMinNode( parseName(false) ) );
        break;
      case Token::OPERATION:
      {
        getNextToken();
        int inputs = 1;
        GTLCore::String name = parseName(true);
        GTLCore::String args, optargs;
        while(currentToken().type != Token::END_OF_FILE and currentToken().type != Token::ENDBRACKET)
        {
          isOfType(currentToken(), Token::COMA);
          getNextToken(); // eat the ,
          isOfType(currentToken(), Token::STRING_CONSTANT);
          GTLCore::String name = currentToken().string.trimmed();
          getNextToken(); // eat the name
          isOfType(currentToken(), Token::EQUAL); // in the future, some option might not need an equal
          getNextToken(); // eat the equal
          int bracket = 0;
          GTLCore::String val;
          while(currentToken().type != Token::END_OF_FILE and ( (currentToken().type != Token::COMA and currentToken().type != Token::ENDBRACKET) or bracket > 0) )
          {
            switch(currentToken().type)
            {
              case Token::STARTBRACKET:
                ++bracket;
                if(bracket > 1)
                {
                  val += "(";
                }
                break;
              case Token::ENDBRACKET:
                --bracket;
                if(bracket > 0)
                {
                  val += ")";
                }
                break;
              case Token::EQUAL:
                val += "=";
                break;
              case Token::COMA:
                val += ",";
                break;
              case Token::STRING_CONSTANT:
                val += currentToken().string;
                break;
              default:
                reportUnexpected(currentToken());
            }
            getNextToken();
          }
          if(name == "inputs")
          {
            inputs = val.toInt();
          } else if(name == "arguments")
          {
            args += val + ", ";
          } else if(name == "optional_arguments")
          {
            optargs += ", " + val;
          }
          else {
            reportError( "Unknown parameter '" + name + "' for operation", currentToken() );
          }
        }
        isOfType(currentToken(), Token::ENDBRACKET);
        nodes.push_back( new OperationNode( name, inputs, args, optargs ) );
        break;
      }
      case Token::UNIT:
        getNextToken();
        nodes.push_back( new NamedUnitNode( parseName(false) ) );
        break;
      case Token::TYPE:
        getNextToken();
        nodes.push_back( new NamedTypeNode( parseName(false) ) );
        break;
      case Token::INCLUDE:
      {
        getNextToken();
        isOfType(currentToken(), Token::STARTBRACKET);
        getNextToken();
        isOfType(currentToken(), Token::STRING_CONSTANT);
        GTLCore::String sourceName = currentToken().string + ".ctlt";
        TemplateAST::Node* node = 0;
        for( std::list<GTLCore::String>::iterator it = d->includeDirectories.begin();
            it != d->includeDirectories.end(); ++it )
        {
          llvm::sys::Path path( (const std::string&)*it );
          path.appendComponent( (const std::string&)sourceName);
          OCTL_DEBUG("try " << path.c_str() );
          if(path.exists() and path.canRead())
          {
            std::ifstream in;
            in.open(path.c_str() );
            if(in)
            {
              TemplateLexer lexer(&in);
              TemplateParser parser(&lexer, sourceName, d->includeDirectories);
              node = parser.parse();
              if(not node)
              {
                foreach(const GTLCore::CompilationMessage& msg, parser.errorMessages().errors())
                {
                  d->errorMessages.d->appendMessage(msg);
                }
              }
              break;
            }
          }
        }
        
        if(node)
        {
          nodes.push_back(node);
        }
        else
        {
          reportError("Can't find include file: " + sourceName, currentToken());
        }
        getNextToken();
        isOfType(currentToken(), Token::ENDBRACKET);
      }
        break;
      case Token::OUTPUT:
      {
        nodes.push_back(new InOutNode( OpenCTL::TemplateAST::InOutNode::Out, OpenCTL::TemplateAST::InOutNode::All ) );
        break;
      }
      case Token::INPUT:
      {
        getNextToken();
        int idx = 0;
        if(currentToken().type == Token::STARTBRACKET)
        {
          getNextToken(); // eath the '('
          isOfType(currentToken(), Token::STRING_CONSTANT);
          idx = currentToken().string.toInt();
          getNextToken(); // eat the text
          isOfType(currentToken(), Token::ENDBRACKET);
        } else {
          shouldGetNextToken = false;
        }
        nodes.push_back(new InOutNode(OpenCTL::TemplateAST::InOutNode::In, OpenCTL::TemplateAST::InOutNode::All, idx));
        break;
      }
      default:
        reportUnexpected(currentToken());
    }
    if(shouldGetNextToken)
    {
      getNextToken();
    }
  }
  if(d->errorMessages.errors().empty() )
  {
    return new NodesList(nodes);
  } else {
    foreach(Node* node, nodes)
    {
      delete node;
    }
    return 0;
  }
}

void TemplateParser::getNextToken()
{
  d->token = d->m_lexer->nextToken();
}

const GTLCore::Token& TemplateParser::currentToken()
{
  return d->token;
}

void TemplateParser::reportError( const GTLCore::String& errMsg, const GTLCore::Token& token )
{
  GTL_DEBUG( errMsg );
  d->errorMessages.d->appendMessage( GTLCore::CompilationMessage( GTLCore::CompilationMessage::ERROR, errMsg, token.line, d->m_fileName ) );
}

void TemplateParser::reportUnexpected( const GTLCore::Token& token )
{
  reportError("Unexpected: " + GTLCore::Token::typeToString( token.type ), token );
  getNextToken();
}

const GTLCore::CompilationMessages& TemplateParser::errorMessages() const
{
  return d->errorMessages;
}

bool TemplateParser::isOfType( const GTLCore::Token& token, GTLCore::Token::Type type )
{
  if( token.type == type )
  {
    return true;
  } else {
    reportError("Expected " + GTLCore::Token::typeToString(type) + " before " + GTLCore::Token::typeToString(token.type)  + ".", token);
    return false;
  }
}

TemplateAST::Node* TemplateParser::parseAllChannels( TemplateAST::AllChannelsNode::WhichChannel _whichChannel )
{
  using GTLCore::Token;
  std::list<Node*> nodes;
  getNextToken(); // Remove the @allchannels or @colorchannels or @alphachannel

  // Parse the list of commands
  if(isOfType(currentToken(), Token::STARTBRACKET) )
  {
    int bracketCount = 1; // This is use to manage this kind of situation @allchannels( a * ( 1 + b) )
    bool shouldGetNextToken = true;
    while(currentToken().type != Token::END_OF_FILE and bracketCount > 0)
    {
      if(shouldGetNextToken)
      {
        getNextToken();
      } else {
        shouldGetNextToken = true;
      }
      switch(currentToken().type)
      {
        case Token::STARTBRACKET:
          nodes.push_back(new TextNode( "(" ));
          ++bracketCount;
          break;
        case Token::ENDBRACKET:
        {
          if(bracketCount != 1 ) { // Don't add the bracket that correspond to the first one
            nodes.push_back(new TextNode( ")" ));
          }
          --bracketCount;
        }
          break;
        case Token::COMA: // coma are considered as strings
          nodes.push_back( new TextNode(",") );
          break;
      case Token::EQUAL: // In a channel, coma are always considered as strings
          nodes.push_back( new TextNode("=") );
        break;
        case Token::STRING_CONSTANT:
          nodes.push_back(new TextNode( currentToken().string ));
          break;
        case Token::TYPE:
          nodes.push_back(new ChannelTypeNode());
          break;
        case Token::ALPHA:
          nodes.push_back( new AlphaNode );
          break;
        case Token::ALPHAMAX:
          nodes.push_back( new AlphaMaxNode );
          break;
        case Token::ALPHAMIN:
          nodes.push_back( new AlphaMinNode );
          break;
        case Token::ALPHAUNIT:
          nodes.push_back( new AlphaUnitNode );
          break;
        case Token::MAX:
          nodes.push_back(new ChannelMaxNode());
          break;
        case Token::MIN:
          nodes.push_back(new ChannelMinNode());
          break;
        case Token::UNIT:
          nodes.push_back(new ChannelUnitNode());
          break;
        case Token::VAR:
        {
          getNextToken();
          nodes.push_back(new VarNode(parseName(false)));
          break;
        }
        case Token::OUTPUT:
          nodes.push_back(new InOutNode(OpenCTL::TemplateAST::InOutNode::Out, OpenCTL::TemplateAST::InOutNode::One));
          break;
        case Token::INPUT:
        {
          getNextToken();
          int idx = 0;
          if(currentToken().type == Token::STARTBRACKET)
          {
            getNextToken(); // eath the '('
            isOfType(currentToken(), Token::STRING_CONSTANT);
            idx = currentToken().string.toInt();
            getNextToken(); // eat the text
            isOfType(currentToken(), Token::ENDBRACKET);
          } else {
            shouldGetNextToken = false;
          }
          nodes.push_back(new InOutNode(OpenCTL::TemplateAST::InOutNode::In, OpenCTL::TemplateAST::InOutNode::One, idx));
          break;
        }
        default:
          reportUnexpected(currentToken());
      }
    }
    GTL_ASSERT(bracketCount >= 0);
    isOfType(currentToken(), Token::ENDBRACKET);
  }
  return new AllChannelsNode( new NodesList(nodes), _whichChannel);
}

GTLCore::String TemplateParser::parseName(bool _isOperation) {
  if(isOfType(currentToken(), GTLCore::Token::STARTBRACKET)) {
    getNextToken(); // eat the '('
    if(isOfType(currentToken(), GTLCore::Token::STRING_CONSTANT)) {
      GTLCore::String string = currentToken().string;
      getNextToken(); // eat the name
      if( not _isOperation)
      {
        isOfType(currentToken(), GTLCore::Token::ENDBRACKET); // check that we end with ')'
      }
      
      return string;
    }
  }
  return "";
}

