/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Program.h"

// LLVM
#include <llvm/DerivedTypes.h>
#include <llvm/Instructions.h>
#include <llvm/Module.h>
#include <llvm/Transforms/Utils/Cloning.h>

// Passes
#include <llvm/PassManager.h>
#include <llvm/Analysis/LoopPass.h>
#include <llvm/Analysis/Verifier.h>
#include <llvm/Target/TargetData.h>
#include <llvm/ExecutionEngine/ExecutionEngine.h>
#include <llvm/GlobalVariable.h>

// GTLCore
#include "GTLCore/Buffer.h"
#include "GTLCore/BufferImage.h"
#include "GTLCore/LLVMBackend/CodeGenerator_p.h"
#include "GTLCore/LLVMBackend/GenerationContext_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/LLVMBackend/ExpressionResult_p.h"
#include "GTLCore/Function.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/String.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/Value.h"
#include "GTLCore/VariableNG_p.h"
#include "GTLCore/Utils_p.h"

// OpenCTL
#include "Module.h"
#include "GTLCore/ModuleData_p.h"
#include "Debug.h"
#include "GTLCore/VirtualMachine_p.h"

using namespace OpenCTL;

struct Program::Private {
  Private(const GTLCore::PixelDescription& _srcPixelDescription, const GTLCore::PixelDescription& _dstPixelDescription) : dstPixelDescription(_dstPixelDescription)
  {
    srcPixelDescriptions.push_back(_srcPixelDescription);
  }
  Private(const std::list<GTLCore::PixelDescription>& _srcPixelDescription, const GTLCore::PixelDescription& _dstPixelDescription) : srcPixelDescriptions(_srcPixelDescription), dstPixelDescription(_dstPixelDescription)
  {
    GTL_ASSERT(srcPixelDescriptions.size() > 0);
  }
  llvm::Module* module;
  GTLCore::ModuleData* moduleData;
  void (*func)( const char**, char*, int, GTLCore::ProgressReport*, gtl_uint64);
  std::list<GTLCore::PixelDescription> srcPixelDescriptions;
  GTLCore::PixelDescription dstPixelDescription;
  static int s_id;
  std::map< GTLCore::String, llvm::GlobalVariable*> varyings;
  std::map< GTLCore::String, void*> varyingsPtr;
  std::list<GTLCore::String> varyingsName;
};

int Program::Private::s_id = 0;

void configureBuffer(LLVMBackend::CodeGenerator& cg, const std::vector< const GTLCore::Type*>& channelsTypes, llvm::Value**& indexes, bool*& needBuffer, int countChannels, llvm::LLVMContext& context)
{
  gtl_int32 currentPos = 0;
  for( int i = 0; i < countChannels; ++i)
  {
    indexes[i] = cg.integerToConstant( context, currentPos );
    currentPos += channelsTypes[i]->bitsSize() / 8;
    OCTL_ASSERT( channelsTypes[i]->bitsSize() % 8 == 0);
    if( channelsTypes[i] == GTLCore::Type::Integer8
        or channelsTypes[i] == GTLCore::Type::UnsignedInteger8
        or channelsTypes[i] == GTLCore::Type::Integer16
        or channelsTypes[i] == GTLCore::Type::UnsignedInteger16
        or channelsTypes[i] == GTLCore::Type::Float16 )
    {
      needBuffer[i] = true;
    } else {
      needBuffer[i] = false;
    }
  }
}

Program::Program(const GTLCore::String& functionName, const Module* module, const GTLCore::PixelDescription& pixelDescription) : d(new Private(pixelDescription, pixelDescription))
{
  init(functionName, module);
}


Program::Program(const GTLCore::String& functionName, const Module* module, const GTLCore::PixelDescription& srcPixelDescription, const GTLCore::PixelDescription& dstPixelDescription) : d(new Private(srcPixelDescription, dstPixelDescription))
{
  init(functionName, module);
}

Program::Program(const GTLCore::String& functionName, const Module* module, const std::list<GTLCore::PixelDescription>& srcPixelDescriptions, const GTLCore::PixelDescription& dstPixelDescription ) : d(new Private(srcPixelDescriptions,dstPixelDescription))
{
  init(functionName, module);
}

struct InputGenInfo {
  explicit InputGenInfo( const GTLCore::PixelDescription& _pd, llvm::LLVMContext& context ) : channelsTypes(_pd.channelTypes()), pos(0)
  {
    pixelSize = _pd.bitsSize() / 8;
    OCTL_ASSERT(_pd.bitsSize() % 8== 0);
    indexes = new llvm::Value*[ channelsTypes.size() ];
    needConversion = new bool[ channelsTypes.size() ]; // CTL doesn't support Int8 or Int16 (and OpenCTL's VM doesn't know about half), so when some pixel channel is in Int8 or Int16 (or half) it first need to be converted to Int32 (or float)
    countChannels = channelsTypes.size();
    pixelSizeValue = LLVMBackend::CodeGenerator::integerToConstant( context, pixelSize );
  }
  ~InputGenInfo() {
    delete[] indexes;
    delete[] needConversion;
    delete pos;
  }
  const std::vector< const GTLCore::Type*>& channelsTypes;
  gtl_int32 pixelSize;
  llvm::Value** indexes;
  bool* needConversion;
  llvm::Value* pixelSizeValue;
  int countChannels;
  GTLCore::VariableNG* pos;
  llvm::Value* addr;
  private:
    InputGenInfo(const InputGenInfo& igi) : channelsTypes(igi.channelsTypes) {}
    InputGenInfo& operator=(const InputGenInfo&) { return *this; }
};

void Program::init(const GTLCore::String& functionName, const Module* module)
{
  d->func = 0;
  d->moduleData = 0;
  // Clone the module
  if( module->data())
  {
    d->module = llvm::CloneModule(module->data()->llvmLinkedModule());
    d->moduleData = new GTLCore::ModuleData( d->module );
    // Select the function
    std::vector<GTLCore::Parameter> arguments;
    foreach(const GTLCore::PixelDescription& pd, d->srcPixelDescriptions)
    {
      foreach(const GTLCore::Type* type, pd.channelTypes())
      {
        arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value()));
      }
    }
    foreach(const GTLCore::Type* type, d->dstPixelDescription.channelTypes())
    {
      arguments.push_back(GTLCore::Parameter("", type, true, false, GTLCore::Value()));
    }
    const GTLCore::Function* functionDef = module->function( functionName, arguments );
    if( not functionDef ) return;
    
    llvm::Function* function = d->module->getFunction( (const std::string&)GTLCore::Function::Data::symbolName( GTLCore::ScopedName( "", functionName), functionDef->returnType(),functionDef->parameters() ) );
    if(function)
    {
      LLVMBackend::CodeGenerator cg( d->moduleData );
      // Initialise pixel information and Configure input buffers
      OCTL_DEBUG("Configure input buffer");
      std::vector<InputGenInfo*> srcs;
      for( std::list<GTLCore::PixelDescription>::iterator it = d->srcPixelDescriptions.begin();
           it != d->srcPixelDescriptions.end(); ++it )
      {
        InputGenInfo* src = new InputGenInfo(*it, d->module->getContext());
        srcs.push_back( src);
        configureBuffer( cg, src->channelsTypes, src->indexes, src->needConversion, src->countChannels, d->module->getContext());
      }
      // Initialise pixel information and Configure output buffers
      InputGenInfo dst( d->dstPixelDescription, d->module->getContext());
      configureBuffer( cg, dst.channelsTypes, dst.indexes, dst.needConversion, dst.countChannels, d->module->getContext());
      
      //---------- Program-Pseudo-code ----------//
      // This just a rough explanation of the function that get generated here.
      //
      // template<int pixelSize, typename _TYPE_>
      // void program(const char** ins8, char* out8, int size, ProgressReport* _report)
      // {
      //   _BUFFER_TYPE_ buffer[pixelSize]; // buffer for Int8, Int16 and half channels
      //   for(int i = 0; i < size; i += pixelSize)
      //   {
      //     const _TYPE_* in0 = (const _TYPE_*)(in8[0] + i);
      //     const _TYPE_* in1 = (const _TYPE_*)(in8[1] + i);
      //     _TYPE_* out = (_TYPE_*)(out8 + 1);
      //     if(needBuffer)
      //     {
      //       
      //     }
      //     function(in0[i], in0[i+indexes[1]] ..., out + i, out + i +indexes[1], ...);
      //     _report->nextPixel();
      //   }
      //   return;
      // }
      OCTL_DEBUG("Initialize the function type");
      std::vector<const llvm::Type*> params;
      params.push_back( llvm::PointerType::get( llvm::PointerType::get( llvm::Type::getInt8Ty(d->module->getContext()), 0 ), 0 ) ); // const char** int8
      params.push_back( llvm::PointerType::get( llvm::Type::getInt8Ty(d->module->getContext()), 0 )); // char* out8
      params.push_back( llvm::Type::getInt32Ty(d->module->getContext()) ); // int size
      params.push_back( GTLCore::Type::Pointer->d->type(d->module->getContext()) ); // ProgressReport* _report
      params.push_back( llvm::Type::getInt64Ty(d->module->getContext()) );
      // void program(const char* in8, char* out8, int size)
      llvm::FunctionType* definitionType = llvm::FunctionType::get( llvm::Type::getVoidTy(d->module->getContext()), params, false );
      int programNb = ++Program::Private::s_id;
      llvm::Function* func = cg.createFunction( d->module, definitionType, "CTLProgram" + GTLCore::String::number(programNb));
      // Initialise a generation context
      LLVMBackend::GenerationContext gc( &cg, &d->module->getContext(), func, 0, d->moduleData, d->module );
      // {
      OCTL_DEBUG("Initial block");
      llvm::BasicBlock* initialBlock = llvm::BasicBlock::Create(d->module->getContext());
      func->getBasicBlockList().push_back( initialBlock );
      // Initialise the buffer, as needed
      OCTL_DEBUG("Initialise buffer");
      GTLCore::VariableNG** buffer = new GTLCore::VariableNG*[dst.countChannels];
      for(int i = 0; i < dst.countChannels; ++i)
      {
        GTLCore::VariableNG* vng = 0;
        if(dst.needConversion[i])
        {
          if(dst.channelsTypes[i] == GTLCore::Type::Float16)
          {
            vng = new GTLCore::VariableNG(GTLCore::Type::Float32, false, false );
          } else {
            vng = new GTLCore::VariableNG(GTLCore::Type::Integer32, false, false );
          }
        } else {
          vng = new GTLCore::VariableNG(dst.channelsTypes[i], false, false );
        }
        vng->initialise( gc, initialBlock, LLVMBackend::ExpressionResult(), std::list<llvm::Value*>());
        buffer[i] = vng;
      }
      OCTL_DEBUG("Get the arguments");
      // Get the args.
      llvm::Function::arg_iterator arg_it = func->arg_begin();
      //   const char* in8 = first arg;
      llvm::Value* ins = arg_it;
      
      for(std::size_t k = 0; k < srcs.size(); ++k)
      {
        srcs[k]->addr = new llvm::LoadInst( llvm::GetElementPtrInst::Create( ins, cg.integerToConstant(d->module->getContext(), gtl_uint32(k)), "", initialBlock ), "", initialBlock );
      }
      //   char* out8 = second arg;
      ++arg_it;
      llvm::Value* out = arg_it;
      // size = third arg
      ++arg_it;
      llvm::Value* size = arg_it;
      //   void* _report = fourth arg;
      ++arg_it;
      llvm::Value* arg_report = arg_it;
      //   void* _channelFlags = fifth arg;
      ++arg_it;
      llvm::Value* arg_channelFlags = arg_it;
      
    // Construct the "conditions" of the loop
      
      //   int i = 0;
      OCTL_DEBUG("int i = 0");
      
      for(std::size_t k = 0; k < srcs.size(); ++k)
      {
        srcs[k]->pos = new GTLCore::VariableNG( GTLCore::Type::Integer32, false, false );
        srcs[k]->pos->initialise( gc, initialBlock, LLVMBackend::ExpressionResult(cg.integerToConstant(d->module->getContext(), INT32_C(0)), GTLCore::Type::Integer32), std::list<llvm::Value*>());
      }
      dst.pos = new GTLCore::VariableNG( GTLCore::Type::Integer32, false, false );
      dst.pos->initialise( gc, initialBlock, LLVMBackend::ExpressionResult(cg.integerToConstant(d->module->getContext(), INT32_C(0)), GTLCore::Type::Integer32), std::list<llvm::Value*>());
      
      // i < size
      OCTL_DEBUG("i < size");
      llvm::BasicBlock* forTestBlock = llvm::BasicBlock::Create(d->module->getContext(), "forTestBlock");
      func->getBasicBlockList().push_back( forTestBlock);
      llvm::Value* forTest = cg.createStrictInferiorExpression(forTestBlock, srcs[0]->pos->get( gc, forTestBlock ), srcs[0]->pos->type(), size, GTLCore::Type::Integer32 );
      
      // i += pixelSize
      OCTL_DEBUG("i += pixelSize");
      llvm::BasicBlock* updateBlock = llvm::BasicBlock::Create(d->module->getContext(), "updateBlock");
      func->getBasicBlockList().push_back( updateBlock);
      for(std::size_t k = 0; k < srcs.size(); ++k)
      {
        srcs[k]->pos->set( gc, updateBlock, cg.createAdditionExpression( updateBlock, srcs[k]->pos->get( gc, updateBlock), srcs[k]->pos->type(), srcs[k]->pixelSizeValue, GTLCore::Type::Integer32 ), GTLCore::Type::Integer32 );
      }
      dst.pos->set( gc, updateBlock, cg.createAdditionExpression( updateBlock, dst.pos->get( gc, updateBlock), dst.pos->type(), dst.pixelSizeValue, GTLCore::Type::Integer32 ), GTLCore::Type::Integer32 );
      
      // Construct the body of the for loop
      OCTL_DEBUG("bodyBlock");
      llvm::BasicBlock* bodyBlock = llvm::BasicBlock::Create(d->module->getContext(), "bodyBlock");
      llvm::BasicBlock* startBodyBlock = bodyBlock;
      func->getBasicBlockList().push_back( bodyBlock);
      
      // function(in[i], in[i+indexes[1]] ..., out + i, out + i +indexes[1], ...);
      std::vector<llvm::Value*> arguments;
      
      // Generate in[i], in[i+indexes[1]] ...
      OCTL_DEBUG("Generate in[i], in[i+indexes[1]] ...");
      for(std::size_t k = 0; k < srcs.size(); ++k)
      {
        llvm::Value* srcIndex = srcs[k]->pos->get( gc, bodyBlock);
        for(int i = 0; i < srcs[k]->countChannels; ++i)
        {
          // Load the value from the input buffer
          OCTL_DEBUG("Load the value from the input buffer");
          llvm::Value* convertedIn = new llvm::LoadInst(
                        cg.convertPointerTo( bodyBlock,
                                  llvm::GetElementPtrInst::Create( srcs[k]->addr, cg.createAdditionExpression( bodyBlock, srcIndex, srcs[k]->pos->type(), srcs[k]->indexes[i], GTLCore::Type::Integer32), "", bodyBlock), srcs[k]->channelsTypes[i]->d->type(d->module->getContext())),
                        "", bodyBlock);
          if( srcs[k]->needConversion[i])
          { // if a buffer is needed that means that the value must be converted
            OCTL_DEBUG("if a buffer is needed that means that the value must be converted");
            if(srcs[k]->channelsTypes[i] == GTLCore::Type::Float16)
            {
              convertedIn = LLVMBackend::CodeGenerator::convertFromHalf( gc, bodyBlock, convertedIn);
            } else {
              convertedIn = LLVMBackend::CodeGenerator::convertValueTo( bodyBlock, convertedIn, srcs[k]->channelsTypes[i], GTLCore::Type::Integer32);
            }
          }
          arguments.push_back( convertedIn );
        }
      }
      // Generate ut + i, out + i +indexes[1], ...
      OCTL_DEBUG("Generate ut + i, out + i +indexes[1], ...");
      llvm::Value* dstIndex = dst.pos->get( gc, bodyBlock);
      for(int i = 0; i < dst.countChannels; ++i)
      {
        arguments.push_back( buffer[i]->pointer(bodyBlock));
      }
      // Check if there are more parameters to call
      OCTL_DEBUG("Check if there are more parameters to call");
      const std::vector< GTLCore::Parameter >& parameters = functionDef->parameters();
      if( arguments.size() < parameters.size() )
      {
        OCTL_DEBUG("Filling with constant parameters");
        for( unsigned int i = arguments.size(); i < parameters.size(); ++i)
        {
          llvm::GlobalVariable* globalVar =
              new llvm::GlobalVariable(
                    *d->module,
                    parameters[i].type()->d->type(d->module->getContext()),
                    false, llvm::GlobalValue::ExternalLinkage,
                    LLVMBackend::CodeGenerator::valueToConstant( gc, parameters[i].defaultValue() ), "" );
          arguments.push_back( new llvm::LoadInst( globalVar, "", bodyBlock ) );
          d->varyings[ parameters[i].name() ] = globalVar;
          d->varyingsName.push_back( parameters[i].name() );
        }
      }
      // Some debug
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
      for( unsigned int i = 0; i < arguments.size(); i++)
      {
        OCTL_DEBUG("arguments[" << i << "] == " << *arguments[i]);
        OCTL_DEBUG("type[" << i << "] == " << function->getFunctionType()->getParamType(i) << " " << arguments[i]->getType());
      }
      OCTL_DEBUG( *function->getFunctionType());
#endif
      llvm::CallInst *CallFunc = llvm::CallInst::Create(function, arguments.begin(), arguments.end(), "", bodyBlock);
      CallFunc->setTailCall(false);
      // If there was buffering, save to output
      OCTL_DEBUG("If there was buffering, save to output");
      for(int i = 0; i < dst.countChannels; ++i)
      {
        
        llvm::BasicBlock* ifBlock = llvm::BasicBlock::Create(gc.llvmContext());
        func->getBasicBlockList().push_back( ifBlock );
        
        const llvm::Type* channelType = dst.channelsTypes[i]->d->type(d->module->getContext());
        llvm::Value* pointer = cg.convertPointerTo( ifBlock, llvm::GetElementPtrInst::Create( out, cg.createAdditionExpression( ifBlock, dstIndex, GTLCore::Type::Integer32, dst.indexes[i], GTLCore::Type::Integer32), "", ifBlock), channelType);
        llvm::Value* result = buffer[i]->get( gc, ifBlock );
        if( dst.needConversion[i])
        {
          if( dst.channelsTypes[i] == GTLCore::Type::Float16 )
          {
            result = LLVMBackend::CodeGenerator::convertToHalf( gc, ifBlock, result, GTLCore::Type::Float32 );
          } else {
            result = LLVMBackend::CodeGenerator::convertValueTo( ifBlock, result, GTLCore::Type::Integer32, dst.channelsTypes[i] );
          }
        }
        
        //      gtl_uint64 channelMask = 1 << k;
        gtl_uint64 channelMask = gtl_uint64(1) << i;
        llvm::Value* channelMaskValue = LLVMBackend::CodeGenerator::valueToConstant(gc, channelMask );
        // (_channelFlags & channelMask)
        llvm::Value* lhs = LLVMBackend::CodeGenerator::createBitAndExpression(bodyBlock,
                                  arg_channelFlags, GTLCore::Type::UnsignedInteger64,
                                  channelMaskValue, GTLCore::Type::UnsignedInteger64 );
        
        //  (_channelFlags & channelMask) == channelMask
        llvm::Value* comp = LLVMBackend::CodeGenerator::createEqualExpression(bodyBlock,
                                  lhs, GTLCore::Type::UnsignedInteger64,
                                  channelMaskValue, GTLCore::Type::UnsignedInteger64);
        
        //      if( (_channelFlags & channelMask) == channelMask )
        llvm::BasicBlock* nextCurrentBlock = llvm::BasicBlock::Create(gc.llvmContext());
        func->getBasicBlockList().push_back( nextCurrentBlock );
          
        LLVMBackend::CodeGenerator::createIfStatement(bodyBlock, comp, GTLCore::Type::Boolean, ifBlock, ifBlock, nextCurrentBlock);

        bodyBlock = nextCurrentBlock;
        
        OCTL_DEBUG( *result->getType() << *pointer->getType() );
        new llvm::StoreInst( result, pointer, bodyBlock);
      }
    // Call the progress report
      bodyBlock = LLVMBackend::CodeGenerator::callProgressReportNextPixel( gc, arg_report, bodyBlock );
    // Put the for loop together
      OCTL_DEBUG("Put the for loop together");
      // for(int i = 0; i < size; ++i)
      OCTL_DEBUG("for(int i = 0; i < size; ++i)");
      llvm::BasicBlock* finBlock = llvm::BasicBlock::Create(d->module->getContext(), "finBlock");
      func->getBasicBlockList().push_back( finBlock);
      cg.createForStatement(initialBlock, forTestBlock, forTestBlock, forTest, GTLCore::Type::Boolean, updateBlock, startBodyBlock, bodyBlock, finBlock);
      // return;
      OCTL_DEBUG("return;");
      llvm::ReturnInst::Create(d->module->getContext(), finBlock);
      OCTL_DEBUG(*d->module);
      // Optimize
      OCTL_DEBUG("Optimize");
      
      // Register module in the VM
      GTLCore::VirtualMachine::instance()->registerModule( d->module );
      GTLCore::VirtualMachine::instance()->executionEngine()->clearAllGlobalMappings();
      d->func = ( void(*)(const char**, char*,int,GTLCore::ProgressReport*, gtl_uint64)) GTLCore::VirtualMachine::instance()->getPointerToFunction( func );
      for( std::map< GTLCore::String, llvm::GlobalVariable*>::iterator it = d->varyings.begin(); it != d->varyings.end(); ++it)
      {
          d->varyingsPtr[ it->first ] = GTLCore::VirtualMachine::instance()->getGlobalVariablePointer( it->second );
      }
//       OCTL_DEBUG( ((void*)d->func) << " ==== " << func << "  " << func->isDeclaration() << *func);
      // Cleanup
      for(int i = 0; i < dst.countChannels; ++i)
      {
          delete buffer[i];
      }
      GTLCore::deleteAll(srcs);
      delete[] buffer;
    } else {
      OCTL_DEBUG("Function: " << functionName << " not found in module");
      delete d->module;
      d->module = 0;
    }
  } else {
    OCTL_DEBUG("No module was supplied");
    d->module = 0;
  }
}

Program::~Program()
{
  if(d->module)
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->module);
  }
  delete d->moduleData;
  delete d;
}

bool Program::isInitialised() const
{
  return d->func;
}

void Program::apply(const GTLCore::Buffer& input, GTLCore::Buffer& output, GTLCore::ProgressReport* _report, const GTLCore::ChannelsFlags& flags ) const
{
  GTL_ASSERT(d->srcPixelDescriptions.size() == 1);
  OCTL_ASSERT( (input.size() / (d->srcPixelDescriptions.begin()->bitsSize() / 8) ) == (output.size() / (d->dstPixelDescription.bitsSize() / 8) ) );
  OCTL_ASSERT( d->func );
  OCTL_ASSERT( (input.size() % (d->srcPixelDescriptions.begin()->bitsSize() / 8)) == 0 );
  OCTL_ASSERT( (output.size() % (d->dstPixelDescription.bitsSize() / 8)) == 0 );
  OCTL_DEBUG("Apply program on a buffer of size " << input.size() << " for a source pixel of size " << (d->srcPixelDescriptions.begin()->bitsSize() / 8) << " for a destination pixel of size " << (d->dstPixelDescription.bitsSize() / 8));
  const char * in[1];
  in[0] = input.rawData();
  d->func( in, output.rawData(), input.size(), _report, flags.value() );
  OCTL_DEBUG("Applied");
}

void Program::apply(const std::list<GTLCore::Buffer*>& inputs, GTLCore::Buffer& output, GTLCore::ProgressReport* _report, const GTLCore::ChannelsFlags& flags ) const
{
  GTL_ASSERT(d->srcPixelDescriptions.size() == inputs.size());
  OCTL_ASSERT( (output.size() % (d->dstPixelDescription.bitsSize() / 8)) == 0 );
  OCTL_ASSERT( d->func );
  const char** ins = new const char*[ inputs.size() ];
  int idx = 0;
  for(std::list<GTLCore::Buffer*>::const_iterator it = inputs.begin(); it != inputs.end(); ++it, ++idx )
  {
    ins[idx] = (*it)->rawData();
//     OCTL_ASSERT( ((*it)->size() / (d->srcPixelDescription.bitsSize() / 8) ) == (output.size() / (d->dstPixelDescription.bitsSize() / 8) ) );
  }
  d->func(ins, output.rawData(), (*inputs.begin())->size(), _report, flags.value());
}


void Program::apply(const GTLCore::AbstractImage& input, GTLCore::AbstractImage& output, GTLCore::ProgressReport* report, const GTLCore::ChannelsFlags& flags ) const
{
  GTLCore::AbstractImage::ConstIterator* itSrc = input.createIterator();
  GTLCore::AbstractImage::Iterator* itDst = output.createIterator();
  
  while( itSrc->next() and itDst->next())
  {
    apply( *itSrc->current(), *itDst->current(), report, flags );
  }
  
  delete itDst;
  delete itSrc;
}

void Program::setVarying( const GTLCore::String& _name, const GTLCore::Value& _value )
{
  std::map< GTLCore::String, void*>::iterator it = d->varyingsPtr.find( _name );
  std::map< GTLCore::String, llvm::GlobalVariable*>::iterator it2 = d->varyings.find( _name );
  if( it != d->varyingsPtr.end() )
  {
    void* ptr = it->second;
    if( it2->second->getType()->getElementType() == llvm::Type::getInt32Ty(d->module->getContext()) )
    {
        *(int*)ptr =_value.asInt32();
    } else if( it2->second->getType()->getElementType() == llvm::Type::getInt1Ty(d->module->getContext()) )
    {
        *(bool*)ptr = _value.asBoolean();
    } else if( it2->second->getType()->getElementType() == llvm::Type::getFloatTy(d->module->getContext()) )
    {
        GTL_DEBUG("Set " << _value.asFloat32() << " on ptr " << ptr << " from value = " << *(float*)ptr);
        *(float*)ptr = _value.asFloat32();
        GTL_DEBUG( *(float*)ptr );
    }
  } else {
    OCTL_DEBUG(" No varying named: " << _name);
  }
}

GTLCore::Value Program::varying( const GTLCore::String& _name ) const
{
  std::map< GTLCore::String, void*>::iterator it = d->varyingsPtr.find( _name );
  std::map< GTLCore::String, llvm::GlobalVariable*>::iterator it2 = d->varyings.find( _name );
  if( it != d->varyingsPtr.end() )
  {
    void* ptr = it->second;
    if( it2->second->getType()->getElementType() == llvm::Type::getInt32Ty(d->module->getContext()) )
    {
        return GTLCore::Value( *(gtl_int32*)ptr);
    }
    if( it2->second->getType()->getElementType() == llvm::Type::getInt1Ty(d->module->getContext()) )
    {
        return GTLCore::Value( *(bool*)ptr);
    }
    if( it2->second->getType()->getElementType() == llvm::Type::getFloatTy(d->module->getContext()) )
    {
        return GTLCore::Value( *(float*)ptr);
    }
    OCTL_DEBUG("Invalid type");
    return GTLCore::Value();
  }
  OCTL_DEBUG(" No varying named: '" << _name << "'");
  return GTLCore::Value();
}

#if 0

void Program::setVarying( const GTLCore::String& _name, const GTLCore::Value& _value )
{
  std::map< GTLCore::String, llvm::GlobalVariable*>::iterator it = d->varyings.find( _name );
  if( it != d->varyings.end() )
  {
    GTLCore::VirtualMachine::instance()->setGlobalVariable( it->second, _value);
  } else {
    OCTL_DEBUG(" No varying named: " << _name);
  }
}

GTLCore::Value Program::varying( const GTLCore::String& _name ) const
{
  std::map< GTLCore::String, llvm::GlobalVariable*>::iterator it = d->varyings.find( _name );
  if( it != d->varyings.end() )
  {
    return GTLCore::VirtualMachine::instance()->getGlobalVariable( it->second);
  }
  OCTL_DEBUG(" No varying named: '" << _name << "'");
  return GTLCore::Value();
}

#endif

const std::list<GTLCore::String>& Program::varyings() const
{
    return d->varyingsName;
}
