/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConstantCompoundExpression.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "../LLVMBackend/CodeGenerator_p.h"
#include "../Debug.h"
#include "../Macros_p.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../LLVMBackend/ExpressionResult_p.h"
#include "../LLVMBackend/ExpressionGenerationContext_p.h"
#include "GenerationVisitor.h"

using namespace GTLCore::AST;

ConstantCompoundExpression::ConstantCompoundExpression( const GTLCore::Type* _type, const std::vector<Expression*>& _expressions) : m_expressions(_expressions), m_type(_type)
{
  GTL_ASSERT(_type);
  GTL_ASSERT(not m_expressions.empty());
  GTL_DEBUG(m_type);
}

ConstantCompoundExpression::~ConstantCompoundExpression( )
{
  for( std::vector<Expression*>::iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
  {
    delete *it;
  }
}

const GTLCore::Type* ConstantCompoundExpression::type() const
{
  GTL_DEBUG(*m_type);
  return m_type;
}

bool ConstantCompoundExpression::isConstant() const
{
  return m_constant;
}

LLVMBackend::ExpressionResult ConstantCompoundExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  GTL_DEBUG( m_type->dataType() << " " << Type::ARRAY << " " << Type::STRUCTURE );
  if( m_type->dataType() == Type::ARRAY or m_type->dataType() == Type::VECTOR)
  {
    const GTLCore::Type* arrayType = m_type->embeddedType();
    std::vector< llvm::Constant* > members;
    for(  std::vector<Expression*>::const_iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( (*it)->generateValue( _gc, _egc).constant(), (*it)->type(), arrayType ) );
    }
    GTL_ASSERT(members.size() == m_expressions.size());
    if( m_type->dataType() == Type::ARRAY )
    {
      return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::constantsToArray(_gc, members, m_type) , type() );
    } else if( m_type->dataType() == Type::VECTOR )
    {
      llvm::Constant* constant = LLVMBackend::CodeGenerator::constantsToVector(_gc, members, m_type);
      GTL_DEBUG( "Coumpound constant = " << *constant << " type = " << *m_type );
      return LLVMBackend::ExpressionResult( constant, m_type );
    }
  } else if( m_type->dataType() == Type::STRUCTURE )
  {
    std::vector<llvm::Constant*> members;
    members.push_back( LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), INT32_C(0) ) ); // GC constant
    for( std::size_t i = 0; i < m_expressions.size(); ++i)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( m_expressions[i]->generateValue( _gc, _egc).constant(), m_expressions[i]->type(), m_type->structDataMember(i).type() ) );
    }
    return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::constantsToStructure(_gc, members, m_type), type() );
  }
  GTL_ABORT("Can't generate coumpound value, neither array, neither vector.");
  return LLVMBackend::ExpressionResult();
}

ExpressionResultSP ConstantCompoundExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  std::vector<ExpressionResultSP> results;
  results.reserve(m_expressions.size());
  foreach( Expression* expr, m_expressions)
  {
    results.push_back( expr->generateValue(_generationVisitor) );
  }
  return _generationVisitor->createCompoundConstant( results, m_type, annotation());
}

void ConstantCompoundExpression::markAsReturnExpression()
{
  GTL_ABORT("ConstantCompoundExpression can't be in a return statement");
}

Expression* ConstantCompoundExpression::expressionAt(unsigned int _idx )
{
  return m_expressions[ _idx ];
}
