/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "GenerationVisitor.h"
#include "ExpressionResult.h"

#include "../Debug.h"
#include "../Macros_p.h"
#include "../String.h"
#include "../Type.h"
#include "../Value.h"

using namespace GTLCore;
using namespace ASTBackend;

GenerationVisitor::GenerationVisitor()
{
}

GenerationVisitor::GenerationVisitor(const GenerationVisitor& )
{
}

GenerationVisitor& GenerationVisitor::operator=(const GenerationVisitor& )
{
  return *this;
}

GenerationVisitor::~GenerationVisitor()
{
}

AST::ExpressionResultSP GenerationVisitor::generateFloat32(float arg1, const AST::Annotation& )
{
  return new ExpressionResult( arg1 );
}

AST::ExpressionResultSP GenerationVisitor::generateInteger32(gtl_int32 arg1, const AST::Annotation& )
{
  return new ExpressionResult( arg1 );
}

AST::ExpressionResultSP GenerationVisitor::generateBoolean(bool arg1, const AST::Annotation& )
{
  return new ExpressionResult( arg1 );
}

AST::ExpressionResultSP GenerationVisitor::generateFunctionCall(Function* _function, const std::list<AST::ExpressionResultSP>& _arguments, const AST::Annotation& _annotation)
{
  reportError("Cannot call function", _annotation);
  return new ExpressionResult( Value() );
}

AST::ExpressionResultSP GenerationVisitor::generateString(String arg1, const AST::Annotation& _annotation )
{
  reportError("Cannot generate a string", _annotation);
  return new ExpressionResult( Value() );
}

#define GET_VALUE( _ARG_N_ ) \
  const Value& val##_ARG_N_ = arg##_ARG_N_.scast< ExpressionResult >( )->value();

#define MAKE_BOOLEAN_EXPRESSION( _EXPR_NAME_, _EXPR_OP_ ) \
  AST::ExpressionResultSP GenerationVisitor::_EXPR_NAME_( AST::ExpressionResultSP arg1, AST::ExpressionResultSP arg2, const AST::Annotation&) \
  { \
    GET_VALUE( 1) \
    GET_VALUE( 2) \
    return new ExpressionResult( val1.asBoolean() _EXPR_OP_ val2.asBoolean() ); \
  }

MAKE_BOOLEAN_EXPRESSION(generateAndExpresion, and)
MAKE_BOOLEAN_EXPRESSION(generateOrExpresion, or)

#define MAKE_INTEGER_BINARY_EXPRESSION( _EXPR_NAME_, _EXPR_OP_ ) \
  AST::ExpressionResultSP GenerationVisitor::_EXPR_NAME_( AST::ExpressionResultSP arg1, AST::ExpressionResultSP arg2, const AST::Annotation&) \
  { \
    GET_VALUE( 1) \
    GET_VALUE( 2) \
    return new ExpressionResult( val1.asInt32() _EXPR_OP_ val2.asInt32() ); \
  }

MAKE_INTEGER_BINARY_EXPRESSION(generateBitOrExpresion, |)
MAKE_INTEGER_BINARY_EXPRESSION(generateBitXorExpresion, ^)
MAKE_INTEGER_BINARY_EXPRESSION(generateBitAndExpresion, &)
MAKE_INTEGER_BINARY_EXPRESSION(generateRightShiftExpresion, >>)
MAKE_INTEGER_BINARY_EXPRESSION(generateLeftShiftExpresion, <<)
MAKE_INTEGER_BINARY_EXPRESSION(generateModuloExpresion, %)

#define MAKE_BINARY_EXPRESSION( _EXPR_NAME_, _EXPR_OP_ ) \
  AST::ExpressionResultSP GenerationVisitor::_EXPR_NAME_( AST::ExpressionResultSP arg1, AST::ExpressionResultSP arg2, const AST::Annotation&) \
  { \
    GET_VALUE( 1) \
    GET_VALUE( 2) \
    if( val1.type() == Type::Float32 or val2.type() == Type::Float32 ) \
    { \
      return new ExpressionResult( val1.asFloat32() _EXPR_OP_ val2.asFloat32() ); \
    } else { \
      return new ExpressionResult( val1.asInt32() _EXPR_OP_ val2.asInt32() ); \
    } \
  }

MAKE_BINARY_EXPRESSION(generateEqualExpresion, ==);
MAKE_BINARY_EXPRESSION(generateDifferentExpresion, !=);
MAKE_BINARY_EXPRESSION(generateInferiorExpresion, <);
MAKE_BINARY_EXPRESSION(generateInferiorEqualExpresion, <=);
MAKE_BINARY_EXPRESSION(generateSupperiorExpresion, >);
MAKE_BINARY_EXPRESSION(generateSupperiorEqualExpresion, >=);
MAKE_BINARY_EXPRESSION(generateAdditionExpresion, +);
MAKE_BINARY_EXPRESSION(generateSubtractionExpresion, -);
MAKE_BINARY_EXPRESSION(generateMultiplicationExpresion, *);
MAKE_BINARY_EXPRESSION(generateDivisionExpresion, /);

AST::ExpressionResultSP GenerationVisitor::generateTildExpression(AST::ExpressionResultSP arg1, const AST::Annotation& _annotation)
{
  GET_VALUE(1)
  return new ExpressionResult( ~val1.asInt32() );
}

AST::ExpressionResultSP GenerationVisitor::generateMinusExpression(AST::ExpressionResultSP arg1, const AST::Annotation& _annotation)
{
  GET_VALUE(1);
  if(val1.type() == Type::Float32)
  {
    return new ExpressionResult( -val1.asFloat32());
  } else {
    return new ExpressionResult( -val1.asInt32());
  }
}

AST::ExpressionResultSP GenerationVisitor::generateNotExpression(AST::ExpressionResultSP arg1, const AST::Annotation& _annotation)
{
  GET_VALUE(1);
  return new ExpressionResult(!val1.asBoolean());
}

AST::ExpressionResultSP GenerationVisitor::convertExpressionTo(AST::ExpressionResultSP arg1, const GTLCore::Type* _type, const AST::Annotation& _annotation)
{
  GET_VALUE(1);
  switch(_type->dataType())
  {
    case Type::BOOLEAN:
      return new ExpressionResult(val1.asBoolean());
    case Type::INTEGER32:
      return new ExpressionResult(val1.asInt32());
    case Type::FLOAT32:
      return new ExpressionResult(val1.asFloat32());
    default:
      break;
  }
  GTL_ABORT("Unsupported");
}

AST::ExpressionResultSP GenerationVisitor::convertArrayToVector(AST::ExpressionResultSP arg1, const GTLCore::Type* type, const AST::Annotation& _annotation)
{
  GET_VALUE(1);
  return new ExpressionResult( Value( *val1.asArray(), type ) );
}

AST::ExpressionResultSP GenerationVisitor::createCompoundConstant(std::vector< AST::ExpressionResultSP> arg1, const Type* type, const AST::Annotation& arg2)
{
  std::vector< Value > values;
  foreach(AST::ExpressionResultSP val, arg1)
  {
    values.push_back( val.scast<ExpressionResult>()->value() );
  }
  GTL_ASSERT( type->dataType() != Type::STRUCTURE or values.size() == type->countStructDataMembers());
  GTL_ASSERT( type->dataType() != Type::VECTOR or values.size() == type->vectorSize() );
  return new ExpressionResult( Value(values, type) );
}

AST::ExpressionResultSP GenerationVisitor::createCompoundValue(std::vector< AST::ExpressionResultSP> arg1, const Type*, const AST::Annotation& _annotation)
{
  GTL_ABORT("Unimplemented");
  return 0;
}
