/*
 * Copyright (C) 2010 Neil Jagdish Patel
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

namespace GwibberGtk
{
  public class StreamView : Gtk.HBox
  {
    private Gtk.Adjustment     adjustment;
    private Gtk.VScrollbar     scrollbar;
    private Gtk.Layout         layout;
    private Gtk.VBox           view_box;

    private Dee.Model? _model = null;
    private uint _position = 0;

    public Dee.Model model {
      get { return _model; }
      set {
        if (_model is Dee.Model)
          {
            _model.row_added.disconnect (on_row_added);
            _model.row_removed.disconnect (on_row_removed);
            _model.row_changed.disconnect (on_row_changed);
          }

        _model = value;
        _model.row_added.connect (on_row_added);
        _model.row_removed.connect (on_row_removed);
        _model.row_changed.connect (on_row_changed);

        adjustment.set_upper (_model.get_n_rows ());
        refresh ();
      }
    }

    public uint position {
      get { return _position; }
      set {
        if (_position != value)
          {
            _position = value;
            value = value.clamp (0, _model.get_n_rows ());
            adjustment.set_value (value);
            refresh ();
          }
      }
    }

    private SList<StreamViewTile> tiles;

    private int last_width = 0;
    private int last_height = 0;

    public StreamView ()
    {
      Object (homogeneous:false, spacing:6);
    }

    ~StreamView ()
    {

    }

    construct
    {
      adjustment = new Gtk.Adjustment (0, 0, 1, 1, 1, 1);
      adjustment.notify["value"].connect (()=> {
        _position = (int)adjustment.value;

        refresh ();
      });

      layout = new Gtk.Layout (null, null);
      pack_start (layout, true, true, 0);
      layout.expose_event.connect (on_layout_exposed);
                 
      view_box = new Gtk.VBox (false, 8);
      layout.put (view_box, 0, 0);

      scrollbar = new Gtk.VScrollbar (adjustment);
      pack_start (scrollbar, false, false, 0);

      for (int i = 0; i < 3; i++)
        {
          var tile = new StreamViewTile ();
          tile.show_all ();
          view_box.pack_start (tile, false, false, 0);
          tiles.append (tile);
        }

      layout.size_allocate.connect_after ((a) => {
        if (last_width != a.width
            || last_height != a.height)
          {
            last_width = a.width;
            last_height = a.height;
            view_box.set_size_request (last_width, -1);

            refresh ();
          }
      });
    }

    /*
     * This refreshes the view with the latest model and position
     */
    private void refresh ()
    {
      uint i = 0;
      uint n_rows = _model.get_n_rows ();

      if (!(_model is Dee.Model))
        {
          foreach (StreamViewTile tile in tiles)
            tile.reset ();
          return;
        }

      unowned Dee.ModelIter? iter;
      iter = _model.get_iter_at_row (_position);
      
      foreach (StreamViewTile tile in tiles)
        {
          if (iter != null && position + i < n_rows)
            {
              tile.set_details (_position + i,
                                _model.get_string (iter, StreamModelColumn.SERVICE),
                                _model.get_string (iter, StreamModelColumn.NAME),
                                _model.get_uint32 (iter, StreamModelColumn.TIMESTAMP),
                                _model.get_string (iter, StreamModelColumn.MESSAGE),
                                _model.get_string (iter, StreamModelColumn.ICON_URI),
                                _model.get_string (iter, StreamModelColumn.COLOR));
            }
          else
            {
              tile.reset ();
            }
          i++;
          if (position + i > n_rows)
            iter = null;
          else
            iter = _model.get_iter_at_row (_position + i);
        }

      Timeout.add (0, () => {
        queue_draw ();
        fill_up_remaining_space ();
        return false;
      });
    }

    private void fill_up_remaining_space ()
    {
      Gtk.Allocation alloc;
      int overall_height = 0;
      bool one_was_hidden = false;

      layout.get_allocation (out alloc);

      foreach (StreamViewTile tile in tiles)
        {
          Gtk.Allocation a;

          if (tile.get_visible ())
            {
              tile.get_allocation (out a);
              overall_height += a.height + 8;
            }
          else
            one_was_hidden = true;
        }

      if (alloc.height > overall_height && !one_was_hidden)
        {
          var tile = new StreamViewTile ();
          tile.show_all ();
          view_box.pack_start (tile, false, false, 0);
          tiles.append (tile);

          refresh ();
        }
    }

    private bool on_layout_exposed (Gdk.EventExpose event)
    {
      return false;
    }

    private void on_row_added ()
    {
      refresh ();
    }

    private void on_row_removed ()
    {
      refresh ();
    }

    private void on_row_changed ()
    {
      refresh ();
    }
  }
}
