/* -*- mode: C; c-basic-offset: 4 -*- */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <unistd.h>
#include <string.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-ui-engine.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-window.h>
#include <launchpad-integration.h>

/* ----------------------------- */

/* wrapper for gettext that sets up our translation domain on first use */
static void
initialise_gettext_domain (void)
{
    static gboolean initialised = FALSE;

    if (initialised) return;

    bindtextdomain (PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif
    initialised = TRUE;
}

static G_CONST_RETURN char *
lpint_gettext (const char *str)
{
    initialise_gettext_domain();

    return dgettext (PACKAGE, str);
}

#undef gettext
#define gettext lpint_gettext


#define LP_GET_HELP_ITEM_LABEL   N_("Get Help Online...")
#define LP_TRANSLATE_ITEM_LABEL  N_("Translate This Application...")
#define LP_REPORT_BUG_ITEM_LABEL N_("Report a Problem...")


/* ------------------------------ */

static gchar *
escape_slashes (const gchar *text)
{
    GString *str;
    gint length;
    const gchar *p;
    const gchar *end;

    g_return_val_if_fail (text != NULL, NULL);

    length = strlen (text);

    str = g_string_sized_new (length);

    p = text;
    end = text + length;

    while (p != end) {
	const gchar *next;
	next = g_utf8_next_char (p);

	if (*p == '/')
	    g_string_append (str, "-");
	else
	    g_string_append_len (str, p, next - p);

	p = next;
    }

    return g_string_free (str, FALSE);
}

static void
lpint_bonobo_add_menu_item (BonoboUIComponent *ui_component, const gchar *path,
			    const gchar *name, const gchar *label, gboolean icon_type,
			    const char *icon, BonoboUIVerbFn cb)
{
    gchar *escaped_name;
    gchar *item_path;
    gchar *cmd;

    g_return_if_fail (path != NULL);
    g_return_if_fail (label != NULL);
    g_return_if_fail (cb != NULL);
	
    escaped_name = escape_slashes (name);

    item_path = g_strconcat (path, escaped_name, NULL);
    if (!bonobo_ui_component_path_exists (ui_component, item_path, NULL)) {
	GString *xml;

	xml = g_string_new("");
	g_string_printf (xml,"<menuitem name=\"%s\" verb=\"\""
			       " label=\"%s\" hidden=\"0\" ",
			       escaped_name, _(label));
	if(icon != NULL) {
	    if (icon_type == 0) {
		GtkIconTheme *icon_theme;
		GError *error = NULL;
		GdkPixbuf *pixbuf;

		icon_theme = gtk_icon_theme_get_default ();
		pixbuf = gtk_icon_theme_load_icon (icon_theme,
						   icon, /* icon name */
						   16,
						   0,
						   &error);
		if (pixbuf != NULL) {
		    g_string_append_printf(xml," pixtype=\"pixbuf\" pixname=\"%s\" ", bonobo_ui_util_pixbuf_to_xml (pixbuf));
		    g_object_unref (pixbuf);
		}
	    }
	    else
		g_string_append_printf(xml," pixtype=\"stock\" pixname=\"%s\" ", icon);
	}
	g_string_append(xml,"/>");

	cmd = g_strdup_printf ("<cmd name=\"%s\" />", escaped_name);

	bonobo_ui_component_set_translate (ui_component, path, xml->str, NULL);

	bonobo_ui_component_set_translate (ui_component, "/commands/",
					   cmd, NULL);

	bonobo_ui_component_add_verb (ui_component, escaped_name, cb, NULL);

	g_string_free (xml, TRUE);
	g_free (cmd);
    }

    g_free (item_path);
    g_free (escaped_name);
}


/*
 * callbacks used to  invoke launchpad on the given URLs:
 *
 * + void            launchpad_integration_show_info         (void);
 * + void            launchpad_integration_show_translations (void);
 * + void            launchpad_integration_show_bugs         (void);
 *
 * */

/* bonobo compatible callbacks wrappers for ^^^^ */

static void
lpint_bonobo_show_info_cb (BonoboUIComponent *uic, gpointer user_data,
			   const gchar* verbname)
{
    launchpad_integration_show_info();
}

static void
lpint_bonobo_show_translations_cb (BonoboUIComponent *uic, gpointer user_data,
				   const gchar* verbname)
{
    launchpad_integration_show_translations();
}

static void
lpint_bonobo_show_bugs_cb (BonoboUIComponent *uic, gpointer user_data,
			   const gchar* verbname)
{
    launchpad_integration_show_bugs();
}

/* ---------------------- */



void
launchpad_integration_add_bonobo_ui (BonoboUIComponent *ui, const gchar *path)
{
    launchpad_integration_add_item_factory ();
    
    lpint_bonobo_add_menu_item (ui, path, "LpShowInfo",
				LP_GET_HELP_ITEM_LABEL, 0,
				"lpi-help",
				lpint_bonobo_show_info_cb);
    
    lpint_bonobo_add_menu_item (ui, path, "LpShowTranslations",
				LP_TRANSLATE_ITEM_LABEL, 0,
				"lpi-translate",
				lpint_bonobo_show_translations_cb);

#if 1 // Disable this on stable releases
    lpint_bonobo_add_menu_item (ui, path, "LpShowBugs",
				LP_REPORT_BUG_ITEM_LABEL, 0,
				"lpi-bug",
				lpint_bonobo_show_bugs_cb);
#endif
}
