#line 1 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
/*
 * parse SVG path specifications
 *
 * Copyright 2007 MenTaLguY <mental@rydia.net>
 * Copyright 2007 Aaron Spike <aaron@ekips.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it either under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation
 * (the "LGPL") or, at your option, under the terms of the Mozilla
 * Public License Version 1.1 (the "MPL"). If you do not alter this
 * notice, a recipient may use your version of this file under either
 * the MPL or the LGPL.
 *
 * You should have received a copy of the LGPL along with this library
 * in the file COPYING-LGPL-2.1; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * You should have received a copy of the MPL along with this library
 * in the file COPYING-MPL-1.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY
 * OF ANY KIND, either express or implied. See the LGPL or the MPL for
 * the specific language governing rights and limitations.
 *
 */


#include <cmath>
#include <vector>
#include <glib.h>

#include "point.h"

#include "svg-path-parser.h"

namespace Geom {

namespace {

class Parser {
public:
    Parser(SVGPathSink &sink) : _sink(sink) {}

    void parse(char const *str) throw(SVGPathParseError);

private:
    bool _absolute;
    Point _current;
    Point _initial;
    Point _cubic_tangent;
    Point _quad_tangent;
    std::vector<double> _params;
    SVGPathSink &_sink;

    void _reset() {
        _absolute = false;
        _current = _initial = Point(0, 0);
        _quad_tangent = _cubic_tangent = Point(0, 0);
        _params.clear();
    }

    void _push(double value) {
        _params.push_back(value);
    }

    double _pop() {
        double value = _params.back();
        _params.pop_back();
        return value;
    }

    bool _pop_flag() {
        return _pop() != 0.0;
    }

    double _pop_coord(Geom::Dim2 axis) {
        if (_absolute) {
            return _pop();
        } else {
            return _pop() + _current[axis];
        }
    }

    Point _pop_point() {
        double y = _pop_coord(Geom::Y);
        double x = _pop_coord(Geom::X);
        return Point(x, y);
    }

    void _moveTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = _initial = p;
        _sink.moveTo(p);
    }

    void _lineTo(Point p) {
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.lineTo(p);
    }

    void _curveTo(Point c0, Point c1, Point p) {
        _quad_tangent = _current = p;
        _cubic_tangent = p + ( p - c1 );
        _sink.curveTo(c0, c1, p);
    }

    void _quadTo(Point c, Point p) {
        _cubic_tangent = _current = p;
        _quad_tangent = p + ( p - c );
        _sink.quadTo(c, p);
    }

    void _arcTo(double rx, double ry, double angle,
                bool large_arc, bool sweep, Point p)
    {
        _quad_tangent = _cubic_tangent = _current = p;
        _sink.arcTo(rx, ry, angle, large_arc, sweep, p);
    }

    void _closePath() {
        _quad_tangent = _cubic_tangent = _current = _initial;
        _sink.closePath();
    }
};


#line 133 "/home/mental/trees/lib2geom/src/svg-path-parser.cpp"
static const char _svg_path_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 5, 1, 15, 1, 
	16, 2, 1, 0, 2, 1, 2, 2, 
	1, 3, 2, 1, 6, 2, 1, 7, 
	2, 1, 8, 2, 1, 9, 2, 1, 
	10, 2, 1, 11, 2, 1, 12, 2, 
	1, 13, 2, 1, 14, 2, 2, 1, 
	2, 3, 1, 2, 4, 0, 2, 5, 
	0, 2, 15, 16, 3, 1, 6, 0, 
	3, 1, 6, 16, 3, 1, 7, 0, 
	3, 1, 7, 16, 3, 1, 8, 0, 
	3, 1, 8, 16, 3, 1, 9, 0, 
	3, 1, 9, 16, 3, 1, 10, 0, 
	3, 1, 10, 16, 3, 1, 11, 0, 
	3, 1, 11, 16, 3, 1, 12, 0, 
	3, 1, 12, 16, 3, 1, 13, 0, 
	3, 1, 13, 16, 3, 1, 14, 0, 
	3, 1, 14, 16
};

static const short _svg_path_key_offsets[] = {
	0, 7, 7, 16, 25, 28, 30, 42, 
	52, 55, 57, 90, 121, 124, 126, 138, 
	148, 151, 153, 186, 195, 207, 216, 249, 
	256, 263, 265, 275, 283, 290, 292, 304, 
	314, 317, 319, 328, 335, 341, 346, 353, 
	359, 364, 374, 377, 379, 391, 401, 404, 
	406, 437, 466, 476, 488, 498, 507, 509, 
	521, 533, 544, 554, 561, 567, 572, 584, 
	595, 607, 617, 620, 622, 655, 664, 695, 
	704, 713, 716, 718, 730, 740, 743, 745, 
	757, 767, 770, 772, 784, 794, 797, 799, 
	811, 821, 824, 826, 838, 848, 851, 853, 
	886, 917, 929, 938, 950, 959, 971, 980, 
	992, 1001, 1013, 1022, 1055, 1059, 1061, 1092, 
	1101, 1110, 1113, 1115, 1148, 1179, 1182, 1184, 
	1217, 1226, 1259, 1263, 1265, 1296, 1305, 1314, 
	1323, 1326, 1328, 1340, 1350, 1353, 1355, 1367, 
	1377, 1380, 1382, 1394, 1404, 1407, 1409, 1442, 
	1473, 1485, 1494, 1506, 1515, 1527, 1536, 1569, 
	1573, 1575, 1606, 1615, 1624, 1627, 1629, 1641, 
	1651, 1654, 1656, 1668, 1678, 1681, 1683, 1695, 
	1705, 1708, 1710, 1743, 1774, 1786, 1795, 1807, 
	1816, 1828, 1837, 1870, 1874, 1876, 1907, 1916, 
	1925, 1928, 1930, 1942, 1952, 1955, 1957, 1990, 
	2021, 2033, 2042, 2075, 2079, 2081, 2112, 2121, 
	2130, 2133, 2135, 2168, 2199, 2202, 2204, 2237, 
	2246, 2279, 2283, 2285, 2316, 2341, 2366, 2373, 
	2382, 2391, 2424, 2428, 2430, 2461, 2494, 2503, 
	2512, 2524, 2533, 2566, 2570, 2572, 2603, 2612, 
	2621, 2630, 2639, 2672, 2676, 2678, 2709, 2742, 
	2746, 2748, 2758, 2791, 2795, 2797, 2807, 2811, 
	2813, 2823, 2827, 2829, 2839, 2843, 2845, 2855, 
	2859, 2861, 2871, 2875, 2877, 2887, 2891, 2893, 
	2903, 2936, 2940, 2942, 2952, 2956, 2958, 2968, 
	2972, 2974, 2984, 2988, 2990, 3000, 3004, 3006, 
	3016, 3020, 3022, 3051, 3055, 3057, 3088, 3121, 
	3130, 3142, 3146, 3148, 3158, 3170, 3175, 3185, 
	3189, 3191, 3198, 3208, 3212, 3214, 3224, 3236, 
	3248, 3260, 3264, 3266, 3276, 3288, 3292, 3294, 
	3304, 3313, 3317, 3319, 3327, 3336, 3341, 3346, 
	3358, 3362, 3364, 3395, 3399, 3401, 3411
};

static const char _svg_path_trans_keys[] = {
	0, 13, 32, 77, 109, 9, 10, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 46, 9, 10, 48, 57, 
	13, 32, 46, 9, 10, 48, 57, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 48, 57, 13, 32, 44, 46, 9, 
	10, 48, 57, 13, 32, 46, 9, 10, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 69, 101, 9, 10, 48, 57, 
	13, 32, 44, 48, 49, 9, 10, 13, 
	32, 48, 49, 9, 10, 13, 32, 44, 
	9, 10, 13, 32, 44, 48, 49, 9, 
	10, 13, 32, 48, 49, 9, 10, 13, 
	32, 44, 9, 10, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 46, 48, 
	57, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 44, 46, 48, 49, 9, 
	10, 43, 45, 50, 57, 13, 32, 43, 
	45, 46, 48, 49, 9, 10, 50, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	48, 57, 13, 32, 44, 48, 49, 9, 
	10, 13, 32, 48, 49, 9, 10, 13, 
	32, 44, 9, 10, 13, 32, 44, 46, 
	48, 49, 9, 10, 43, 45, 50, 57, 
	13, 32, 43, 45, 46, 48, 49, 9, 
	10, 50, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 46, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 9, 10, 43, 45, 
	48, 57, 46, 48, 57, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 44, 46, 9, 
	10, 43, 45, 48, 57, 46, 48, 57, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	46, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 44, 46, 69, 101, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 46, 48, 
	57, 48, 57, 0, 13, 32, 44, 46, 
	65, 67, 69, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 101, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 46, 48, 57, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 69, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 101, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 43, 45, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 46, 48, 57, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 46, 48, 
	57, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 46, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	69, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 101, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 43, 45, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 44, 46, 69, 101, 9, 
	10, 43, 45, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 43, 45, 48, 57, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	46, 48, 57, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 44, 46, 9, 10, 43, 
	45, 48, 57, 46, 48, 57, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 46, 48, 
	57, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 44, 46, 9, 10, 43, 45, 48, 
	57, 46, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 43, 45, 46, 9, 
	10, 48, 57, 13, 32, 44, 46, 69, 
	101, 9, 10, 43, 45, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 43, 45, 
	48, 57, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 46, 48, 57, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	46, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 13, 32, 44, 
	46, 69, 101, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 0, 13, 32, 44, 46, 65, 
	67, 69, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 101, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 43, 45, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 46, 48, 57, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 69, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 101, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 46, 
	48, 57, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 69, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 101, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 72, 76, 77, 81, 
	83, 84, 86, 90, 97, 99, 104, 108, 
	109, 113, 115, 116, 118, 122, 9, 10, 
	43, 45, 48, 57, 0, 13, 32, 65, 
	67, 72, 76, 77, 81, 83, 84, 86, 
	90, 97, 99, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 0, 13, 32, 
	65, 67, 72, 76, 77, 81, 83, 84, 
	86, 90, 97, 99, 104, 108, 109, 113, 
	115, 116, 118, 122, 9, 10, 13, 32, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 69, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 101, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 43, 45, 
	48, 57, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 13, 32, 43, 45, 46, 
	9, 10, 48, 57, 13, 32, 43, 45, 
	46, 9, 10, 48, 57, 13, 32, 43, 
	45, 46, 9, 10, 48, 57, 13, 32, 
	43, 45, 46, 9, 10, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 69, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 101, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 43, 45, 48, 57, 0, 13, 32, 
	44, 46, 65, 67, 69, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 101, 
	104, 108, 109, 113, 115, 116, 118, 122, 
	9, 10, 43, 45, 48, 57, 43, 45, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 69, 72, 76, 
	77, 81, 83, 84, 86, 90, 97, 99, 
	101, 104, 108, 109, 113, 115, 116, 118, 
	122, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 43, 
	45, 48, 57, 48, 57, 13, 32, 44, 
	46, 9, 10, 43, 45, 48, 57, 0, 
	13, 32, 44, 46, 65, 67, 69, 72, 
	76, 77, 81, 83, 84, 86, 90, 97, 
	99, 101, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 0, 13, 
	32, 44, 46, 65, 67, 72, 76, 77, 
	81, 83, 84, 86, 90, 97, 99, 104, 
	108, 109, 113, 115, 116, 118, 122, 9, 
	10, 48, 57, 43, 45, 48, 57, 48, 
	57, 0, 13, 32, 44, 46, 65, 67, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 104, 108, 109, 113, 115, 116, 
	118, 122, 9, 10, 43, 45, 48, 57, 
	0, 13, 32, 44, 46, 65, 67, 69, 
	72, 76, 77, 81, 83, 84, 86, 90, 
	97, 99, 101, 104, 108, 109, 113, 115, 
	116, 118, 122, 9, 10, 43, 45, 48, 
	57, 13, 32, 43, 45, 46, 9, 10, 
	48, 57, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 43, 45, 
	48, 57, 48, 57, 13, 32, 44, 46, 
	9, 10, 43, 45, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 43, 45, 
	48, 57, 13, 32, 44, 9, 10, 13, 
	32, 44, 46, 69, 101, 9, 10, 48, 
	57, 43, 45, 48, 57, 48, 57, 13, 
	32, 44, 9, 10, 48, 57, 13, 32, 
	44, 46, 69, 101, 9, 10, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	13, 32, 44, 46, 69, 101, 9, 10, 
	43, 45, 48, 57, 43, 45, 48, 57, 
	48, 57, 13, 32, 44, 46, 9, 10, 
	43, 45, 48, 57, 13, 32, 44, 46, 
	69, 101, 9, 10, 43, 45, 48, 57, 
	43, 45, 48, 57, 48, 57, 13, 32, 
	44, 46, 9, 10, 43, 45, 48, 57, 
	13, 32, 43, 45, 46, 9, 10, 48, 
	57, 43, 45, 48, 57, 48, 57, 13, 
	32, 44, 46, 9, 10, 48, 57, 13, 
	32, 43, 45, 46, 9, 10, 48, 57, 
	13, 32, 44, 9, 10, 13, 32, 44, 
	9, 10, 13, 32, 44, 46, 69, 101, 
	9, 10, 43, 45, 48, 57, 43, 45, 
	48, 57, 48, 57, 0, 13, 32, 44, 
	46, 65, 67, 72, 76, 77, 81, 83, 
	84, 86, 90, 97, 99, 104, 108, 109, 
	113, 115, 116, 118, 122, 9, 10, 43, 
	45, 48, 57, 43, 45, 48, 57, 48, 
	57, 13, 32, 44, 46, 9, 10, 43, 
	45, 48, 57, 0
};

static const char _svg_path_single_lengths[] = {
	5, 0, 5, 5, 1, 0, 6, 4, 
	1, 0, 27, 25, 1, 0, 6, 4, 
	1, 0, 27, 5, 6, 5, 27, 3, 
	3, 0, 6, 4, 3, 0, 6, 4, 
	1, 0, 5, 5, 4, 3, 5, 4, 
	3, 4, 1, 0, 6, 4, 1, 0, 
	27, 25, 6, 6, 4, 5, 0, 6, 
	6, 7, 6, 5, 4, 3, 6, 7, 
	6, 4, 1, 0, 27, 5, 27, 5, 
	5, 1, 0, 6, 4, 1, 0, 6, 
	4, 1, 0, 6, 4, 1, 0, 6, 
	4, 1, 0, 6, 4, 1, 0, 27, 
	25, 6, 5, 6, 5, 6, 5, 6, 
	5, 6, 5, 27, 2, 0, 25, 5, 
	5, 1, 0, 27, 25, 1, 0, 27, 
	5, 27, 2, 0, 25, 5, 5, 5, 
	1, 0, 6, 4, 1, 0, 6, 4, 
	1, 0, 6, 4, 1, 0, 27, 25, 
	6, 5, 6, 5, 6, 5, 27, 2, 
	0, 25, 5, 5, 1, 0, 6, 4, 
	1, 0, 6, 4, 1, 0, 6, 4, 
	1, 0, 27, 25, 6, 5, 6, 5, 
	6, 5, 27, 2, 0, 25, 5, 5, 
	1, 0, 6, 4, 1, 0, 27, 25, 
	6, 5, 27, 2, 0, 25, 5, 5, 
	1, 0, 27, 25, 1, 0, 27, 5, 
	27, 2, 0, 25, 23, 23, 3, 5, 
	5, 27, 2, 0, 25, 27, 5, 5, 
	6, 5, 27, 2, 0, 25, 5, 5, 
	5, 5, 27, 2, 0, 25, 27, 2, 
	0, 4, 27, 2, 0, 4, 2, 0, 
	4, 2, 0, 4, 2, 0, 4, 2, 
	0, 4, 2, 0, 4, 2, 0, 4, 
	27, 2, 0, 4, 2, 0, 4, 2, 
	0, 4, 2, 0, 4, 2, 0, 4, 
	2, 0, 25, 2, 0, 25, 27, 5, 
	6, 2, 0, 4, 6, 3, 6, 2, 
	0, 3, 6, 2, 0, 4, 6, 6, 
	6, 2, 0, 4, 6, 2, 0, 4, 
	5, 2, 0, 4, 5, 3, 3, 6, 
	2, 0, 25, 2, 0, 4, 0
};

static const char _svg_path_range_lengths[] = {
	1, 0, 2, 2, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 2, 3, 2, 3, 2, 
	2, 1, 2, 2, 2, 1, 3, 3, 
	1, 1, 2, 1, 1, 1, 1, 1, 
	1, 3, 1, 1, 3, 3, 1, 1, 
	2, 2, 2, 3, 3, 2, 1, 3, 
	3, 2, 2, 1, 1, 1, 3, 2, 
	3, 3, 1, 1, 3, 2, 2, 2, 
	2, 1, 1, 3, 3, 1, 1, 3, 
	3, 1, 1, 3, 3, 1, 1, 3, 
	3, 1, 1, 3, 3, 1, 1, 3, 
	3, 3, 2, 3, 2, 3, 2, 3, 
	2, 3, 2, 3, 1, 1, 3, 2, 
	2, 1, 1, 3, 3, 1, 1, 3, 
	2, 3, 1, 1, 3, 2, 2, 2, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	3, 2, 3, 2, 3, 2, 3, 1, 
	1, 3, 2, 2, 1, 1, 3, 3, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	1, 1, 3, 3, 3, 2, 3, 2, 
	3, 2, 3, 1, 1, 3, 2, 2, 
	1, 1, 3, 3, 1, 1, 3, 3, 
	3, 2, 3, 1, 1, 3, 2, 2, 
	1, 1, 3, 3, 1, 1, 3, 2, 
	3, 1, 1, 3, 1, 1, 2, 2, 
	2, 3, 1, 1, 3, 3, 2, 2, 
	3, 2, 3, 1, 1, 3, 2, 2, 
	2, 2, 3, 1, 1, 3, 3, 1, 
	1, 3, 3, 1, 1, 3, 1, 1, 
	3, 1, 1, 3, 1, 1, 3, 1, 
	1, 3, 1, 1, 3, 1, 1, 3, 
	3, 1, 1, 3, 1, 1, 3, 1, 
	1, 3, 1, 1, 3, 1, 1, 3, 
	1, 1, 2, 1, 1, 3, 3, 2, 
	3, 1, 1, 3, 3, 1, 2, 1, 
	1, 2, 2, 1, 1, 3, 3, 3, 
	3, 1, 1, 3, 3, 1, 1, 3, 
	2, 1, 1, 2, 2, 1, 1, 3, 
	1, 1, 3, 1, 1, 3, 0
};

static const short _svg_path_index_offsets[] = {
	0, 7, 7, 15, 23, 26, 28, 38, 
	46, 49, 51, 82, 111, 114, 116, 126, 
	134, 137, 139, 170, 178, 188, 196, 227, 
	233, 239, 241, 250, 257, 263, 265, 275, 
	283, 286, 288, 296, 303, 309, 314, 321, 
	327, 332, 340, 343, 345, 355, 363, 366, 
	368, 398, 426, 435, 445, 453, 461, 463, 
	473, 483, 493, 502, 509, 515, 520, 530, 
	540, 550, 558, 561, 563, 594, 602, 632, 
	640, 648, 651, 653, 663, 671, 674, 676, 
	686, 694, 697, 699, 709, 717, 720, 722, 
	732, 740, 743, 745, 755, 763, 766, 768, 
	799, 828, 838, 846, 856, 864, 874, 882, 
	892, 900, 910, 918, 949, 953, 955, 984, 
	992, 1000, 1003, 1005, 1036, 1065, 1068, 1070, 
	1101, 1109, 1140, 1144, 1146, 1175, 1183, 1191, 
	1199, 1202, 1204, 1214, 1222, 1225, 1227, 1237, 
	1245, 1248, 1250, 1260, 1268, 1271, 1273, 1304, 
	1333, 1343, 1351, 1361, 1369, 1379, 1387, 1418, 
	1422, 1424, 1453, 1461, 1469, 1472, 1474, 1484, 
	1492, 1495, 1497, 1507, 1515, 1518, 1520, 1530, 
	1538, 1541, 1543, 1574, 1603, 1613, 1621, 1631, 
	1639, 1649, 1657, 1688, 1692, 1694, 1723, 1731, 
	1739, 1742, 1744, 1754, 1762, 1765, 1767, 1798, 
	1827, 1837, 1845, 1876, 1880, 1882, 1911, 1919, 
	1927, 1930, 1932, 1963, 1992, 1995, 1997, 2028, 
	2036, 2067, 2071, 2073, 2102, 2127, 2152, 2158, 
	2166, 2174, 2205, 2209, 2211, 2240, 2271, 2279, 
	2287, 2297, 2305, 2336, 2340, 2342, 2371, 2379, 
	2387, 2395, 2403, 2434, 2438, 2440, 2469, 2500, 
	2504, 2506, 2514, 2545, 2549, 2551, 2559, 2563, 
	2565, 2573, 2577, 2579, 2587, 2591, 2593, 2601, 
	2605, 2607, 2615, 2619, 2621, 2629, 2633, 2635, 
	2643, 2674, 2678, 2680, 2688, 2692, 2694, 2702, 
	2706, 2708, 2716, 2720, 2722, 2730, 2734, 2736, 
	2744, 2748, 2750, 2778, 2782, 2784, 2813, 2844, 
	2852, 2862, 2866, 2868, 2876, 2886, 2891, 2900, 
	2904, 2906, 2912, 2921, 2925, 2927, 2935, 2945, 
	2955, 2965, 2969, 2971, 2979, 2989, 2993, 2995, 
	3003, 3011, 3015, 3017, 3024, 3032, 3037, 3042, 
	3052, 3056, 3058, 3087, 3091, 3093, 3101
};

static const short _svg_path_indicies[] = {
	73, 74, 74, 75, 76, 74, 0, 571, 
	571, 572, 572, 573, 571, 574, 0, 631, 
	631, 632, 632, 633, 631, 634, 0, 670, 
	513, 0, 512, 0, 509, 509, 511, 548, 
	514, 514, 509, 510, 512, 0, 490, 490, 
	434, 436, 490, 435, 437, 0, 393, 127, 
	0, 126, 0, 122, 123, 123, 125, 148, 
	128, 129, 130, 131, 132, 133, 134, 135, 
	136, 137, 138, 139, 140, 130, 141, 142, 
	143, 144, 145, 146, 147, 138, 123, 124, 
	126, 0, 73, 611, 611, 613, 614, 78, 
	79, 80, 81, 75, 82, 83, 84, 85, 
	86, 87, 88, 89, 90, 76, 91, 92, 
	93, 94, 86, 611, 612, 615, 0, 671, 
	519, 0, 518, 0, 515, 515, 517, 550, 
	520, 520, 515, 516, 518, 0, 491, 491, 
	438, 440, 491, 439, 441, 0, 394, 155, 
	0, 154, 0, 150, 151, 151, 153, 176, 
	156, 157, 158, 159, 160, 161, 162, 163, 
	164, 165, 166, 167, 168, 158, 169, 170, 
	171, 172, 173, 174, 175, 166, 151, 152, 
	154, 0, 613, 613, 612, 612, 614, 613, 
	615, 0, 515, 515, 517, 518, 520, 520, 
	515, 516, 519, 0, 438, 438, 439, 439, 
	440, 438, 441, 0, 150, 151, 151, 153, 
	154, 156, 157, 158, 159, 160, 161, 162, 
	163, 164, 165, 166, 167, 168, 158, 169, 
	170, 171, 172, 173, 174, 175, 166, 151, 
	152, 155, 0, 47, 47, 48, 47, 49, 
	0, 8, 8, 9, 8, 10, 0, 385, 
	0, 25, 25, 26, 27, 31, 31, 25, 
	498, 0, 12, 12, 13, 14, 12, 15, 
	0, 13, 13, 14, 13, 15, 0, 35, 
	0, 32, 32, 34, 38, 37, 37, 32, 
	33, 35, 0, 20, 20, 16, 18, 20, 
	17, 19, 0, 2, 3, 0, 23, 0, 
	21, 21, 22, 24, 24, 21, 23, 0, 
	11, 11, 5, 6, 7, 11, 0, 5, 
	5, 6, 7, 5, 0, 45, 45, 46, 
	45, 0, 722, 722, 716, 717, 718, 722, 
	0, 716, 716, 717, 718, 716, 0, 567, 
	567, 568, 567, 0, 651, 651, 647, 430, 
	651, 429, 433, 0, 676, 69, 0, 68, 
	0, 351, 351, 352, 375, 70, 70, 351, 
	66, 68, 0, 497, 497, 478, 480, 497, 
	479, 481, 0, 406, 380, 0, 379, 0, 
	350, 377, 377, 378, 382, 355, 356, 381, 
	358, 359, 360, 361, 362, 363, 364, 365, 
	366, 367, 381, 368, 369, 370, 371, 372, 
	373, 374, 365, 377, 379, 0, 73, 95, 
	95, 8, 9, 78, 79, 80, 81, 75, 
	82, 83, 84, 85, 86, 87, 88, 89, 
	90, 76, 91, 92, 93, 94, 86, 95, 
	10, 0, 25, 25, 26, 29, 31, 31, 
	25, 30, 0, 40, 40, 41, 562, 561, 
	561, 40, 33, 30, 0, 483, 483, 425, 
	426, 483, 17, 427, 0, 425, 425, 17, 
	17, 426, 425, 427, 0, 501, 0, 499, 
	499, 500, 38, 503, 503, 499, 33, 501, 
	0, 482, 482, 422, 18, 423, 424, 482, 
	17, 19, 0, 422, 422, 17, 17, 18, 
	423, 424, 422, 19, 0, 63, 63, 64, 
	23, 24, 24, 63, 3, 0, 723, 723, 
	719, 720, 721, 723, 0, 719, 719, 720, 
	721, 719, 0, 569, 569, 570, 569, 0, 
	489, 489, 428, 430, 431, 432, 489, 429, 
	433, 0, 428, 428, 429, 429, 430, 431, 
	432, 428, 433, 0, 71, 71, 72, 68, 
	70, 70, 71, 66, 69, 0, 496, 496, 
	474, 476, 496, 475, 477, 0, 405, 354, 
	0, 353, 0, 350, 351, 351, 352, 375, 
	355, 356, 357, 358, 359, 360, 361, 362, 
	363, 364, 365, 366, 367, 357, 368, 369, 
	370, 371, 372, 373, 374, 365, 351, 66, 
	353, 0, 478, 478, 479, 479, 480, 478, 
	481, 0, 350, 377, 377, 378, 379, 355, 
	356, 381, 358, 359, 360, 361, 362, 363, 
	364, 365, 366, 367, 381, 368, 369, 370, 
	371, 372, 373, 374, 365, 377, 380, 0, 
	50, 50, 51, 51, 52, 50, 53, 0, 
	98, 98, 97, 97, 99, 98, 100, 0, 
	391, 392, 0, 507, 0, 504, 504, 506, 
	546, 508, 508, 504, 505, 507, 0, 484, 
	484, 486, 487, 484, 485, 488, 0, 790, 
	788, 0, 755, 0, 751, 751, 753, 754, 
	756, 756, 751, 752, 755, 0, 787, 787, 
	773, 775, 787, 774, 776, 0, 748, 739, 
	0, 710, 0, 689, 689, 691, 692, 711, 
	711, 689, 690, 710, 0, 736, 736, 724, 
	726, 736, 725, 727, 0, 677, 656, 0, 
	655, 0, 652, 652, 654, 704, 657, 657, 
	652, 653, 655, 0, 648, 648, 635, 637, 
	648, 636, 638, 0, 672, 525, 0, 524, 
	0, 521, 521, 523, 552, 526, 526, 521, 
	522, 524, 0, 492, 492, 458, 460, 492, 
	459, 461, 0, 401, 243, 0, 242, 0, 
	238, 239, 239, 241, 264, 244, 245, 246, 
	247, 248, 249, 250, 251, 252, 253, 254, 
	255, 256, 246, 257, 258, 259, 260, 261, 
	262, 263, 254, 239, 240, 242, 0, 73, 
	96, 96, 98, 99, 78, 79, 80, 81, 
	75, 82, 83, 84, 85, 86, 87, 88, 
	89, 90, 76, 91, 92, 93, 94, 86, 
	96, 97, 100, 0, 504, 504, 506, 507, 
	508, 508, 504, 505, 392, 0, 486, 486, 
	485, 485, 487, 486, 488, 0, 751, 751, 
	753, 755, 756, 756, 751, 752, 788, 0, 
	773, 773, 774, 774, 775, 773, 776, 0, 
	689, 689, 691, 710, 711, 711, 689, 690, 
	739, 0, 724, 724, 725, 725, 726, 724, 
	727, 0, 652, 652, 654, 655, 657, 657, 
	652, 653, 656, 0, 635, 635, 636, 636, 
	637, 635, 638, 0, 521, 521, 523, 524, 
	526, 526, 521, 522, 525, 0, 458, 458, 
	459, 459, 460, 458, 461, 0, 238, 239, 
	239, 241, 242, 244, 245, 246, 247, 248, 
	249, 250, 251, 252, 253, 254, 255, 256, 
	246, 257, 258, 259, 260, 261, 262, 263, 
	254, 239, 240, 243, 0, 416, 416, 265, 
	0, 265, 0, 238, 239, 239, 241, 264, 
	244, 245, 247, 248, 249, 250, 251, 252, 
	253, 254, 255, 256, 257, 258, 259, 260, 
	261, 262, 263, 254, 239, 240, 265, 0, 
	583, 583, 584, 584, 585, 583, 586, 0, 
	446, 446, 447, 447, 448, 446, 449, 0, 
	396, 397, 0, 388, 0, 178, 179, 179, 
	181, 204, 184, 185, 624, 187, 188, 189, 
	190, 191, 192, 193, 194, 195, 196, 624, 
	197, 198, 199, 200, 201, 202, 203, 194, 
	179, 180, 623, 0, 73, 621, 621, 442, 
	444, 78, 79, 80, 81, 75, 82, 83, 
	84, 85, 86, 87, 88, 89, 90, 76, 
	91, 92, 93, 94, 86, 621, 443, 445, 
	0, 395, 183, 0, 182, 0, 178, 179, 
	179, 181, 204, 184, 185, 186, 187, 188, 
	189, 190, 191, 192, 193, 194, 195, 196, 
	186, 197, 198, 199, 200, 201, 202, 203, 
	194, 179, 180, 182, 0, 442, 442, 443, 
	443, 444, 442, 445, 0, 178, 179, 179, 
	181, 182, 184, 185, 186, 187, 188, 189, 
	190, 191, 192, 193, 194, 195, 196, 186, 
	197, 198, 199, 200, 201, 202, 203, 194, 
	179, 180, 183, 0, 412, 412, 205, 0, 
	205, 0, 178, 179, 179, 181, 204, 184, 
	185, 187, 188, 189, 190, 191, 192, 193, 
	194, 195, 196, 197, 198, 199, 200, 201, 
	202, 203, 194, 179, 180, 205, 0, 575, 
	575, 576, 576, 577, 575, 578, 0, 761, 
	761, 762, 762, 763, 761, 764, 0, 781, 
	781, 782, 782, 783, 781, 784, 0, 750, 
	741, 0, 714, 0, 699, 699, 701, 702, 
	715, 715, 699, 700, 714, 0, 738, 738, 
	732, 734, 738, 733, 735, 0, 679, 668, 
	0, 667, 0, 664, 664, 666, 708, 669, 
	669, 664, 665, 667, 0, 650, 650, 643, 
	645, 650, 644, 646, 0, 674, 537, 0, 
	536, 0, 533, 533, 535, 556, 538, 538, 
	533, 534, 536, 0, 494, 494, 466, 468, 
	494, 467, 469, 0, 403, 299, 0, 298, 
	0, 294, 295, 295, 297, 320, 300, 301, 
	302, 303, 304, 305, 306, 307, 308, 309, 
	310, 311, 312, 302, 313, 314, 315, 316, 
	317, 318, 319, 310, 295, 296, 298, 0, 
	73, 786, 786, 781, 783, 78, 79, 80, 
	81, 75, 82, 83, 84, 85, 86, 87, 
	88, 89, 90, 76, 91, 92, 93, 94, 
	86, 786, 782, 784, 0, 699, 699, 701, 
	714, 715, 715, 699, 700, 741, 0, 732, 
	732, 733, 733, 734, 732, 735, 0, 664, 
	664, 666, 667, 669, 669, 664, 665, 668, 
	0, 643, 643, 644, 644, 645, 643, 646, 
	0, 533, 533, 535, 536, 538, 538, 533, 
	534, 537, 0, 466, 466, 467, 467, 468, 
	466, 469, 0, 294, 295, 295, 297, 298, 
	300, 301, 302, 303, 304, 305, 306, 307, 
	308, 309, 310, 311, 312, 302, 313, 314, 
	315, 316, 317, 318, 319, 310, 295, 296, 
	299, 0, 418, 418, 321, 0, 321, 0, 
	294, 295, 295, 297, 320, 300, 301, 303, 
	304, 305, 306, 307, 308, 309, 310, 311, 
	312, 313, 314, 315, 316, 317, 318, 319, 
	310, 295, 296, 321, 0, 757, 757, 758, 
	758, 759, 757, 760, 0, 777, 777, 778, 
	778, 779, 777, 780, 0, 749, 740, 0, 
	712, 0, 694, 694, 696, 697, 713, 713, 
	694, 695, 712, 0, 737, 737, 728, 730, 
	737, 729, 731, 0, 678, 662, 0, 661, 
	0, 658, 658, 660, 706, 663, 663, 658, 
	659, 661, 0, 649, 649, 639, 641, 649, 
	640, 642, 0, 673, 531, 0, 530, 0, 
	527, 527, 529, 554, 532, 532, 527, 528, 
	530, 0, 493, 493, 462, 464, 493, 463, 
	465, 0, 402, 271, 0, 270, 0, 266, 
	267, 267, 269, 292, 272, 273, 274, 275, 
	276, 277, 278, 279, 280, 281, 282, 283, 
	284, 274, 285, 286, 287, 288, 289, 290, 
	291, 282, 267, 268, 270, 0, 73, 785, 
	785, 777, 779, 78, 79, 80, 81, 75, 
	82, 83, 84, 85, 86, 87, 88, 89, 
	90, 76, 91, 92, 93, 94, 86, 785, 
	778, 780, 0, 694, 694, 696, 712, 713, 
	713, 694, 695, 740, 0, 728, 728, 729, 
	729, 730, 728, 731, 0, 658, 658, 660, 
	661, 663, 663, 658, 659, 662, 0, 639, 
	639, 640, 640, 641, 639, 642, 0, 527, 
	527, 529, 530, 532, 532, 527, 528, 531, 
	0, 462, 462, 463, 463, 464, 462, 465, 
	0, 266, 267, 267, 269, 270, 272, 273, 
	274, 275, 276, 277, 278, 279, 280, 281, 
	282, 283, 284, 274, 285, 286, 287, 288, 
	289, 290, 291, 282, 267, 268, 271, 0, 
	417, 417, 293, 0, 293, 0, 266, 267, 
	267, 269, 292, 272, 273, 275, 276, 277, 
	278, 279, 280, 281, 282, 283, 284, 285, 
	286, 287, 288, 289, 290, 291, 282, 267, 
	268, 293, 0, 579, 579, 580, 580, 581, 
	579, 582, 0, 618, 618, 617, 617, 619, 
	618, 620, 0, 675, 543, 0, 542, 0, 
	539, 539, 541, 558, 544, 544, 539, 540, 
	542, 0, 495, 495, 470, 472, 495, 471, 
	473, 0, 404, 327, 0, 326, 0, 322, 
	323, 323, 325, 348, 328, 329, 330, 331, 
	332, 333, 334, 335, 336, 337, 338, 339, 
	340, 330, 341, 342, 343, 344, 345, 346, 
	347, 338, 323, 324, 326, 0, 73, 616, 
	616, 618, 619, 78, 79, 80, 81, 75, 
	82, 83, 84, 85, 86, 87, 88, 89, 
	90, 76, 91, 92, 93, 94, 86, 616, 
	617, 620, 0, 539, 539, 541, 542, 544, 
	544, 539, 540, 543, 0, 470, 470, 471, 
	471, 472, 470, 473, 0, 322, 323, 323, 
	325, 326, 328, 329, 330, 331, 332, 333, 
	334, 335, 336, 337, 338, 339, 340, 330, 
	341, 342, 343, 344, 345, 346, 347, 338, 
	323, 324, 327, 0, 419, 419, 349, 0, 
	349, 0, 322, 323, 323, 325, 348, 328, 
	329, 331, 332, 333, 334, 335, 336, 337, 
	338, 339, 340, 341, 342, 343, 344, 345, 
	346, 347, 338, 323, 324, 349, 0, 587, 
	587, 588, 588, 589, 587, 590, 0, 454, 
	454, 455, 455, 456, 454, 457, 0, 399, 
	400, 0, 390, 0, 208, 209, 209, 211, 
	234, 214, 215, 628, 217, 218, 219, 220, 
	221, 222, 223, 224, 225, 226, 628, 227, 
	228, 229, 230, 231, 232, 233, 224, 209, 
	210, 627, 0, 73, 622, 622, 450, 452, 
	78, 79, 80, 81, 75, 82, 83, 84, 
	85, 86, 87, 88, 89, 90, 76, 91, 
	92, 93, 94, 86, 622, 451, 453, 0, 
	398, 213, 0, 212, 0, 208, 209, 209, 
	211, 234, 214, 215, 216, 217, 218, 219, 
	220, 221, 222, 223, 224, 225, 226, 216, 
	227, 228, 229, 230, 231, 232, 233, 224, 
	209, 210, 212, 0, 450, 450, 451, 451, 
	452, 450, 453, 0, 208, 209, 209, 211, 
	212, 214, 215, 216, 217, 218, 219, 220, 
	221, 222, 223, 224, 225, 226, 216, 227, 
	228, 229, 230, 231, 232, 233, 224, 209, 
	210, 213, 0, 414, 414, 235, 0, 235, 
	0, 208, 209, 209, 211, 234, 214, 215, 
	217, 218, 219, 220, 221, 222, 223, 224, 
	225, 226, 227, 228, 229, 230, 231, 232, 
	233, 224, 209, 210, 235, 0, 101, 102, 
	102, 103, 104, 105, 106, 107, 108, 109, 
	110, 111, 112, 113, 114, 115, 116, 117, 
	118, 119, 120, 121, 112, 102, 0, 73, 
	77, 77, 78, 79, 80, 81, 75, 82, 
	83, 84, 85, 86, 87, 88, 89, 90, 
	76, 91, 92, 93, 94, 86, 77, 0, 
	54, 54, 55, 54, 56, 0, 57, 57, 
	58, 58, 59, 57, 60, 0, 603, 603, 
	604, 604, 605, 603, 606, 0, 178, 179, 
	179, 181, 625, 184, 185, 624, 187, 188, 
	189, 190, 191, 192, 193, 194, 195, 196, 
	624, 197, 198, 199, 200, 201, 202, 203, 
	194, 179, 180, 626, 0, 413, 413, 387, 
	0, 387, 0, 178, 179, 179, 181, 204, 
	184, 185, 187, 188, 189, 190, 191, 192, 
	193, 194, 195, 196, 197, 198, 199, 200, 
	201, 202, 203, 194, 179, 180, 206, 0, 
	178, 179, 179, 181, 207, 184, 185, 186, 
	187, 188, 189, 190, 191, 192, 193, 194, 
	195, 196, 186, 197, 198, 199, 200, 201, 
	202, 203, 194, 179, 180, 206, 0, 595, 
	595, 596, 596, 597, 595, 598, 0, 591, 
	591, 592, 592, 593, 591, 594, 0, 509, 
	509, 511, 512, 514, 514, 509, 510, 513, 
	0, 434, 434, 435, 435, 436, 434, 437, 
	0, 122, 123, 123, 125, 126, 128, 129, 
	130, 131, 132, 133, 134, 135, 136, 137, 
	138, 139, 140, 130, 141, 142, 143, 144, 
	145, 146, 147, 138, 123, 124, 127, 0, 
	410, 410, 149, 0, 149, 0, 122, 123, 
	123, 125, 148, 128, 129, 131, 132, 133, 
	134, 135, 136, 137, 138, 139, 140, 141, 
	142, 143, 144, 145, 146, 147, 138, 123, 
	124, 149, 0, 769, 769, 770, 770, 771, 
	769, 772, 0, 765, 765, 766, 766, 767, 
	765, 768, 0, 599, 599, 600, 600, 601, 
	599, 602, 0, 607, 607, 608, 608, 609, 
	607, 610, 0, 208, 209, 209, 211, 629, 
	214, 215, 628, 217, 218, 219, 220, 221, 
	222, 223, 224, 225, 226, 628, 227, 228, 
	229, 230, 231, 232, 233, 224, 209, 210, 
	630, 0, 415, 415, 389, 0, 389, 0, 
	208, 209, 209, 211, 234, 214, 215, 217, 
	218, 219, 220, 221, 222, 223, 224, 225, 
	226, 227, 228, 229, 230, 231, 232, 233, 
	224, 209, 210, 236, 0, 208, 209, 209, 
	211, 237, 214, 215, 216, 217, 218, 219, 
	220, 221, 222, 223, 224, 225, 226, 216, 
	227, 228, 229, 230, 231, 232, 233, 224, 
	209, 210, 236, 0, 682, 682, 549, 0, 
	549, 0, 509, 509, 511, 548, 509, 510, 
	549, 0, 208, 209, 209, 211, 237, 214, 
	215, 628, 217, 218, 219, 220, 221, 222, 
	223, 224, 225, 226, 628, 227, 228, 229, 
	230, 231, 232, 233, 224, 209, 210, 627, 
	0, 687, 687, 559, 0, 559, 0, 539, 
	539, 541, 558, 539, 540, 559, 0, 685, 
	685, 555, 0, 555, 0, 527, 527, 529, 
	554, 527, 528, 555, 0, 746, 746, 707, 
	0, 707, 0, 658, 658, 660, 706, 658, 
	659, 707, 0, 743, 743, 698, 0, 698, 
	0, 694, 694, 696, 697, 694, 695, 698, 
	0, 686, 686, 557, 0, 557, 0, 533, 
	533, 535, 556, 533, 534, 557, 0, 747, 
	747, 709, 0, 709, 0, 664, 664, 666, 
	708, 664, 665, 709, 0, 744, 744, 703, 
	0, 703, 0, 699, 699, 701, 702, 699, 
	700, 703, 0, 178, 179, 179, 181, 207, 
	184, 185, 624, 187, 188, 189, 190, 191, 
	192, 193, 194, 195, 196, 624, 197, 198, 
	199, 200, 201, 202, 203, 194, 179, 180, 
	623, 0, 684, 684, 553, 0, 553, 0, 
	521, 521, 523, 552, 521, 522, 553, 0, 
	745, 745, 705, 0, 705, 0, 652, 652, 
	654, 704, 652, 653, 705, 0, 742, 742, 
	693, 0, 693, 0, 689, 689, 691, 692, 
	689, 690, 693, 0, 791, 791, 789, 0, 
	789, 0, 751, 751, 753, 754, 751, 752, 
	789, 0, 681, 681, 547, 0, 547, 0, 
	504, 504, 506, 546, 504, 505, 547, 0, 
	421, 421, 383, 0, 383, 0, 350, 377, 
	377, 378, 382, 355, 356, 358, 359, 360, 
	361, 362, 363, 364, 365, 366, 367, 368, 
	369, 370, 371, 372, 373, 374, 365, 377, 
	383, 0, 420, 420, 376, 0, 376, 0, 
	350, 351, 351, 352, 375, 355, 356, 358, 
	359, 360, 361, 362, 363, 364, 365, 366, 
	367, 368, 369, 370, 371, 372, 373, 374, 
	365, 351, 66, 376, 0, 350, 351, 351, 
	352, 353, 355, 356, 357, 358, 359, 360, 
	361, 362, 363, 364, 365, 366, 367, 357, 
	368, 369, 370, 371, 372, 373, 374, 365, 
	351, 66, 354, 0, 474, 474, 475, 475, 
	476, 474, 477, 0, 351, 351, 352, 68, 
	70, 70, 351, 66, 69, 0, 688, 688, 
	560, 0, 560, 0, 351, 351, 352, 375, 
	351, 66, 560, 0, 65, 65, 67, 68, 
	70, 70, 65, 66, 69, 0, 565, 565, 
	566, 565, 0, 21, 21, 22, 23, 24, 
	24, 21, 3, 0, 4, 4, 1, 0, 
	1, 0, 21, 21, 22, 21, 1, 0, 
	61, 61, 62, 23, 24, 24, 61, 3, 
	0, 680, 680, 545, 0, 545, 0, 499, 
	499, 500, 38, 499, 33, 545, 0, 499, 
	499, 500, 501, 503, 503, 499, 33, 502, 
	0, 40, 40, 41, 27, 561, 561, 40, 
	33, 498, 0, 40, 40, 41, 42, 561, 
	561, 40, 33, 498, 0, 409, 409, 386, 
	0, 386, 0, 40, 40, 41, 27, 40, 
	33, 28, 0, 40, 40, 41, 42, 37, 
	37, 40, 33, 28, 0, 408, 408, 39, 
	0, 39, 0, 32, 32, 34, 38, 32, 
	33, 39, 0, 16, 16, 17, 17, 18, 
	16, 19, 0, 407, 407, 384, 0, 384, 
	0, 25, 25, 26, 27, 25, 28, 0, 
	647, 647, 429, 429, 430, 647, 433, 0, 
	563, 563, 564, 563, 0, 43, 43, 44, 
	43, 0, 32, 32, 34, 35, 37, 37, 
	32, 33, 36, 0, 411, 411, 177, 0, 
	177, 0, 150, 151, 151, 153, 176, 156, 
	157, 159, 160, 161, 162, 163, 164, 165, 
	166, 167, 168, 169, 170, 171, 172, 173, 
	174, 175, 166, 151, 152, 177, 0, 683, 
	683, 551, 0, 551, 0, 515, 515, 517, 
	550, 515, 516, 551, 0, 0, 0
};

static const short _svg_path_trans_targs_wi[] = {
	1, 297, 33, 294, 296, 36, 37, 318, 
	24, 25, 50, 35, 27, 28, 29, 319, 
	312, 32, 33, 294, 31, 35, 36, 34, 
	295, 27, 28, 29, 308, 26, 51, 313, 
	31, 32, 312, 30, 319, 309, 33, 311, 
	52, 53, 30, 38, 39, 38, 39, 24, 
	25, 50, 72, 73, 74, 97, 24, 25, 
	50, 72, 73, 74, 97, 59, 60, 59, 
	60, 65, 46, 287, 44, 288, 289, 65, 
	287, 326, 0, 2, 223, 213, 23, 71, 
	111, 125, 126, 154, 182, 198, 212, 214, 
	215, 216, 222, 230, 231, 232, 233, 49, 
	96, 73, 72, 74, 97, 326, 213, 23, 
	71, 111, 125, 2, 126, 154, 182, 198, 
	212, 214, 215, 216, 222, 223, 230, 231, 
	232, 233, 326, 11, 12, 19, 10, 226, 
	23, 71, 227, 111, 125, 2, 126, 154, 
	182, 198, 212, 214, 215, 216, 222, 223, 
	230, 231, 232, 233, 13, 229, 326, 11, 
	12, 19, 18, 22, 23, 71, 320, 111, 
	125, 2, 126, 154, 182, 198, 212, 214, 
	215, 216, 222, 223, 230, 231, 232, 233, 
	13, 322, 326, 116, 117, 120, 119, 121, 
	23, 71, 122, 111, 125, 2, 126, 154, 
	182, 198, 212, 214, 215, 216, 222, 223, 
	230, 231, 232, 233, 118, 124, 221, 119, 
	326, 203, 204, 207, 206, 208, 23, 71, 
	209, 111, 125, 2, 126, 154, 182, 198, 
	212, 214, 215, 216, 222, 223, 230, 231, 
	232, 233, 205, 211, 238, 206, 326, 96, 
	73, 72, 95, 107, 23, 71, 108, 111, 
	125, 2, 126, 154, 182, 198, 212, 214, 
	215, 216, 222, 223, 230, 231, 232, 233, 
	74, 110, 326, 171, 156, 155, 170, 178, 
	23, 71, 179, 111, 125, 2, 126, 154, 
	182, 198, 212, 214, 215, 216, 222, 223, 
	230, 231, 232, 233, 157, 181, 326, 143, 
	128, 127, 142, 150, 23, 71, 151, 111, 
	125, 2, 126, 154, 182, 198, 212, 214, 
	215, 216, 222, 223, 230, 231, 232, 233, 
	129, 153, 326, 191, 184, 183, 190, 194, 
	23, 71, 195, 111, 125, 2, 126, 154, 
	182, 198, 212, 214, 215, 216, 222, 223, 
	230, 231, 232, 233, 185, 197, 326, 45, 
	69, 68, 286, 23, 71, 283, 111, 125, 
	2, 126, 154, 182, 198, 212, 214, 215, 
	216, 222, 223, 230, 231, 232, 233, 47, 
	285, 49, 24, 48, 70, 280, 25, 282, 
	315, 26, 307, 220, 115, 237, 202, 74, 
	97, 9, 17, 118, 114, 217, 205, 201, 
	234, 94, 169, 141, 189, 67, 47, 314, 
	310, 306, 228, 321, 123, 219, 210, 236, 
	109, 180, 152, 196, 284, 281, 57, 58, 
	298, 53, 54, 302, 63, 42, 43, 64, 
	292, 288, 225, 8, 9, 226, 21, 16, 
	17, 22, 120, 117, 118, 121, 112, 113, 
	114, 217, 207, 204, 205, 208, 199, 200, 
	201, 234, 106, 93, 94, 107, 177, 168, 
	169, 178, 149, 140, 141, 150, 193, 188, 
	189, 194, 287, 66, 67, 286, 69, 46, 
	47, 70, 56, 52, 76, 77, 98, 78, 
	99, 62, 7, 15, 92, 167, 139, 187, 
	65, 45, 304, 56, 57, 55, 302, 299, 
	76, 77, 98, 75, 277, 7, 8, 225, 
	6, 224, 239, 15, 16, 21, 14, 20, 
	323, 92, 93, 106, 91, 105, 265, 167, 
	168, 177, 166, 176, 246, 139, 140, 149, 
	138, 148, 255, 187, 188, 193, 186, 192, 
	243, 301, 78, 279, 9, 241, 17, 325, 
	94, 267, 169, 248, 141, 257, 189, 245, 
	291, 305, 303, 41, 316, 62, 63, 41, 
	316, 62, 63, 3, 4, 5, 224, 19, 
	12, 13, 20, 183, 184, 185, 192, 112, 
	113, 114, 217, 199, 200, 201, 234, 3, 
	4, 5, 224, 19, 12, 13, 20, 183, 
	184, 185, 192, 112, 113, 114, 217, 199, 
	200, 201, 234, 11, 12, 19, 13, 20, 
	191, 184, 183, 185, 192, 116, 203, 264, 
	218, 115, 217, 242, 235, 202, 234, 3, 
	4, 5, 224, 104, 89, 90, 105, 175, 
	164, 165, 176, 147, 136, 137, 148, 316, 
	88, 163, 135, 41, 88, 89, 104, 87, 
	103, 268, 163, 164, 175, 162, 174, 249, 
	135, 136, 147, 134, 146, 258, 5, 13, 
	90, 165, 137, 185, 43, 86, 161, 133, 
	300, 278, 240, 324, 266, 247, 256, 244, 
	290, 84, 85, 102, 86, 273, 159, 160, 
	173, 161, 254, 131, 132, 145, 133, 263, 
	90, 270, 165, 251, 137, 260, 83, 271, 
	158, 252, 130, 261, 39, 40, 317, 60, 
	61, 293, 38, 59, 102, 85, 86, 103, 
	173, 160, 161, 174, 145, 132, 133, 146, 
	84, 159, 131, 101, 172, 144, 272, 253, 
	262, 269, 250, 259, 82, 157, 129, 80, 
	81, 100, 82, 79, 274, 155, 156, 157, 
	172, 127, 128, 129, 144, 155, 156, 157, 
	172, 127, 128, 129, 144, 100, 81, 82, 
	101, 155, 156, 157, 172, 127, 128, 129, 
	144, 171, 143, 80, 99, 276, 78, 275
};

static const unsigned char _svg_path_trans_actions_wi[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 1, 0, 0, 0, 1, 1, 
	0, 1, 1, 1, 0, 3, 3, 0, 
	0, 3, 3, 17, 17, 17, 17, 0, 
	3, 17, 3, 0, 0, 0, 17, 0, 
	3, 3, 17, 5, 5, 7, 7, 9, 
	59, 59, 9, 59, 59, 59, 11, 62, 
	62, 11, 62, 62, 62, 20, 20, 23, 
	23, 53, 17, 53, 0, 0, 0, 56, 
	56, 15, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 0, 1, 1, 65, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 13, 13, 13, 13, 13, 13, 
	13, 13, 72, 26, 68, 26, 0, 0, 
	26, 26, 0, 26, 26, 26, 26, 26, 
	26, 26, 26, 26, 26, 26, 26, 26, 
	26, 26, 26, 26, 68, 0, 80, 29, 
	76, 29, 0, 0, 29, 29, 0, 29, 
	29, 29, 29, 29, 29, 29, 29, 29, 
	29, 29, 29, 29, 29, 29, 29, 29, 
	76, 0, 88, 32, 84, 32, 0, 0, 
	32, 32, 0, 32, 32, 32, 32, 32, 
	32, 32, 32, 32, 32, 32, 32, 32, 
	32, 32, 32, 32, 84, 0, 84, 84, 
	96, 35, 92, 35, 0, 0, 35, 35, 
	0, 35, 35, 35, 35, 35, 35, 35, 
	35, 35, 35, 35, 35, 35, 35, 35, 
	35, 35, 92, 0, 92, 92, 104, 38, 
	100, 38, 0, 0, 38, 38, 0, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	38, 38, 38, 38, 38, 38, 38, 38, 
	100, 0, 112, 41, 108, 41, 0, 0, 
	41, 41, 0, 41, 41, 41, 41, 41, 
	41, 41, 41, 41, 41, 41, 41, 41, 
	41, 41, 41, 41, 108, 0, 120, 44, 
	116, 44, 0, 0, 44, 44, 0, 44, 
	44, 44, 44, 44, 44, 44, 44, 44, 
	44, 44, 44, 44, 44, 44, 44, 44, 
	116, 0, 128, 47, 124, 47, 0, 0, 
	47, 47, 0, 47, 47, 47, 47, 47, 
	47, 47, 47, 47, 47, 47, 47, 47, 
	47, 47, 47, 47, 124, 0, 136, 3, 
	3, 0, 0, 50, 50, 0, 50, 50, 
	50, 50, 50, 50, 50, 50, 50, 50, 
	50, 50, 50, 50, 50, 50, 50, 17, 
	0, 50, 50, 0, 0, 0, 132, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	1, 0, 1, 1, 0, 1, 1, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 0, 0, 1, 0, 1, 
	1, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 17, 3, 3, 0, 0, 0, 
	3, 17, 3, 0, 0, 3, 17, 3, 
	0, 0, 0, 3, 17, 3, 0, 0, 
	0, 3, 17, 3, 0, 0, 0, 3, 
	17, 3, 0, 0, 0, 3, 17, 3, 
	0, 0, 0, 3, 17, 3, 0, 0, 
	0, 0, 17, 0, 17, 0, 17, 0, 
	17, 0, 17, 0, 17, 0, 17, 0, 
	0, 0, 17, 5, 5, 5, 5, 7, 
	7, 7, 7, 9, 59, 59, 59, 9, 
	59, 59, 59, 9, 59, 59, 59, 9, 
	59, 59, 59, 9, 59, 59, 59, 11, 
	62, 62, 62, 11, 62, 62, 62, 11, 
	62, 62, 62, 11, 62, 62, 62, 11, 
	62, 62, 62, 0, 1, 0, 1, 1, 
	0, 1, 0, 1, 1, 0, 0, 84, 
	0, 84, 84, 92, 0, 92, 92, 0, 
	1, 1, 1, 0, 1, 1, 1, 0, 
	1, 1, 1, 0, 1, 1, 1, 0, 
	0, 0, 0, 0, 3, 17, 3, 0, 
	0, 0, 3, 17, 3, 0, 0, 0, 
	3, 17, 3, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 3, 17, 3, 17, 0, 3, 17, 
	3, 17, 0, 3, 17, 3, 17, 0, 
	17, 0, 17, 0, 17, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 1, 1, 
	0, 1, 1, 1, 0, 1, 1, 1, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 3, 
	17, 3, 17, 0, 0, 9, 59, 59, 
	59, 9, 59, 59, 59, 11, 62, 62, 
	62, 11, 62, 62, 62, 0, 1, 1, 
	1, 0, 1, 1, 1, 0, 1, 1, 
	1, 0, 0, 0, 0, 0, 0, 0
};

static const int svg_path_start = 0;

static const int svg_path_first_final = 326;

#line 133 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"


void Parser::parse(char const *str)
throw(SVGPathParseError)
{
    char const *p = str;
    char const *start = NULL;
    int cs;

    _reset();

    
#line 1373 "/home/mental/trees/lib2geom/src/svg-path-parser.cpp"
	{
	cs = svg_path_start;
	}
	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

_resume:
	if ( cs == 1 )
		goto _out;
	_keys = _svg_path_trans_keys + _svg_path_key_offsets[cs];
	_trans = _svg_path_index_offsets[cs];

	_klen = _svg_path_single_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*p) < *_mid )
				_upper = _mid - 1;
			else if ( (*p) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _svg_path_range_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*p) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*p) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _svg_path_indicies[_trans];
	cs = _svg_path_trans_targs_wi[_trans];

	if ( _svg_path_trans_actions_wi[_trans] == 0 )
		goto _again;

	_acts = _svg_path_actions + _svg_path_trans_actions_wi[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 145 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            start = p;
        }
	break;
	case 1:
#line 149 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            char const *end=p;
            std::string buf(start, end);
            _push(g_ascii_strtod(buf.c_str(), NULL));
            start = NULL;
        }
	break;
	case 2:
#line 156 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _push(1.0);
        }
	break;
	case 3:
#line 160 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _push(0.0);
        }
	break;
	case 4:
#line 164 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _absolute = true;
        }
	break;
	case 5:
#line 168 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _absolute = false;
        }
	break;
	case 6:
#line 172 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _moveTo(_pop_point());
        }
	break;
	case 7:
#line 176 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _lineTo(_pop_point());
        }
	break;
	case 8:
#line 180 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _lineTo(Point(_pop_coord(X), _current[Y]));
        }
	break;
	case 9:
#line 184 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _lineTo(Point(_current[X], _pop_coord(Y)));
        }
	break;
	case 10:
#line 188 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            Point p = _pop_point();
            Point c1 = _pop_point();
            Point c0 = _pop_point();
            _curveTo(c0, c1, p);
        }
	break;
	case 11:
#line 195 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            Point p = _pop_point();
            Point c1 = _pop_point();
            _curveTo(_cubic_tangent, c1, p);
        }
	break;
	case 12:
#line 201 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            Point p = _pop_point();
            Point c = _pop_point();
            _quadTo(c, p);
        }
	break;
	case 13:
#line 207 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            Point p = _pop_point();
            _quadTo(_quad_tangent, p);
        }
	break;
	case 14:
#line 212 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            Point point = _pop_point();
            bool sweep = _pop_flag();
            bool large_arc = _pop_flag();
            double angle = _pop();
            double ry = _pop();
            double rx = _pop();

            _arcTo(rx, ry, angle, large_arc, sweep, point);
        }
	break;
	case 15:
#line 223 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{
            _closePath();
        }
	break;
	case 16:
#line 360 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"
	{goto _out;}
	break;
#line 1566 "/home/mental/trees/lib2geom/src/svg-path-parser.cpp"
		}
	}

_again:
	p += 1;
	goto _resume;
	_out: {}
	}
#line 370 "/home/mental/trees/lib2geom/src/svg-path-parser.rl"


    if ( cs < svg_path_first_final ) {
        throw SVGPathParseError();
    }
}

}

void parse_svg_path(char const *str, SVGPathSink &sink)
throw(SVGPathParseError)
{
    Parser parser(sink);
    parser.parse(str);
    sink.finish();
}

}

/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:encoding=utf-8:textwidth=99 :
