/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
  A menu that should be close to the user, it's the user's status,
  their friends.  All about them.  It's a user-focused-menu.

  Copyright 2009 Canonical Ltd.

  Authors:
  Ted Gould <ted@canonical.com>
  Cody Russell <cody.russell@canonical.com>

  This program is free software: you can redistribute it and/or modify it
  under the terms of the GNU General Public License version 3, as published
  by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranties of
  MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libdbusmenu-glib/menuitem.h>
#include <libdbusmenu-gtk/menu.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <libindicator/indicator.h>
#include <libindicator/indicator-object.h>
#include <libindicator/indicator-service-manager.h>
#include <libindicator/indicator-image-helper.h>
#include <libido/idoentrymenuitem.h>

#include "about-me-menu-item.h"

#include "dbus-shared-names.h"
#include "me-service-client.h"

#define INDICATOR_ME_TYPE            (indicator_me_get_type ())
#define INDICATOR_ME(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), INDICATOR_ME_TYPE, IndicatorMe))
#define INDICATOR_ME_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), INDICATOR_ME_TYPE, IndicatorMeClass))
#define IS_INDICATOR_ME(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), INDICATOR_ME_TYPE))
#define IS_INDICATOR_ME_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), INDICATOR_ME_TYPE))
#define INDICATOR_ME_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), INDICATOR_ME_TYPE, IndicatorMeClass))

#define DEFAULT_ICON   "user-offline"

typedef struct _IndicatorMe      IndicatorMe;
typedef struct _IndicatorMeClass IndicatorMeClass;

struct _IndicatorMeClass {
	IndicatorObjectClass parent_class;
};

struct _IndicatorMe {
	IndicatorObject parent;
	IndicatorServiceManager * service;
};

GType indicator_me_get_type (void);

/* Indicator stuff */
INDICATOR_SET_VERSION
INDICATOR_SET_TYPE(INDICATOR_ME_TYPE)

/* Globals */
static GtkImage * status_image = NULL;
static GtkLabel *label = NULL;
static DBusGProxy * status_proxy = NULL;

/* Prototypes */
static GtkLabel * get_label (IndicatorObject * io);
static GtkImage * get_icon (IndicatorObject * io);
static GtkMenu * get_menu (IndicatorObject * io);

static void indicator_me_class_init (IndicatorMeClass *klass);
static void indicator_me_init       (IndicatorMe *self);
static void indicator_me_dispose    (GObject *object);
static void indicator_me_finalize   (GObject *object);
static void connection_changed      (IndicatorServiceManager * sm, gboolean connected, gpointer userdata);
static void status_icon_cb (DBusGProxy * proxy, char * icons, GError *error, gpointer userdata);

G_DEFINE_TYPE (IndicatorMe, indicator_me, INDICATOR_OBJECT_TYPE);

static void
indicator_me_class_init (IndicatorMeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->dispose = indicator_me_dispose;
	object_class->finalize = indicator_me_finalize;

	IndicatorObjectClass * io_class = INDICATOR_OBJECT_CLASS(klass);
	io_class->get_label = get_label;
	io_class->get_image = get_icon;
	io_class->get_menu = get_menu;

	return;
}

static void
indicator_me_init (IndicatorMe *self)
{

	/* Init variables */
	self->service = NULL;

	/* Do stuff with them */
	self->service = indicator_service_manager_new_version(INDICATOR_ME_DBUS_NAME, INDICATOR_ME_DBUS_VERSION);
	g_signal_connect(G_OBJECT(self->service), INDICATOR_SERVICE_MANAGER_SIGNAL_CONNECTION_CHANGE, G_CALLBACK(connection_changed), self);

	return;
}

static void
indicator_me_dispose (GObject *object)
{

	G_OBJECT_CLASS (indicator_me_parent_class)->dispose (object);
	return;
}

static void
indicator_me_finalize (GObject *object)
{

	G_OBJECT_CLASS (indicator_me_parent_class)->finalize (object);
	return;
}

static void
username_cb (DBusGProxy * proxy, char * username, GError *error, gpointer userdata)
{
  if (label == NULL) {
    label = GTK_LABEL(gtk_label_new(NULL));
    if (label == NULL) return;
  }

  if (username != NULL && username[0] != '\0') {
    g_debug ("Updating username label");
    gtk_label_set_text (label, username);
    gtk_widget_show(GTK_WIDGET(label));
  } else {
    gtk_widget_hide(GTK_WIDGET(label));
  }

}

static GtkLabel *
get_label (IndicatorObject * io)
{
  if (label == NULL) {
		/* Create the label if it doesn't exist already */
    username_cb (NULL, NULL, NULL, NULL);
  }

	return label;
}

static GtkImage *
get_icon (IndicatorObject * io)
{
	if (status_image == NULL) {
		/* Will create the status icon if it doesn't exist already */
		status_icon_cb(NULL, DEFAULT_ICON, NULL, NULL);
	}
	return status_image;
}

static void
status_icon_cb (DBusGProxy * proxy, char * icons, GError *error, gpointer userdata)
{
	g_return_if_fail(icons != NULL);
	g_return_if_fail(icons[0] != '\0');

	if (status_image == NULL) {
    status_image = indicator_image_helper (DEFAULT_ICON "-panel");
		gtk_widget_show(GTK_WIDGET(status_image));
	}

  gchar *panel_icon = g_strconcat (icons, "-panel", NULL);
  indicator_image_helper_update (status_image, panel_icon);
  g_free (panel_icon);

	return;
}

static void
status_icon_changed (DBusGProxy * proxy, gchar * icon, gpointer userdata)
{
	g_debug("Changing status icon: '%s'", icon);

	return status_icon_cb(proxy, icon, NULL, NULL);
}

static void
connection_changed (IndicatorServiceManager * sm, gboolean connected, gpointer userdata)
{
	if (connected) {
		if (status_proxy == NULL) {
			GError * error = NULL;

			DBusGConnection * sbus = dbus_g_bus_get(DBUS_BUS_SESSION, NULL);

			status_proxy = dbus_g_proxy_new_for_name_owner(sbus,
                                                     INDICATOR_ME_DBUS_NAME,
                                                     INDICATOR_ME_SERVICE_DBUS_OBJECT,
                                                     INDICATOR_ME_SERVICE_DBUS_INTERFACE,
                                                     &error);

			if (error != NULL) {
				g_warning("Unable to get status proxy: %s", error->message);
				g_error_free(error);
			}

      if (status_proxy == NULL) return;

			dbus_g_proxy_add_signal(status_proxy, "StatusIconsChanged", G_TYPE_STRING, G_TYPE_INVALID);
			dbus_g_proxy_connect_signal(status_proxy, "StatusIconsChanged", G_CALLBACK(status_icon_changed), NULL, NULL);

			dbus_g_proxy_add_signal(status_proxy, "UserChanged", G_TYPE_STRING, G_TYPE_INVALID);
			dbus_g_proxy_connect_signal(status_proxy, "UserChanged", G_CALLBACK(username_cb), NULL, NULL);
		}

		org_ayatana_indicator_me_service_status_icons_async(status_proxy, status_icon_cb, NULL);

    /* query the service for the username to display */
		org_ayatana_indicator_me_service_pretty_user_name_async(status_proxy, username_cb, NULL);

	} else {
		/* If we're disconnecting, go back to offline */
		status_icon_cb(NULL, DEFAULT_ICON, NULL, NULL);
	}

	return;
}

static void
entry_prop_change_cb (DbusmenuMenuitem *mi, gchar *prop, GValue *value, GtkEntry *entry)
{
	if (g_strcmp0 (prop, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT) == 0) {
		gtk_entry_set_text (entry, g_value_get_string (value));
	}
}

static void
entry_activate_cb (GtkEntry *entry, DbusmenuMenuitem *mi)
{
  g_return_if_fail (GTK_IS_ENTRY (entry));
  g_return_if_fail (DBUSMENU_IS_MENUITEM (mi));

	GValue value = { 0 };
	g_value_init (&value, G_TYPE_STRING);
	g_value_set_static_string (&value, gtk_entry_get_text (entry));

	g_debug ("user typed: %s", g_value_get_string (&value));

	dbusmenu_menuitem_handle_event (mi, "send", &value, gtk_get_current_event_time());
}

static gboolean
menu_visibility_changed (GtkWidget          *widget,
                         IdoEntryMenuItem   *menuitem)
{
  if (GTK_IS_WIDGET (widget)
      && IDO_IS_ENTRY_MENU_ITEM (menuitem))
    gtk_menu_shell_select_item (GTK_MENU_SHELL (widget), GTK_WIDGET (menuitem));

  return FALSE;
}

static void
entry_parent_changed (GtkWidget *widget,
                      gpointer   user_data)
{
  GtkWidget *parent = gtk_widget_get_parent (widget);

  if (parent && GTK_IS_MENU_SHELL (parent))
    {
      g_signal_connect (parent,
                        "map", G_CALLBACK (menu_visibility_changed),
                        widget);
    }
}

static gboolean
new_entry_item (DbusmenuMenuitem * newitem,
                DbusmenuMenuitem * parent,
                DbusmenuClient   * client)
{
	g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
	g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
	/* Note: not checking parent, it's reasonable for it to be NULL */

	IdoEntryMenuItem *ido = IDO_ENTRY_MENU_ITEM (ido_entry_menu_item_new ());
	GtkEntry *entry = GTK_ENTRY(ido_entry_menu_item_get_entry (ido));
	if (dbusmenu_menuitem_property_get (newitem, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT) != NULL)
		gtk_entry_set_text(entry, dbusmenu_menuitem_property_get(newitem, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT));
	gtk_entry_set_width_chars (entry, 23); /* set some nice aspect ratio for the menu */
  gtk_entry_set_max_length (entry, 140); /* enforce current gwibber limit */

  g_signal_connect (ido,
                    "notify::parent", G_CALLBACK (entry_parent_changed),
                    NULL);

	dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem, GTK_MENU_ITEM(ido), parent);
	/* disconnect the activate signal that newitem_base connected with the wrong
	   widget, ie menuitem, and re-connect it with the /entry/ instead */
	gulong signal_id = g_signal_handler_find (GTK_MENU_ITEM (ido), G_SIGNAL_MATCH_DATA, 0, 0, 0, 0, parent);
  if (signal_id > 0)
    g_signal_handler_disconnect(GTK_MENU_ITEM (ido), signal_id);

	g_signal_connect (DBUSMENU_MENUITEM (newitem), DBUSMENU_MENUITEM_SIGNAL_PROPERTY_CHANGED, G_CALLBACK (entry_prop_change_cb), entry);
	g_signal_connect (GTK_ENTRY (entry), "activate", G_CALLBACK (entry_activate_cb), newitem);

	return TRUE;
}

/* Whenever we have a property change on a DbusmenuMenuitem
   we need to be responsive to that. */
static void
about_me_prop_change_cb (DbusmenuMenuitem * mi, gchar * prop, GValue * value, AboutMeMenuItem *item)
{
  g_return_if_fail (ABOUT_IS_ME_MENU_ITEM (item));

	if (!g_strcmp0(prop, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON)) {
    /* reload the avatar icon */
    about_me_menu_item_load_avatar (item, g_value_get_string(value));
  } else if (!g_strcmp0(prop, DBUSMENU_MENUITEM_PROP_VISIBLE)) {
    /* normal, ignore */
	} else {
		g_warning("Indicator Item property '%s' unknown", prop);
	}

	return;
}

static gboolean
new_about_me_item (DbusmenuMenuitem * newitem,
                   DbusmenuMenuitem * parent,
                   DbusmenuClient * client)
{
	g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
	g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
	/* Note: not checking parent, it's reasonable for it to be NULL */

	const gchar *name = dbusmenu_menuitem_property_get (newitem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_NAME);
	AboutMeMenuItem *about = ABOUT_ME_MENU_ITEM (about_me_menu_item_new (name));
  if (about != NULL) {
    dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem, GTK_MENU_ITEM(about), parent);
    const gchar *avatar = dbusmenu_menuitem_property_get (newitem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON);
    about_me_menu_item_load_avatar (about, avatar);
    g_signal_connect(G_OBJECT(newitem), DBUSMENU_MENUITEM_SIGNAL_PROPERTY_CHANGED, G_CALLBACK(about_me_prop_change_cb), about);
  }

	return TRUE;
}

/* Builds the dbusmenu for the service. */
static GtkMenu *
get_menu (IndicatorObject * io)
{
	DbusmenuGtkMenu *menu = dbusmenu_gtkmenu_new(INDICATOR_ME_DBUS_NAME, INDICATOR_ME_DBUS_OBJECT);
	DbusmenuGtkClient * client = dbusmenu_gtkmenu_get_client(menu);

	dbusmenu_client_add_type_handler(DBUSMENU_CLIENT(client), DBUSMENU_ENTRY_MENUITEM_TYPE, new_entry_item);
	dbusmenu_client_add_type_handler(DBUSMENU_CLIENT(client), DBUSMENU_ABOUT_ME_MENUITEM_TYPE, new_about_me_item);

	return GTK_MENU (menu);
}
