#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from urllib import quote
from twisted.internet import reactor

from imagestore.lib.tests.mocker import ARGS, KWARGS
from imagestore.lib.tests import TestCase
from imagestore.lib.fetch import fetch, PyCurlError, HTTPCodeError

from imagestore.tests.helpers import ServiceTestCase
from imagestore.proxyservice import (
    ProxyService, ProxyServiceError, GetDashboardTask, GetUpgradesTask,
    SearchTask, UpstreamError)
from imagestore.model import Image, ImageStoreResponse


class ProxyServiceTest(ServiceTestCase):

    def setUp(self):
        self.service = ProxyService(reactor, "http://is.c.c/api")

    def testGetDashboard(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)

        response = ImageStoreResponse(
            {"sections": [
                {"title": "New",
                 "summary": "Appliances recently made available:",
                 "image-uris": [
                     image1["uri"],
                     image2["uri"],
                     ],
                 },
                ],
             "images": [image1, image2],
             })

        fetchMock = self.mocker.replace(fetch)
        fetchMock(self.service.endpoint + "/dashboard",
                  post=True, data="upgrade-uri=%s&upgrade-uri=%s" % (
                      quote(image1["uri"], safe=""),
                      quote(image2["uri"], safe="")))
        self.mocker.result(str(response))

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        task = GetDashboardTask([image1["uri"], image2["uri"]])
        deferred = self.service.addTask(task)

        def callback(result):
            self.assertEquals(result, response)

        deferred.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred)

    def test_wb__fetchFailsWithPyCurlError(self):
        fetchMock = self.mocker.replace(fetch)
        fetchMock("http://non-existing/api/nonesuch",
                  post=True, data="yadayada")
        self.mocker.throw(PyCurlError(123, "Frobnicator Segfault"))

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        try:
            self.service._fetch("http://non-existing/api/nonesuch", data="yadayada")
            self.fail("Should have raised an exception")
        except ProxyServiceError, e:
            self.assertEquals(e.message, "Error 123: Frobnicator Segfault")

    def test_wb__fetchFailsWithHTTPCodeError(self):
        fetchMock = self.mocker.replace(fetch)
        fetchMock("http://non-existing/api/nonesuch",
                  post=True, data="yadayada")
        self.mocker.throw(HTTPCodeError(404, "Not Found"))

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        try:
            self.service._fetch("http://non-existing/api/nonesuch", data="yadayada")
            self.fail("Should have raised an exception")
        except ProxyServiceError, e:
            self.assertEquals(e.message, "Server returned HTTP code 404")

    def test_wb__fetchRaisesUpstreamError(self):
        fetchMock = self.mocker.replace(fetch)
        fetchMock("http://non-existing/api/nonesuch", post=True, data="")
        self.mocker.result('{"error-message": "Some error"}')

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        try:
            self.service._fetch("http://non-existing/api/nonesuch")
        except UpstreamError, e:
            self.assertEquals(e.message, "Some error")
        else:
            self.fail("UpstreamError not raised")


    def testGetUpgrades(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)
        image3 = self.createImage(1)
        image4 = self.createImage(2)

        response = ImageStoreResponse(
            {"images": [image3, image4]})

        fetchMock = self.mocker.replace(fetch)
        fetchMock(self.service.endpoint + "/upgrades",
                  post=True, data="image-uri=%s&image-uri=%s" % (
                      quote(image1["uri"], safe=""),
                      quote(image2["uri"], safe="")))
        self.mocker.result(str(response))

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        task = GetUpgradesTask([image1["uri"], image2["uri"]])
        deferred = self.service.addTask(task)

        def callback(result):
            self.assertEquals(result, response)

        deferred.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred)

    def testSearch(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)

        response = ImageStoreResponse({"hello": "there!"})

        searchTerms = "foo/bar"

        fetchMock = self.mocker.replace(fetch)
        fetchMock("%s/search?q=%s" % (self.service.endpoint,
                                      quote(searchTerms, safe="")),
                  post=False, data="")
        self.mocker.result(str(response))

        fetchMock(ARGS, KWARGS)
        self.mocker.count(0)
        self.mocker.replay()

        deferred = self.service.addTask(SearchTask(searchTerms))
        def callback(result):
            self.assertEquals(result, response)
        deferred.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred)
