#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from urllib import quote
import weakref
import os
import gc

from twisted.internet.defer import Deferred

from imagestore.lib.tests.mocker import ARGS, KWARGS
from imagestore.lib.fetch import fetch, PyCurlError, HTTPCodeError
from imagestore.lib.twistedutil import mergeDeferreds
from imagestore.lib.service import (
        Service, ThreadedService, taskHandler, ServiceHub)

from imagestore.model import ImageState, ImageRegistration
from imagestore.tests.helpers import ServiceTestCase
from imagestore.signatureservice import CheckImageSignatureTask
from imagestore.downloadservice import DownloadService, DownloadFileTask
from imagestore.eucaservice import EucaService, BundleAndUploadImageTask
from imagestore.installservice import (
    GetStatesTask, InstallService, InstallImageTask, CancelChangesTask)
from imagestore.storageservice import (
    GetStoredImagesTask, GetStoredImageStatesTask, SetImageRegistrationTask,
    SetErrorMessageTask)

from imagestore.tests.helpers import ServiceTestCase


class InstallServiceTest(ServiceTestCase):

    def setUp(self):
        class Tracker(object):
            def __init__(self):
                self.names = []
            def count(self, name):
                return self.names.count(name)
            def __call__(self, name):
                self.names.append(name)
        self.track = Tracker()

    def assertTrack(self, deferred, *expectedNames):
        def callback(result):
            self.assertEquals(self.track.names, list(expectedNames))
            return result
        def errback(result):
            print "WARNING: assertTrack not happening due to error"
            return result
        deferred.addCallbacks(callback, errback)

    def createInstallService(self, fakeHandlersServiceClass):
        serviceHub = ServiceHub()
        serviceHub.addService(fakeHandlersServiceClass())
        return InstallService(serviceHub)

    def startWaitStop(self, installService, deferred):
        serviceHub = installService._serviceHub
        return self.runServicesAndWaitForDeferred([installService,
                                                   serviceHub], deferred)
 
    def getFile(self, image, kind):
        for file in image["files"]:
            if file["kind"] == kind:
                return file
        return None

    def testCheckSignatureTask(self):
        """
        Before doing anything else, the install service must check to see
        if the signature is valid.  If then passes the image on to be
        installed.
        """
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                self.track("check-signature")
                self.assertEquals(task.image, image)
                task.image["tainted"] = True
                return task.image

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")
                # The image we got here must have passed through the
                # check signature task before.
                self.assertEquals(task.imageRegistration.image["tainted"], True)

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistration(s, task):
                pass # Not important here.

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred,
                         "check-signature", "download", "download", "download",
                         "register")
        return self.startWaitStop(installService, deferred)

    def testDownloadFileTask(self):
        """
        When installing an image, the first big task which must be done is
        downloading all the image files.  It should download the smaller
        ones first, since it'd be frustrating to get a big file and then
        fail on the small ones.
        """
        image = self.createImage(1, withFiles=True)

        kernelFile = self.getFile(image, "kernel")
        ramdiskFile = self.getFile(image, "ramdisk")
        imageFile = self.getFile(image, "image")

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                if self.track.count("download") == 1:
                    file = self.getFile(image, "kernel")
                    localPath = "/downloaded/kernel"
                elif self.track.count("download") == 2:
                    file = self.getFile(image, "ramdisk")
                    localPath = "/downloaded/ramdisk"
                elif self.track.count("download") == 3:
                    file = self.getFile(image, "image")
                    localPath = "/downloaded/image"
                else:
                    self.fail("Too many files being downloaded.")
                    return
                self.assertEquals(task.url, file["url"])
                self.assertEquals(task.sha256, file["sha256"])
                self.assertEquals(task.size, file["size-in-bytes"])
                return localPath

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                # The local paths individually returned above must have been
                # set in the image registration which we get in here.
                self.track("register")
                registration = task.imageRegistration
                self.assertEquals(registration.eki.path, "/downloaded/kernel")
                self.assertEquals(registration.eri.path, "/downloaded/ramdisk")
                self.assertEquals(registration.emi.path, "/downloaded/image")
                registration.eki.eid = "eki-foo"
                registration.eri.eid = "eri-foo"
                registration.emi.eid = "emi-foo"
                return registration

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistration(s, task):
                # Finally, the image registation decorated with actual eids
                # must be seen in this task so that it may be stored.
                self.track("store")
                registration = task.imageRegistration
                self.assertEquals(registration.eki.eid, "eki-foo")
                self.assertEquals(registration.eri.eid, "eri-foo")
                self.assertEquals(registration.emi.eid, "emi-foo")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred, "download", "download", "download",
                                   "register", "store")
        return self.startWaitStop(installService, deferred)

    def testDownloadFailureAbortsOtherDownloads(self):
        """
        When the downloading of an image file fails, there's no reason to
        continue downloading other files of the same image.  Instead, we
        cancel all files of the same image.
        """
        image1 = self.createImage(1, withFiles=True)
        image2 = self.createImage(2, withFiles=True)

        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                # A download error raised inside the DownloadService...
                self.track("download")
                if task.url in (self.getFile(image1, "kernel")["url"],
                                self.getFile(image2, "kernel")["url"]):
                    self.track("raised")
                    raise MyException("Something bad happened.")
                elif task.progress.wasCancelled():
                    # ... must cancel all the other on going tasks.
                    self.track("got cancelled")
                else:
                    self.track("got active")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # We don't care about this in this test.

        installService = self.createInstallService(FakeHandlers)
        deferred1 = installService.addTask(InstallImageTask(image1))
        deferred2 = installService.addTask(InstallImageTask(image2))
        # The error should reach the outside deferred too.
        self.assertFailure(deferred1, MyException)
        self.assertFailure(deferred2, MyException)
        # If fails on the first download for each image, and that cancels
        # the following ones.
        self.assertTrack(deferred2,
                         "download", "raised",
                         "download", "got cancelled",
                         "download", "got cancelled",
                         "download", "raised",
                         "download", "got cancelled",
                         "download", "got cancelled")
        return self.startWaitStop(installService,
                                  mergeDeferreds([deferred1, deferred2]))

    def testCheckSignatureFailureSetsImageErrorMessage(self):
        """
        An error in the signature checking will stop the installation
        and set the image error message.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                self.track("check-signature")
                raise MyException("BOOM!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                self.track("set-error")
                self.assertEquals(task.errorMessage, "BOOM!")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred, "check-signature", "set-error")
        return self.startWaitStop(installService, deferred)

    def testDownloadFailureSetsImageErrorMessage(self):
        """
        When downloading of an image fails, the error message should
        be saved in the storage so that it may be retrieved by future
        state requests.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                # A download error raised inside the DownloadService...
                self.track("download")
                raise MyException("Something bad happened.")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                # ...should end up stored in the StorageService.
                self.track("set-error")
                self.assertEquals(task.imageUri, image["uri"])
                self.assertEquals(task.errorMessage, "Something bad happened.")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        # The error should reach the outside deferred too.
        self.assertFailure(deferred, MyException)
        # We get one error for each file because, due to the synchronicity,
        # downloads are not cancelled until they've already failed. But then,
        # the issue is handled a single time by the error handler.
        self.assertTrack(deferred,
                         "download", "download", "download", "set-error")
        return self.startWaitStop(installService, deferred)

    def testRegistrationFailureSetsImageErrorMessage(self):
        """
        When registration of an image fails, the error message should
        be saved in the storage so that it may be retrieved by future
        state requests.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                pass # We don't care about it in this test.

            @taskHandler(BundleAndUploadImageTask)
            def registerImage(s, task):
                # An error raised inside the EucaService...
                self.track("register")
                raise MyException("Something bad happened.")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                # ...should end up stored in the StorageService.
                self.track("set-error")
                self.assertEquals(task.imageUri, image["uri"])
                self.assertEquals(task.errorMessage, "Something bad happened.")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        # The error should reach the outside deferred too.
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred, "register", "set-error")
        return self.startWaitStop(installService, deferred)

    def testStoreRegistrationFailureSetsImageErrorMessage(self):
        """
        When storing the registration of an image fails, the error
        message should be saved in the storage so that it may be
        retrieved by future state requests.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            @taskHandler(BundleAndUploadImageTask)
            def ignore(s, task):
                pass # We don't care about it in this test.

            @taskHandler(SetImageRegistrationTask)
            def setImageRegstration(s, task):
                # An error raised inside the EucaService...
                self.track("store-registration")
                raise MyException("Something bad happened.")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                # ...should end up stored in the StorageService.
                self.track("set-error")
                self.assertEquals(task.imageUri, image["uri"])
                self.assertEquals(task.errorMessage, "Something bad happened.")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        # The error should reach the outside deferred too.
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred, "store-registration", "set-error")
        return self.startWaitStop(installService, deferred)

    def testRegistration(self):
        image = self.createImage(1, withFiles=True)

        kernelFile = self.getFile(image, "kernel")
        ramdiskFile = self.getFile(image, "ramdisk")
        imageFile = self.getFile(image, "image")

        def checkRegistration(reg):
            self.assertEquals(reg.eki.sha256, kernelFile["sha256"])
            self.assertEquals(reg.eki.size, kernelFile["size-in-bytes"])
            self.assertEquals(reg.eki.kind, kernelFile["kind"])
            self.assertEquals(reg.eri.sha256, ramdiskFile["sha256"])
            self.assertEquals(reg.eri.size, ramdiskFile["size-in-bytes"])
            self.assertEquals(reg.eri.kind, ramdiskFile["kind"])
            self.assertEquals(reg.emi.sha256, imageFile["sha256"])
            self.assertEquals(reg.emi.size, imageFile["size-in-bytes"])
            self.assertEquals(reg.emi.kind, imageFile["kind"])

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                pass # We don't care about it in this test.

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                # The local paths individually returned above must have been
                # set in the image registration which we get in here.
                self.track("register")
                checkRegistration(task.imageRegistration)
                return task.imageRegistration

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistration(s, task):
                # The local paths individually returned above must have been
                # set in the image registration which we get in here.
                self.track("store")
                checkRegistration(task.imageRegistration)

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        def installDone(registration):
            self.track("done")
            checkRegistration(registration)

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(installDone)
        self.assertTrack(deferred, "register", "store", "done")
        return self.startWaitStop(installService, deferred)

    def testNoTaskReferencesLeftAround(self):
        """
        Once an image is dealt with, no task references should be kept.
        """
        image = self.createImage(1, withFiles=True)

        tasks = []

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            @taskHandler(BundleAndUploadImageTask)
            @taskHandler(SetImageRegistrationTask)
            def saveTask(s, task):
                self.track("saved %s" % task.__class__.__name__)
                tasks.append(weakref.ref(task))
                del task

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred,
                         "saved DownloadFileTask",
                         "saved DownloadFileTask",
                         "saved DownloadFileTask",
                         "saved BundleAndUploadImageTask",
                         "saved SetImageRegistrationTask")

        # Rather than hooking into the task deferred, we'll hook into
        # the deferred which the whole test is waiting on, so that
        # hopefully all the references to the existing tasks which are
        # moving back and forth between callbacks already went away.
        finalDeferred = self.startWaitStop(installService, deferred)
        def checkTasks(result):
            gc.collect()
            for taskRef in tasks:
                self.assertEquals(taskRef(), None)
        finalDeferred.addCallback(checkTasks)
        return finalDeferred

    def testNoTaskReferencesLeftAroundOnDownloadError(self):
        """
        Once an image is dealt with, no task references should be kept.
        """
        image = self.createImage(1, withFiles=True)
        tasks = []
        class MyException(Exception): pass

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def saveTask(s, task):
                self.track("saved %s" % task.__class__.__name__)
                tasks.append(weakref.ref(task))
                del task
                raise MyException("Boom!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessage(s, task):
                pass # We don't really care about the error raised above here.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred,
                         "saved DownloadFileTask",
                         "saved DownloadFileTask",
                         "saved DownloadFileTask")

        # Rather than hooking into the task deferred, we'll hook into
        # the deferred which the whole test is waiting on, so that
        # hopefully all the references to the existing tasks which are
        # moving back and forth between callbacks already went away.
        finalDeferred = self.startWaitStop(installService, deferred)
        def checkTasks(result):
            gc.collect()
            for taskRef in tasks:
                self.assertEquals(taskRef(), None)
        finalDeferred.addCallback(checkTasks)
        return finalDeferred

    def testRemoveFilesOnSuccess(self):
        """
        After successfully installing an image, files for the given
        image should be removed from the disk to save space.
        """
        image = self.createImage(1, withFiles=True)

        kernelFile = self.getFile(image, "kernel")
        ramdiskFile = self.getFile(image, "ramdisk")
        imageFile = self.getFile(image, "image")

        createdPaths = []

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                localPath = self.makeFile("content")
                createdPaths.append(localPath)
                return localPath

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                pass

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistration(s, task):
                pass

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        def installFinished(result):
            self.assertEquals(len(createdPaths), 3)
            for localPath in createdPaths:
                self.assertFalse(os.path.exists(localPath))

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(installFinished)
        return self.startWaitStop(installService, deferred)

    def testGetStateWithAnyUnhandledStoredStatus(self):
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                self.track("get-stored")
                self.assertEquals(task.uris, ["http://uri"])
                return [{"image-uri": "http://uri", "status": "some"}]

        def checkResult(states):
            self.assertEquals(len(states), 1)
            state = states[0]
            self.assertEquals(state["image-uri"], "http://uri")
            self.assertEquals(state["status"], "some")
            self.assertEquals(state["actions"], {})

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(GetStatesTask(["http://uri"]))
        deferred.addCallback(checkResult)
        self.assertTrack(deferred, "get-stored")
        return self.startWaitStop(installService, deferred)

    uriBase = "http://localhost:52780/api/"

    def testGetStateWithUninstalledStatus(self):
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                self.track("get-stored")
                self.assertEquals(task.uris, ["http://uri"])
                return [{"image-uri": "http://uri", "status": "uninstalled"}]

        def checkResult(states):
            self.assertEquals(len(states), 1)
            state = states[0]
            self.assertEquals(state["image-uri"], "http://uri")
            self.assertEquals(state["status"], "uninstalled")
            self.assertEquals(state["actions"], {
                "install": self.uriBase + "images/aHR0cDovL3VyaQ==/install",
                })

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(GetStatesTask(["http://uri"]))
        deferred.addCallback(checkResult)
        self.assertTrack(deferred, "get-stored")
        return self.startWaitStop(installService, deferred)

    def testGetStateWithErrorMessage(self):
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                self.track("get-stored")
                self.assertEquals(task.uris, ["http://uri"])
                return [{"image-uri": "http://uri",
                         "status": "some",
                         "error-message": "Some error!"}]

        def checkResult(states):
            self.assertEquals(len(states), 1)
            state = states[0]
            self.assertEquals(state["image-uri"], "http://uri")
            self.assertEquals(state["status"], "some")
            self.assertEquals(state["error-message"], "Some error!")
            self.assertEquals(state["actions"], {
                "clear-error": self.uriBase +
                               "images/aHR0cDovL3VyaQ==/clear-error",
                })

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(GetStatesTask(["http://uri"]))
        deferred.addCallback(checkResult)
        self.assertTrack(deferred, "get-stored")
        return self.startWaitStop(installService, deferred)

    def testGetStateWithDownloadingStatus(self):
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                deferred = installService.addTask(GetStatesTask([image["uri"]]))
                deferred.addCallback(checkResult)
                return deferred

            @taskHandler(BundleAndUploadImageTask)
            @taskHandler(SetImageRegistrationTask)
            def ignore(s, task):
                pass # We don't care about it in this test.

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                self.track("get-stored")
                self.assertEquals(task.uris, [image["uri"]])
                return [{"image-uri": image["uri"], "status": "some"}]

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        def checkResult(states):
            self.assertEquals(len(states), 1)
            state = states[0]
            self.assertEquals(state["image-uri"], image["uri"])
            self.assertEquals(state["status"], "downloading")
            self.assertEquals(state["actions"], {
                "cancel":
                    self.uriBase + "images/aHR0cDovL2V4YW1wbGUvMQ==/cancel",
                })

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred,
                         "download", "download", "download",
                         "get-stored", "get-stored", "get-stored")
        return self.startWaitStop(installService, deferred)

    def testGetStateWithInstallingStatus(self):
        image = self.createImage(1, withFiles=True)

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            @taskHandler(SetImageRegistrationTask)
            def ignore(s, task):
                pass # We don't care about it in this test.

            @taskHandler(BundleAndUploadImageTask)
            def registerImage(s, task):
                self.track("register")
                deferred = installService.addTask(GetStatesTask([image["uri"]]))
                deferred.addCallback(checkResult)
                return deferred

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                self.track("get-stored")
                self.assertEquals(task.uris, [image["uri"]])
                return [{"image-uri": image["uri"], "status": "some"}]

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        def checkResult(states):
            self.assertEquals(len(states), 1)
            state = states[0]
            self.assertEquals(state["image-uri"], image["uri"])
            self.assertEquals(state["status"], "installing")
            self.assertEquals(state["actions"], {})

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred, "register", "get-stored")
        return self.startWaitStop(installService, deferred)

    def testStatusChanges(self):
        """
        The image status should correctly transition from uninstalled to
        downloading to installing to installed.  We can't test installed
        here, because it depends on the storage service functionality, but
        we can test the downloading and installing transitions.
        """
        image = self.createImage(1, withFiles=True)

        def checkStatus(expectedStatus, result=None):
            def callback(states):
                self.assertEquals(states[0]["status"], expectedStatus)
                return result
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                self.track("check-signature")
                return checkStatus("downloading", task.image)

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                return checkStatus("downloading")

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")
                return checkStatus("installing")

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistrtion(s, task):
                self.track("store")
                return checkStatus("installing")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        def installDone(result):
            # It didn't *really* install the image, so it goes back to uninst.
            self.track("done")
            return checkStatus("uninstalled")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(installDone)
        self.assertTrack(deferred,
                         "check-signature",
                         "download", "download", "download",
                         "register", "store", "done")
        return self.startWaitStop(installService, deferred)

    def testStatusChangesWithErrorsOnSignature(self):
        """
        When an error happens on downloads, the status should go back
        to uninstalled, rather than staying as downloading.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        def checkStatus(expectedStatus):
            def callback(states):
                self.assertEquals(states[0]["status"], expectedStatus)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                raise MyException("Boom!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                return checkStatus("uninstalled")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(self, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

        def finished(result):
            return checkStatus("uninstalled")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(finished)
        self.assertFailure(deferred, MyException)
        return self.startWaitStop(installService, deferred)

    def testStatusChangesWithErrorsOnDownload(self):
        """
        When an error happens on downloads, the status should go back
        to uninstalled, rather than staying as downloading.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        def checkStatus(expectedStatus):
            def callback(states):
                self.assertEquals(states[0]["status"], expectedStatus)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                raise MyException("Boom!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                return checkStatus("uninstalled")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(self, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

        def finished(result):
            return checkStatus("uninstalled")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(finished)
        self.assertFailure(deferred, MyException)
        return self.startWaitStop(installService, deferred)

    def testStatusChangesWithErrorsOnRegistration(self):
        """
        When an error happens on image registration, the status should
        go back to uninstalled, rather than staying as installing.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        def checkStatus(expectedStatus):
            def callback(states):
                self.assertEquals(states[0]["status"], expectedStatus)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")
                raise MyException("Boom!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                self.track("set-error")
                return checkStatus("uninstalled")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

            @taskHandler(DownloadFileTask)
            def ignore(s, task):
                pass # We don't care about these in this test.

        def finished(result):
            return checkStatus("uninstalled")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(finished)
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred, "register", "set-error")
        return self.startWaitStop(installService, deferred)

    def testStatusChangesWithErrorsOnStoreRegistration(self):
        """
        When an error happens on image registration storage, the status
        should go back to uninstalled, rather than staying as installing.
        """
        image = self.createImage(1, withFiles=True)

        class MyException(Exception): pass

        def checkStatus(expectedStatus):
            def callback(states):
                self.assertEquals(states[0]["status"], expectedStatus)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            @taskHandler(BundleAndUploadImageTask)
            def ignore(s, task):
                pass # We don't care about these in this test.

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistration(s, task):
                self.track("register")
                raise MyException("Boom!")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                self.track("set-error")
                return checkStatus("uninstalled")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

        def finished(result):
            return checkStatus("uninstalled")

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        deferred.addCallback(finished)
        self.assertFailure(deferred, MyException)
        self.assertTrack(deferred, "register", "set-error")
        return self.startWaitStop(installService, deferred)

    def testDownloadProgressWithZeroProgress(self):
        image = self.createImage(1, withFiles=True)

        def checkProgress(expectedProgress):
            def callback(states):
                self.assertEquals(states[0].get("progress-percentage",
                                                "no-progress"),
                                  expectedProgress)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                return checkProgress(0)

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")
                return checkProgress("no-progress")

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistrtion(s, task):
                self.track("store")
                return checkProgress("no-progress")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred, "download", "download", "download",
                                   "register", "store")
        return self.startWaitStop(installService, deferred)

    def testDownloadProgressWithSomeProgress(self):
        image = self.createImage(1, withFiles=True)

        kernelFile = self.getFile(image, "kernel")
        ramdiskFile = self.getFile(image, "ramdisk")
        imageFile = self.getFile(image, "image")

        kernelSize = kernelFile["size-in-bytes"]
        ramdiskSize = ramdiskFile["size-in-bytes"]
        imageSize = imageFile["size-in-bytes"]

        kernelRatio = 0.25
        ramdiskRatio = 0.50
        imageRatio = 0.15

        totalSize = kernelSize + ramdiskSize + imageSize

        def checkProgress(expectedProgress):
            def callback(states):
                self.assertEquals(states[0].get("progress-percentage",
                                                "no-progress"),
                                  expectedProgress)
            deferred = installService.addTask(GetStatesTask([image["uri"]]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                if task.url == kernelFile["url"]:
                    task.progress(kernelSize, kernelSize * kernelRatio, 0, 0)
                elif task.url == ramdiskFile["url"]:
                    task.progress(ramdiskSize, ramdiskSize * ramdiskRatio, 0, 0)
                elif task.url == imageFile["url"]:
                    task.progress(imageSize, imageSize * imageRatio, 0, 0)
                    currentSize = (kernelSize * kernelRatio +
                                   ramdiskSize * ramdiskRatio +
                                   imageSize * imageRatio)
                    return checkProgress((currentSize * 100.0) / totalSize)

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")
                return checkProgress("no-progress")

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistrtion(s, task):
                self.track("store")
                return checkProgress("no-progress")

            @taskHandler(GetStoredImageStatesTask)
            def getStoredImageStates(s, task):
                return [{"image-uri": uri,
                         "status": "uninstalled"} for uri in task.uris]

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                pass # Any errors here should be from test failures above.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred, "download", "download", "download",
                                   "register", "store")
        return self.startWaitStop(installService, deferred)

    def testCancelChanges(self):
        image = self.createImage(1, withFiles=True)

        downloadTasks = []

        def performAndTestCancel():
            def callback(result):
                for downloadTask in downloadTasks:
                    self.assertTrue(downloadTask.progress.wasCancelled())
                raise PyCurlError(42, "Callback aborted (FAKE)")
            deferred = installService.addTask(CancelChangesTask(image["uri"]))
            deferred.addCallback(callback)
            return deferred

        class FakeHandlers(Service):

            @taskHandler(CheckImageSignatureTask)
            def checkImageSignature(s, task):
                return task.image # Not important here.

            @taskHandler(DownloadFileTask)
            def downloadFile(s, task):
                self.track("download")
                downloadTasks.append(task)
                if len(downloadTasks) == 3:
                    return performAndTestCancel()

            @taskHandler(BundleAndUploadImageTask)
            def bundleAndUploadImage(s, task):
                self.track("register")

            @taskHandler(SetImageRegistrationTask)
            def setImageRegistrtion(s, task):
                self.track("store")

            @taskHandler(SetErrorMessageTask)
            def setErrorMessageTask(s, task):
                self.track("set-error")
                # Must not get here.  Cancellation isn't an error.

        installService = self.createInstallService(FakeHandlers)
        deferred = installService.addTask(InstallImageTask(image))
        self.assertTrack(deferred, "download", "download", "download")
        return self.startWaitStop(installService, deferred)

