#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from urllib import  quote

from imagestore.model import ImageStoreResponse
from imagestore.lib.fetch import fetch, FetchError
from imagestore.lib.service import (
    ServiceTask, ServiceError, ThreadedService, taskHandlerInThread)


class ProxyServiceError(ServiceError):
    pass

class UpstreamError(ProxyServiceError):
    """Raised when the upstream API returns an error message in the response."""


class ProxyServiceTask(ServiceTask):
    pass

class GetDashboardTask(ProxyServiceTask):

    def __init__(self, upgrade_uris):
        self.upgrade_uris = upgrade_uris

class SearchTask(ServiceTask):

    def __init__(self, searchTerms):
        self.searchTerms = searchTerms


class GetUpgradesTask(ProxyServiceTask):

    def __init__(self, uris):
        self.uris = uris


class ProxyService(ThreadedService):

    def __init__(self, reactor, endpoint):
        self.endpoint = endpoint
        ThreadedService.__init__(self, reactor)

    def _fetch(self, uri, data="", post=True):
        try:
            response = fetch(uri, post=post, data=data)
        except FetchError, e:
            raise ProxyServiceError(str(e))
        response = ImageStoreResponse(response)
        if response.get("error-message"):
            raise UpstreamError(response["error-message"])
        return response

    @taskHandlerInThread(GetDashboardTask)
    def _getDashboard(self, task):
        return self._fetch(self.endpoint + "/dashboard",
                           self._quoteParam("upgrade-uri", task.upgrade_uris))

    @taskHandlerInThread(GetUpgradesTask)
    def _getUpgrades(self, task):
        return self._fetch(self.endpoint + "/upgrades",
                           self._quoteParam("image-uri", task.uris))

    @taskHandlerInThread(SearchTask)
    def _search(self, task):
        url = "%s/search?%s" % \
              (self.endpoint, self._quoteParam("q", [task.searchTerms]))
        return self._fetch(url, post=False)

    def _quoteParam(self, name, values):
        return "&".join("%s=%s" % (name, quote(v, safe="")) for v in values)
