/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Streams")]
private interface StreamsInterface : Object {
        public abstract string Messages (string stream, string account, int time, string transient, string recipient, string orderby, string order, int limit) throws GLib.IOError;
        public abstract string Get (string id) throws GLib.IOError;
        public abstract string List () throws GLib.IOError;
        public abstract string Create (string query) throws GLib.IOError;
        public abstract void Delete (string id) throws GLib.IOError;
        public signal void Updated (string id);
        public signal void Created (string id);
        public signal void Deleted (string id);
}

namespace Gwibber
{
    public class Streams : Object
    {
        private const string streams_name  = "com.Gwibber.Streams";
        private const string streams_path  = "/com/gwibber/Streams";

        private StreamsInterface streams_service;
        private Gwibber.Service service;
	private Gwibber.Utils utils;
	private Gwibber.Messages messages_service;
	private Gwibber.Searches searches_service;
        private Dee.Model? _stream_model = null;
        private Dee.Model? _transient_model = null;
        private unowned Dee.ResourceManager resources;
        private Gee.HashMap <string, unowned Dee.ModelIter?> seen;

        public Dee.Model stream_model {
          get { 
            if (_stream_model != null) {
              return _stream_model;
            } else {
              _stream_model = streams_model (false);
              return _stream_model;
            }
          }
          set {
            if (_stream_model != value)
              _stream_model = value;
          }
        }

        public Dee.Model transient_model {
          get {
            if (_transient_model != null) {
              return _transient_model;
            } else {
              _transient_model = streams_model (true);
              return _transient_model;
            }
          }
          set {
            if (_transient_model != value)
              _transient_model = value;
          }
        }

	/** 
            Streams::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Streams availability state changes
	*/
	public signal void is_available (bool is_up);

        [Signal (action=true)]
        public virtual signal void updated (string id)
        {
        }

        public signal void deleted (string id);
        public signal void created (string id);


    public Streams ()
    {
            try
            {
                streams_service = Bus.get_proxy_sync(BusType.SESSION,
                                                          streams_name,
                                                          streams_path);
                utils = new Gwibber.Utils();
                utils.setup(streams_name);
                utils.available.connect(streams_available);
                messages_service = new Gwibber.Messages ();
                searches_service = new Gwibber.Searches ();
                service = new Gwibber.Service ();
                streams_service.Updated.connect((source) => {
                  stream_updated (source);
                });
                streams_service.Created.connect((source) => {
                  stream_created (source);
                });
                streams_service.Deleted.connect((source) => {
                  stream_deleted (source);
                });

            }
            catch (GLib.IOError e)
            {
                warning ("Unable to get Gwibber Streams: "+e.message);
            }
            resources = Dee.ResourceManager.get_default ();
            seen = new Gee.HashMap <string, unowned Dee.ModelIter?> ();
        }


        /**
         * com.Gwibber.Streams
         **/
        public string messages(string stream, string account, int time, string? transient)
        {
            if (transient == null)
              transient = "0";

            try
            {
                Timer timer = new Timer();
                var msgs = streams_service.Messages(stream, account, time, transient, "0", "time", "DESC", 1000);
                debug ("Retrieved messages from stream '%s' in %fms", stream, timer.elapsed()*1000);
                return msgs;
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
                return "";
            }
          
        }   

        public Dee.Model? stream_filter_model (Dee.Model model, string stream)
        {
            Dee.Filter stream_filter;
            // If home stream requested, return full model
            if (stream == "home") {
              return model;
            } else {
              stream_filter = Dee.Filter.new_for_key_column (StreamModelColumn.STREAM, stream);
            }
            var filtered_model = new Dee.FilterModel (model, stream_filter);
            return filtered_model;
        }

        public Dee.Model? transient_filter_model (Dee.Model model, string transient)
        {
            Dee.Filter transient_filter;
            transient_filter = Dee.Filter.new_for_key_column (StreamModelColumn.TRANSIENT, transient);
            var filtered_model = new Dee.FilterModel (model, transient_filter);
            return filtered_model;
        }

        private Dee.SequenceModel? create_model()
        {
            var m = new Dee.SequenceModel();
            m.set_schema ("as", "s", "s", "s", "s", "b", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "d", "b", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s", "s");
            debug ("SCHEMA has %u rows", m.get_n_columns ());
            return m;
        }

	public Dee.Model? streams_model(bool transients)
	{
            uint schema_length = 39;

            resources = Dee.ResourceManager.get_default ();

            if (!transients)
            {
              debug ("Getting non-transient model");
              Dee.SequenceModel? model = null;
              try
              {
                model = resources.load ("gwibber.stream_model") as Dee.SequenceModel;
              }
              catch (Error e)
              {
              }
              if (model is Dee.Model)
                debug ("stream_model from resources has %u rows", model.get_n_rows ());
              else
                debug ("stream_model from resources isn't valid");
              if (!(model is Dee.SequenceModel))
              {
                debug ("Didn't get model from resource manager, creating a new one");
                model = create_model ();
              }
              if (model.get_n_columns () != schema_length)
              {
                debug ("SCHEMA isn't current, recreating");
                model = null;
                model = create_model ();
              }
              model.row_removed.connect((_m, _i) => {
                var to_remove = new GLib.List <string> ();
                foreach (var v in seen.entries)
                {
                  if (v.value == _i)
                  {
                    /* defer calling unset until after we are done iterating 
                     * the HashMap, it will resize making the iterator invalid
                     * This is fixed in the 0.7 series of libgee
                     * https://bugzilla.gnome.org/show_bug.cgi?id=671327
                     */
                    to_remove.prepend (v.key);
                  }
                }
                foreach (var v in to_remove)
                  seen.unset(v);
              });
              Idle.add(() => {
                refresh_model_async.begin ();
                return false;
              });
              messages_service.message.connect((change, data) => {
                  on_refresh(change, data);});
              return model;
            }
            else
            {
              debug ("Getting transient model");
              Dee.SequenceModel? model = null;
              try
              {
                model = resources.load ("gwibber.transient_model") as Dee.SequenceModel;
              }
              catch (Error e)
              {
              }
              debug ("transient_model from resources has %u rows", model.get_n_rows ());
              if (!(model is Dee.SequenceModel))
              {
                debug ("Didn't get the transient model from resource manager, creating a new one");
                model = create_model ();
              }
              if (model.get_n_columns () != schema_length)
              {
                debug ("SCHEMA isn't current, recreating");
                model = null;
                model = create_model ();
              }
              Idle.add (() => {
                  refresh_transient_model_async.begin ();
                  return false;
              });
              return model;
            }
        }

        private void on_refresh(string change, string data)
        {
            //debug ("on_refresh %s - %s", change, data);
            var parser = new Json.Parser();
            try
            {
              parser.load_from_data(data, -1);
            }
            catch (Error e)
            {
            }
            unowned Json.Node node = parser.get_root();
            Json.Object obj = node.get_object();
            if (obj != null)
              parse_message (obj, change);
        }

        private async void refresh_model_async()
        {
            debug ("refresh_model_async");
            var streams = new List<string> ();
            streams.append("replies");
            streams.append("private");
            streams.append("public");
            streams.append("images");
            streams.append("links");
            streams.append("videos");
            streams.append("messages");

            Timeout.add (100, () => {
              var timer = new Timer();
              debug("Refreshing model - fetching streams");
              stream_model.clear ();
              foreach (string stream in streams)
              {
                string msgs = messages(stream, "all", 0, null);
                parse_messages (msgs);
              }
              debug ("Fetched and parsed %u streams in %fms", streams.length(), timer.elapsed()*1000);

              debug ("Storing gwibber.stream_model");
              try
              {
                resources.store ((Dee.SequenceModel)stream_model, "gwibber.stream_model");
              }
              catch (Error e)
              {
              }
              return false;
            });
        }

        private async void refresh_transient_model_async()
        {
            debug ("refresh_transient_model_async");
            Timeout.add (100, () => {
              var timer = new Timer();
              string msgs;
              transient_model.clear ();
              var searches = searches_service.list ();
              foreach (Gee.HashMap <string?, string?> x in searches)
              {
                msgs = messages("all", "all", 0, x["id"]);
                parse_messages (msgs);
              }
              debug ("Fetched and parsed %u searches in %fms", searches.length(), timer.elapsed()*1000);
              timer.reset ();
              var streams_list = list ();
              foreach (Gee.HashMap <string?, string?> y in streams_list)
              {
                msgs = messages("all", y["account"], 0, y["id"]);
                parse_messages (msgs);
              }
              debug ("Fetched and parsed %u transient streams in %fms", streams_list.length(), timer.elapsed()*1000);
              debug ("Storing gwibber.transient_model");
              try
              {
                resources.store ((Dee.SequenceModel)transient_model, "gwibber.transient_model");
              }
              catch (Error e)
              {
              }
              debug ("transient_model has %u rows", transient_model.get_n_rows ());
              return false;
            });
        }
       
        private void parse_messages (string data)
        {
            var timer = new Timer();
            var parser = new Json.Parser();
            try
            {
              parser.load_from_data(data, -1);
            }
            catch (Error e)
            {
            }
            Json.Array nodeArray = parser.get_root().get_array();

            for(int i = 0; i < nodeArray.get_length(); i++) {
              Json.Object obj = nodeArray.get_element(i).get_object();
              if (obj != null)
                parse_message (obj, "new");
            }
            debug ("Parsed %u messages in %fms", nodeArray.get_length(), timer.elapsed()*1000);
        }

        private void parse_message (Json.Object obj, string change)
        {
            string _account = null;
            string _stream = null;
            string _service = null;
            string _operation = null;
            string _mid = null;
            string _source = null;
            string _reply_nick = null;
            string _reply_name = null;
            string _reply_url = null;
            double _likes = 0;
            bool _liked = false;
            string _retweeted_by_nick = null;
            string _retweeted_by_name = null;
            string _retweeted_by_id = null;
            string _link_picture = null;
            string _link_name = null;
            string _link_url = null;
            string _link_description = null;
            string _link_caption = null;
            string _link_icon = null;
            string _image_url = null;
            string _image_src = null;
            string _image_thumb = null;
            string _image_name = null;
            string _video_picture = null;
            string _video_src = null;
            string _video_url = null;
            string _video_name = null;
            bool _from_me = false;
            uint _time = 0;
            string _html = null;
            string _text = null;
            string _icon = null;
            string _url = null;
            string _comments = null;
            string transient = null;
            unowned Dee.ModelIter? iter = null;
            var members = obj.get_members();

            if (obj.has_member("transient"))
              transient = obj.get_string_member("transient");
            
            if (transient == null)
              transient = "0";
            var _model = stream_model;
            if (transient.length > 1)
              _model = transient_model;

            if (!(_model is Dee.Model))
            {
              debug ("ERROR: Not a valid model, transient is %s", transient);
              return;
            }
            if (transient.length > 1 && change != "new")
              return;

            if (obj.has_member("account"))
              _account = obj.get_string_member("account");
            if (obj.has_member("stream"))
              _stream = obj.get_string_member("stream");
            if (_stream == "send_private")
              _stream = "private";
            if (_stream == "send_thread")
              _stream = "replies";
            if (obj.has_member("service"))
              _service = obj.get_string_member("service");
            if (obj.has_member("operation"))
              _operation = obj.get_string_member("operation");
            if (obj.has_member("mid"))
              _mid = obj.get_string_member("id");
            string _acct = _account + ":" + _service + ":" + _mid;
            var builder = new VariantBuilder (new VariantType ("as"));
            builder.add ("s", _acct);

            Json.Object _sender_obj = null;
            string _sender = "";
            string _sender_nick = "";
            Json.Object _recipient_obj = null;
            string _recipient = "";
            string _recipient_nick = "";
            string _recipient_icon = "";

            if (obj.has_member("recipient"))
            {
              _recipient_obj = obj.get_object_member("recipient");

              if (_recipient_obj != null)
              {
                if (_recipient_obj.has_member("name"))
                  _recipient = _recipient_obj.get_string_member("name");
                if (_recipient_obj.has_member("nick"))
                  _recipient_nick = _recipient_obj.get_string_member("nick");
                if (_recipient_obj.has_member("image"))
                  _recipient_icon = _recipient_obj.get_string_member("image");
              }
            }

            if (obj.has_member("sender"))
            {
              _sender_obj = obj.get_object_member("sender");
              
              if (_sender_obj != null)
              {
                if (_sender_obj.has_member("name"))
                  _sender = _sender_obj.get_string_member("name");
                if (_sender_obj.has_member("nick"))
                  _sender_nick = _sender_obj.get_string_member("nick");
              }
            }

            if (_sender_obj.has_member("is_me"))
              _from_me = _sender_obj.get_boolean_member("is_me");

            if (obj.has_member("time"))
              _time = (uint)obj.get_double_member("time");
            if (_time < 1 && obj.has_member("time"))
            {
                _time = (uint)obj.get_int_member("time");
            }
            if (obj.has_member("content"))
              _html = obj.get_string_member("content");
            if (obj.has_member("text"))
              _text = obj.get_string_member("text");

            string _textid = gen_text_id (_text);

            // FIXME: the transient == 0 disables dupe checking for transients, we should fix that
            //string _icon = yield service.avatar_path(_sender_obj.get_string_member("image"));
            if (_sender_obj.has_member("image"))
              _icon = _sender_obj.get_string_member("image");
            if (obj.has_member("url"))
              _url = obj.get_string_member("url");

            foreach (string member in members)
            {
              if (member == "retweeted_by")
              {
                Json.Object _retweet_obj = obj.get_object_member(member);
                if (_retweet_obj != null)
                {
                  if (_retweet_obj.has_member("nick"))
                    _retweeted_by_nick = _retweet_obj.get_string_member("nick");
                  if (_retweet_obj.has_member("name"))
                    _retweeted_by_name = _retweet_obj.get_string_member("name");
                  if (_retweet_obj.has_member("id"))
                    _retweeted_by_id = _retweet_obj.get_int_member("id").to_string();
                }
              }

              if (member == "source")
              {
                _source = obj.get_string_member("source");
                if (_source != null)
                  _source = _source.replace("rel=\"nofollow\"", "");
              }
              if (member == "reply")
              {
                Json.Object _reply_obj = obj.get_object_member("reply");
                if (_reply_obj != null)
                {
                  if (_reply_obj.has_member("nick"))
                    _reply_nick = _reply_obj.get_string_member("nick");
                  if (_reply_obj.has_member("name"))
                    _reply_name = _reply_obj.get_string_member("name");
                  if (_reply_obj.has_member("url"))
                  {
                    _reply_url = _reply_obj.get_string_member("url");
                    if (_reply_url != null)
                      _reply_url = GLib.Markup.escape_text (_reply_url);
                  }
                }
              }
              if (member == "likes")
              {
                Json.Object _likes_obj = obj.get_object_member("likes");
                if (_likes_obj != null)
                {
                  if (_likes_obj.has_member("count"))
                    _likes = (double)_likes_obj.get_int_member("count");
                  if (_likes_obj.has_member("liked"))
                    _liked = _likes_obj.get_boolean_member("liked");
                }
              }
              if (member == "favorited")
              {
                if (obj.get_boolean_member("favorited"))
                {
                  _likes = -1;
                }
              }

              if (member == "link")
              {
                Json.Object _link_obj = obj.get_object_member(member);
                if (_link_obj != null)
                {
                  if (_link_obj.has_member("picture"))
                    _link_picture = _link_obj.get_string_member("picture");
                  if (_link_obj.has_member("name"))
                    _link_name = _link_obj.get_string_member("name");
                  if (_link_obj.has_member("url"))
                  {
                    _link_url = _link_obj.get_string_member("url");
                    if (_link_url != null)
                      _link_url = GLib.Markup.escape_text (_link_url);
                  }
                  if (_link_obj.has_member("description"))
                    _link_description = _link_obj.get_string_member("description");
                  if (_link_obj.has_member("caption"))
                    _link_caption = _link_obj.get_string_member("caption");
                  if (_link_obj.has_member("icon"))
                    _link_icon = _link_obj.get_string_member("icon");
                }
              }

              if (member == "photo")
              {
                Json.Object _photo_obj = obj.get_object_member(member);
                if (_photo_obj != null)
                {
                  if (_photo_obj.has_member("url"))
                  {
                    _image_url = _photo_obj.get_string_member("url");
                    if (_image_url != null)
                      _image_url = GLib.Markup.escape_text (_image_url);
                  }
                  if (_photo_obj.has_member("picture"))
                  {
                    _image_src = _photo_obj.get_string_member("picture");
                  }
                  if (_photo_obj.has_member("name"))
                    _image_name = _photo_obj.get_string_member("name");
                }
              }
              if (member == "video")
              {
                Json.Object _video_obj = obj.get_object_member(member);
                if (_video_obj != null)
                {
                  if (_video_obj.has_member("name"))
                    _video_name = _video_obj.get_string_member("name");
                  if (_video_obj.has_member("picture"))
                  {
                    _video_picture = _video_obj.get_string_member("picture");
                  }
                  if (_video_obj.has_member("url"))
                  {
                    _video_url = _video_obj.get_string_member("url");
                    if (_video_url != null)
                      _video_url = GLib.Markup.escape_text (_video_url);
                  }
                  if (_video_obj.has_member("source"))
                  {
                    _video_src = _video_obj.get_string_member("source");
                  }
                }
              }

              if (member == "images")
              {
                Json.Array _images_array = obj.get_array_member(member);
                if (_images_array != null)
                {
                  Json.Object _image_obj = _images_array.get_element(0).get_object();
                  if (_image_obj != null)
                  {
                    if (_image_obj.has_member("url"))
                    {
                      _image_url = _image_obj.get_string_member("url");
                      if (_image_url != null)
                        _image_url = GLib.Markup.escape_text (_image_url);
                    }
                    if (_image_obj.has_member("src"))
                    {
                      _image_src = _image_obj.get_string_member("src");
                      if (_image_src != null)
                        _image_src = GLib.Markup.escape_text (_image_src);
                    }
                    if (_image_obj.has_member("thumb"))
                    {
                      _image_thumb = _image_obj.get_string_member("thumb");
                      if (_image_thumb != null)
                        _image_thumb = GLib.Markup.escape_text (_image_thumb);
                    }
                    if (_image_obj.has_member("name"))
                      _image_name = _image_obj.get_string_member("name");
                  }
                }
              }
              if (member == "comments")
              {
                size_t length;
                var _comments_array = obj.get_array_member(member);

                var comment_gen = new Json.Generator();
                var root = new Json.Node(Json.NodeType.OBJECT);
                var object = new Json.Object();
                root.set_object(object);
                object.set_array_member ("comments", _comments_array);
                comment_gen.set_root(root);
                _comments = comment_gen.to_data(out length);
                //debug ("_comments: %s", _comments);
              }
            }

            /* escape markup in some strings, pango doesn't like it */
            if (_link_name != null)
              _link_name = GLib.Markup.escape_text (_link_name);
            if (_image_name != null)
              _image_name = GLib.Markup.escape_text (_image_name);
            if (_video_name != null)
              _video_name = GLib.Markup.escape_text (_video_name);

            if (_html != null)
              _html = scrub (_html);

            if (_link_description != null)
              _link_description = scrub (_link_description);

            /* If textid is empty, lets fallback to check some other fields */
            if (_textid.length < 1 && _html != null)
              _textid = gen_text_id (_html);
            if (_textid.length < 1 && _link_description != null)
              _textid = gen_text_id (_link_description);
            if (_textid.length < 1 && _link_name != null)
              _textid = gen_text_id (_link_name);
            if (_textid.length < 1 && _image_url != null)
              _textid = _image_url.strip ();
            if (_textid.length < 1 && _image_name != null)
              _textid = gen_text_id (_image_name);
            if (_textid.length < 1 && _video_url != null)
              _textid = _video_url.strip ();
            if (_textid.length < 1 && _video_name != null)
              _textid = gen_text_id (_video_name);

            string _t = utils.generate_time_string(_time);

            if (seen.has_key(_textid) && transient == "0")
            {
              iter = seen.get(_textid);
              if (iter == null)
                seen.unset(_textid);
              else
              {
                string[] _accounts_array = (string[])_model.get_value (iter, StreamModelColumn.ACCOUNTS);
                if (!(_acct in _accounts_array))
                {
                  foreach (var a in _accounts_array)
                    builder.add ("s", a);
                  var _accounts = new Variant ("as",  builder);
                  _model.set_value (iter, StreamModelColumn.ACCOUNTS, _accounts);
                }
                _model.set_value (iter, StreamModelColumn.MESSAGE, _text);
                _model.set_value (iter, StreamModelColumn.HTML, _html);
                _model.set_value (iter, StreamModelColumn.TIMESTRING, _t);
                _model.set_value (iter, StreamModelColumn.LIKES, _likes);
                _model.set_value (iter, StreamModelColumn.LIKED, _liked);
                iter = null;
                return;
              }
            }

            var _accounts = new Variant ("as",  builder);

            iter = _model.append(
              _accounts,
              _stream,
              transient,
              _sender,
              _sender_nick,
              _from_me,
              _time.to_string(),
              _text,
              _html,
              _icon,
              _url,
              _source,
              _t,
              _reply_nick,
              _reply_name,
              _reply_url,
              _likes,
              _liked,
              _retweeted_by_nick,
              _retweeted_by_name,
              _retweeted_by_id,
              _link_picture,
              _link_name,
              _link_url,
              _link_description,
              _link_caption,
              _link_icon,
              _image_url,
              _image_src,
              _image_thumb,
              _image_name,
              _video_picture,
              _video_src,
              _video_url,
              _video_name,
              _comments,
              _recipient,
              _recipient_nick,
              _recipient_icon
              );

            if (_textid.length > 1 && iter != null)
              seen[_textid] = iter;
            iter = null;
            //debug ("_model has %u ROWS", _model.get_n_rows ());
        }

        private string scrub (string content)
        {
          /* FIXME: hacky scrubbing of the html, we should find a 
          better way */
          string res = content;
          res = res.replace("&query", "&amp;query");
          res = res.replace("&name", "&amp;name");
          res = res.replace("class=\"nick\"", "");
          res = res.replace("class=\"hash\"", "");
          res = res.replace("<p>", "");
          res = res.replace("</p>", "");
          res = res.replace("<b>", "");
          res = res.replace("</b>", "");
          res = res.replace(" & ", " &amp; ");
          //debug ("res: %s", res);
          return res;
        }

        private string gen_text_id (string t)
        {
          string id = "";
          if (t.length > 0)
          { 
            foreach (var x in t.replace(".", "").split(" "))
              if ((!x.has_prefix("http://")) && (!x.has_prefix("https://")) 
                && (!x.has_prefix("gwibber:/")) && (!x.has_prefix("#"))
                && (!x.has_prefix("!")))
                  id += x + " ";
            id = id.strip();
          }
          return id;
        }



        /**
         * com.Gwibber.Streams
         **/
        public Gee.HashMap <string?, string?> lookup(string id)
        {
            var stream_map = new Gee.HashMap <string?, string?> ();
            try
            {
                var result = streams_service.Get(id);
                var parser = new Json.Parser();
                try
                {
                  parser.load_from_data(result, -1);
                }
                catch (Error e)
                {
                }
                Json.Object obj = parser.get_root().get_object();
                var members = obj.get_members();
                foreach (string member in members)
                {
                  stream_map[member] = obj.get_string_member(member);
                }
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
            }
            return stream_map;
        }


        /**
         * com.Gwibber.Streams
         **/
        public void delete(string id)
        {
            try
            {
                streams_service.Delete(id);
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
            }
            return;
        }


        /**
         * com.Gwibber.Streams
         **/

        public string create(string account, string name, string operation)
        {
            string parameters = "";

            var streams_list = list ();
            foreach (Gee.HashMap<string?, string?> s in streams_list)
            {
              if (s["name"] == name && s["operation"] == operation && s["account"] == account)
              {
                debug ("user stream exists, updating");
                updated(s["id"]);
                return s["id"];
              }
            }
            
            if (operation == "user_messages")
              parameters = "{\"count\": 50, \"id\": \"" + name + "\"}";

            try
            {
                return streams_service.Create("{\"account\": \"" + account + "\", \"parameters\": " + parameters + ", \"name\": \"" + name + "\", \"operation\": \"" + operation + "\"}");
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
                return "";
            }
        }


        /**
         * com.Gwibber.Streams
         **/
        public List<Gee.HashMap?> list()
        {
            var streams_list = new List<Gee.HashMap<string?, string?>> ();
            try
            {
                var result = streams_service.List();
                var parser = new Json.Parser();
                parser.load_from_data(result, -1);
                Json.Array nodeArray = parser.get_root().get_array();
                for(int i = 0; i < nodeArray.get_length(); i++) {
                  var stream_map = new Gee.HashMap <string?, string?> ();
                  var obj = nodeArray.get_element(i).get_object();
                  if (obj != null)
                  {
                    var members = obj.get_members();
                    foreach (string member in members)
                    {
                      if (member != "parameters")
                        stream_map[member] = obj.get_string_member(member);
                    }
                  }
                  streams_list.append(stream_map);
                }
            }
            catch (GLib.Error e)
            {
                warning (e.message);
            }
            return streams_list;
        }


        public void stream_deleted(string data)
        {
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                deleted(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public void stream_created(string data)
        {
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                created(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public void stream_updated(string data)
        {
            debug ("stream_updated");
            try {
                var parser = new Json.Parser();
                parser.load_from_data(data, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                updated(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public Gee.HashMap<string?, Value?> user_profile(string id)
        {
            var profile = new Gee.HashMap<string?, Value?> ();
            try
            {
                string name = "";
                string nick = "";
                string account = "";
                string service = "";
                string description = "";
                string image = "";
                string url = "";
                int followers = 0;
                int friends = 0;
                int statuses = 0;
                bool following = false;
                bool protected = false;

                var result = streams_service.Messages("profile", "all", 0, id, "0", "time", "DESC", 1);
                var parser = new Json.Parser();
                try
                {
                  parser.load_from_data(result, -1);
                }
                catch (Error e)
                {
                }
                Json.Array nodeArray = parser.get_root().get_array();

                Json.Object obj = nodeArray.get_element(0).get_object ();
                name = obj.get_string_member("name");
                nick = obj.get_string_member("nick");
                account = obj.get_string_member("account");
                service = obj.get_string_member("service");
                if (obj.has_member("text"))
                  description = obj.get_string_member("text");
                image = obj.get_string_member("image");
                if (obj.has_member("url"))
                  url = obj.get_string_member("url");
                followers = (int)obj.get_int_member("followers");
                friends = (int)obj.get_int_member("friends");
                statuses = (int)obj.get_int_member("statuses");
                following = obj.get_boolean_member("following");
                protected = obj.get_boolean_member("protected");

                profile["name"] = name;
                profile["nick"] = nick;
                profile["account"] = account;
                profile["service"] = service;
                profile["id"] = id;
                profile["description"] = description;
                profile["image"] = image;
                profile["url"] = url;
                profile["followers"] = followers;
                profile["friends"] = friends;
                profile["statuses"] = statuses;
                profile["following"] = following;
                profile["protected"] = protected;
            }
            catch (GLib.IOError e)
            {
                warning (e.message);
            }
            return profile;
        }


	public void streams_available(bool is_up)
	{
		is_available(is_up);
	}
    }
}
