/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

using GwibberGtk;

public class WriteItem : TabBarItem{

  public WriteItem ()
  {
    Object (stream:"", icon_name:"gwibber-new-post", tooltip:_("New Message"));
  }

  construct
  {
  }

  public override Gtk.Widget? get_toolbar ()
  {
    return null;
  }
  
  public override Gtk.Widget? get_view ()
  {
    return null;
  }
}

public class TabBar : Gtk.Box
{  
  private Gtk.Alignment _align;
  private Gtk.Box _vbox;
  private Gtk.Box _tab_box;
  private Gtk.Box _toolbar_box;

  private Gtk.Layout _layout;
  private int _last_width;
  private int _last_height;

  private Gtk.Box _view_box;
  private Gtk.SizeGroup _view_group;

  private WriteItem _write_item;
  private StreamEntry? _entry = null;

  private TabBarItem? _active_item = null;
  private int _active_item_n = 0;

  public signal void tab_changed (TabBarItem item);

  private uint _anim_id = 0;
  private float _anim_offset = 0.0f;
  private int64 _anim_start = 0;

  public TabBar ()
  {
    Object (homogeneous:false, spacing:0);
  }

  construct
  {
    set_orientation (Gtk.Orientation.VERTICAL);
    _align = new Gtk.Alignment (0.0f, 0.0f, 1.0f, 1.0f);
    _align.get_style_context ().add_class ("primary-toolbar");
    _align.get_style_context ().add_class ("toolbar");
    _align.draw.connect (on_align_draw);
    pack_start (_align, false, true, 0);

    _vbox = new Gtk.Box (Gtk.Orientation.VERTICAL, 8);
    _align.add (_vbox);

    _tab_box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
    _vbox.add (_tab_box);

    _toolbar_box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
    _vbox.add (_toolbar_box);

    _layout = new Gtk.Layout (null, null);
    pack_start (_layout, true, true, 0);
   
    _view_box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
    _view_box.set_homogeneous (true);
    _layout.put (_view_box, 0, 0);

    _view_group = new Gtk.SizeGroup (Gtk.SizeGroupMode.BOTH);

    _write_item = new WriteItem ();
    _write_item.clicked.connect (on_write_clicked);
    _tab_box.pack_end (_write_item, false, true, 0);

    _entry = new StreamEntry ();
    _entry.in_client = true;
    _layout.put (_entry, 0, 0);
    _entry.notify["showing"].connect (()=>
    {
      if (!_entry.showing)
        _entry.text_view.reset ();
      _view_box.set_sensitive (!_entry.showing);
    });

    _layout.size_allocate.connect_after ((a) => {
      if (_last_width != a.width || _last_height != a.height)
      {
        _last_width = a.width;
        _last_height = a.height;
        
        var children = _view_box.get_children ();
        foreach (Gtk.Widget child in children)
        {
          child.set_size_request (_last_width, _last_height);
          child.queue_draw ();
        }
        
        _view_box.set_size_request (_last_width * (int)children.length (),
                                    _last_height);
        _entry.set_size_request (_last_width,
                                 _last_height);

        queue_draw ();
      }
    });

    show_all ();
  }

  public bool on_align_draw (Cairo.Context cr)
  {
    cr.set_operator (Cairo.Operator.OVER);

    // Menu background
    _align.get_style_context ().render_background (cr,
                                                   0,
                                                   0,
                                                   _align.get_allocated_width (),
                                                   _align.get_allocated_height ());

    // Menu frame
    _align.get_style_context ().render_frame (cr,
                                              0,
                                              0,
                                              _align.get_allocated_width (),
                                              _align.get_allocated_height ());

    // Arrow
    if (_active_item is Gtk.Widget && !_entry.get_visible ())
    {
      Gtk.Allocation a;
      _active_item.get_allocation (out a);

      cr.set_line_width (1.0);
      cr.set_source_rgb (1.0f, 1.0f, 1.0f);

      // Main Arrow
      cr.move_to (a.x + a.width/2.0, _align.get_allocated_height () - 5);
      cr.line_to (a.x + a.width/2.0 + 5, _align.get_allocated_height ());
      cr.line_to (a.x + a.width/2.0 - 5, _align.get_allocated_height ());
      cr.close_path ();
      
      cr.fill ();

      if (_active_item.get_toolbar () is Gtk.Widget)
      {
        cr.save ();
        cr.translate (0.5, 0.5);

        draw_arrowed_line (cr, 0, a.height + 3, _align.get_allocated_width (), (int)(a.x + a.width/2.0));
        cr.set_source_rgba (1.0f, 1.0f, 1.0f, 0.2f);
        cr.stroke ();

        draw_arrowed_line (cr, 0, a.height + 2, _align.get_allocated_width (), (int)(a.x + a.width/2.0));
        cr.set_source_rgba (0.0f, 0.0f, 0.0f, 0.5f);
        cr.stroke ();

        cr.restore ();
      }
    }
    
    _align.propagate_draw (_vbox, cr);

    return true;
  }

  private void draw_arrowed_line (Cairo.Context cr, int x, int y, int width, int point)
  {
    int size = 5;

    cr.move_to (x, y);
    cr.line_to (point - size, y);
    cr.line_to (point, y - size);
    cr.line_to (point + size, y); 
    cr.line_to (x + width, y);
  }

  public void add_item (TabBarItem item)
  {
    item.clicked.connect (on_tab_clicked);
    _tab_box.add (item);

    if (item.get_toolbar () is Gtk.Widget)
    {
      var toolbar = item.get_toolbar ();
      toolbar.set_no_show_all (true);
      toolbar.hide ();
      _toolbar_box.add (toolbar);
    }

    if (item.get_view () is Gtk.Widget)
    {
      var view = item.get_view ();
      item.entry = _entry;
      view.show ();

      _view_box.add (view);
      _view_group.add_widget (view);

      var children = _view_box.get_children ();
      _view_box.set_size_request (_last_width * (int)children.length (),
                                  _last_height);
    }

    queue_draw ();
  }

  public void on_tab_clicked (Gtk.Button? gtk_item)
  {
    _entry.showing = false;
    _entry.text_view.reset ();

    TabBarItem? item = gtk_item as TabBarItem;

    if (_active_item == item || gtk_item == null)
      return;

    if (_active_item is TabBarItem)
    {
      _active_item.active = false;
      if (_active_item.get_toolbar () is Gtk.Widget)
        _active_item.get_toolbar ().hide ();
     }

    _active_item = item;
    _active_item.active = true;;

    if (_active_item.get_toolbar () is Gtk.Widget)
    {
      _active_item.get_toolbar ().show ();
      _align.set_padding (0, 8, 0, 0);
    }
    else
      _align.set_padding (0, 0, 0, 0);

    if (_active_item.get_view () is Gtk.Widget)
    {
      var children = _view_box.get_children ();
      int i = 0;
      foreach (Gtk.Widget child in children)
      {
        if (child == _active_item.get_view ())
          break;
        i++;
      }
      //_layout.move (_view_box, _layout.get_allocated_width () * i * -1, 0);
      _active_item_n = i;

      if (_anim_id != 0)
        Source.remove (_anim_id);

      _anim_start = G.get_monotonic_time ();
      
      _anim_id = Timeout.add (15, ()=>
      {
        float LENGTH = 200000.0f;
        int64 diff = G.get_monotonic_time () - _anim_start;
        float progress = diff/LENGTH;
        Gtk.Allocation a;
        _view_box.get_allocation (out a);

        float target_x = _active_item_n * _layout.get_allocated_width () * -1;
        float current_x = a.x;
        float new_x = 0;

        if (target_x > current_x)
        {
          new_x = current_x + ((target_x - current_x) * progress);
        }
        else if (target_x < current_x)
        {
          new_x = current_x - ((current_x - target_x) * progress);
        }

        _layout.move (_view_box, (int)new_x, 0);

        queue_draw ();

        if (diff > LENGTH)
        {
          _anim_id = 0;
          _anim_offset = 1.0f;
          _layout.move (_view_box, _layout.get_allocated_width () * _active_item_n * -1, 0);
          return false;
        }

        return true;
      });

      _active_item.get_view ().queue_draw ();
      _view_box.queue_draw ();
      _layout.queue_draw ();
    }

    _entry.queue_draw ();

    tab_changed (item);

    queue_draw ();
  }

  public void show_entry ()
  {
  }

  public void on_write_clicked ()
  {
    if (_entry.showing)
    {
      _entry.showing = false;
      _entry.text_view.reset ();
      _view_box.set_sensitive (true);
    }
    else
    {
      _entry.showing = true;
      _entry.text_view.grab_focus ();
      _view_box.set_sensitive (false);
    }
    queue_draw ();
  }
}
