/*
 * Copyright (C) 2010 Neil Jagdish Patel
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

using Gwibber;

namespace GwibberGtk
{
  public class StreamViewAvatar : Gtk.Image
  {
    public StreamViewAvatar()
    {
      draw.connect (on_draw);
    }

    private bool on_draw (Cairo.Context cr)
    {
      Gtk.Allocation alloc;
      get_allocation(out alloc);

      var radius = 10.0f;
      cr.move_to(0, radius);
      cr.curve_to(0, 0, 0, 0, radius, 0);
      cr.line_to(alloc.width - radius, 0);
      cr.curve_to(alloc.width, 0, alloc.width, 0, alloc.width, radius);
      cr.line_to(alloc.width, alloc.height - radius);
      cr.curve_to(alloc.width, alloc.height, alloc.width, alloc.height, alloc.width - radius, alloc.height);
      cr.line_to(radius, alloc.height);
      cr.curve_to(0, alloc.height, 0, alloc.height, 0, alloc.height - radius);
      cr.close_path();

      cr.clip();

      base.draw(cr);
      return true;
    }
  }

  public class StreamViewTile : Gtk.EventBox
  {
    private const int our_padding = 6;

    private Gtk.Alignment align;
    private Gtk.HBox   main_box;
    private Gtk.Alignment lalignment;
    private Gtk.Alignment ralignment;
    private Gtk.EventBox  icon_box;
    private StreamViewAvatar  icon;
    private Gtk.Image  private;
    private Gtk.VBox   vbox;
    private Gtk.VBox   comments_box;
    private Gtk.HBox   likes_hbox;
    private Gtk.Label  likes_count;
    private new Gtk.Label  name;
    //private new Gtk.Label  source;
    private Gtk.Label  time;
    private Gtk.Label  message;
    private StreamViewAvatar  thumbnail;
    private Gtk.EventBox thumb_box;
    private Gtk.Label  reply_to;
    private Gtk.Label  retweeted_by;
    public Gwibber.Utils utils { get; construct set; }
    public Gwibber.Service service { get; construct set; }
    public bool show_fullname { get; construct set; }
    private uint _update_time_area_id = 0;
    private uint _cache_avatar_id = 0;
    private List<uint> _to_disconnect;

    public uint uid = 0;

    private string sender = null;
    private string url = null;

    private string time_string = "";

    private GwibberGtk.ActionBox action_box;

    private Cancellable cancellable;
    private string img_uri;
    private string img_src;
    private uint img_id;

    public StreamViewTile (Gwibber.Utils utils, Gwibber.Service service, bool show_fullname)
    {
      Object (above_child:false, visible_window:false, utils:utils, service:service, show_fullname:show_fullname);
    }

    construct
    {
      _to_disconnect = new GLib.List<uint> ();
      draw.connect (on_draw);

      align = new Gtk.Alignment (0.0f, 0.0f, 1.0f, 1.0f);
      align.set_padding (our_padding, our_padding, our_padding, our_padding);
      add (align);

      main_box = new Gtk.HBox (false, 12);
      align.add (main_box);

      lalignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      ralignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      main_box.pack_start (lalignment, false, false, 0);
      main_box.pack_end (ralignment, false, false, 0);

      icon_box = new Gtk.EventBox ();
      icon_box.set_visible_window (false);
      icon = new StreamViewAvatar();
      icon_box.add(icon);
      lalignment.add (icon_box);

      vbox = new Gtk.VBox (false, 0);
      main_box.pack_start (vbox, true, true, 0);
      
      var hbox = new Gtk.HBox (false, 0);
      vbox.pack_start (hbox, false, false, 2);
      
      private = new Gtk.Image.from_icon_name ("status_lock", Gtk.IconSize.MENU);
      private.set_no_show_all (true);
      hbox.pack_start (private, false, false, 0);

      name = new Gtk.Label ("");
      name.set_markup ("<b></b>");
      name.set_single_line_mode (false);
      name.set_line_wrap (true);
      name.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
      name.set_alignment (0.0f, 0.5f);
      hbox.pack_start (name, false, false, 0);

      time = new Gtk.Label ("");
      time.set_markup ("");
      time.set_alignment (0.0f, 1.0f);
      hbox.pack_end (time, false, false, 0);

      action_box = new GwibberGtk.ActionBox ();
      action_box.hide ();
      //action_box.set_no_show_all (true);
      hbox.pack_end (action_box, false, false, 0);

      message = new Gtk.Label ("");
      message.set_selectable (true);
      message.set_single_line_mode (false);
      message.set_line_wrap (true);
      message.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
      message.set_alignment (0.0f, 0.0f);
      vbox.pack_start (message, false, false, 2);

      message.activate_link.connect((uri) => {
        if (uri.has_prefix("gwibber:/"))
        {
          debug ("URI is %s", uri);
          var url = new Soup.URI(uri);
          if (url.path.has_suffix ("tag"))
          {
            string acct = "";
            string query = "";
            string[] args = url.query.split("&");
            foreach (var x in args)
            {
              var args2 = x.split("=");
              if (args2[0] == "acct")
                acct = args2[1];
              if (args2[0] == "query")
                query = "#" + args2[1];
            }
            if (query.length > 0)
            {
              var searches_service = new Gwibber.Searches ();
              var searches = searches_service.list ();
              var _id = searches_service.create(query, query);
            }
          }
          else if (url.path.has_suffix ("user"))
          {
            string acct = "";
            string name = "";
            string[] args = url.query.split("&");
            foreach (var x in args)
            {
              var args2 = x.split("=");
              if (args2[0] == "acct")
                acct = args2[1];
              if (args2[0] == "name")
                name = args2[1];
            }
            if (name.length > 0 && acct.length > 0)
            {
              var streams_service = new Gwibber.Streams ();
              var streams_list = streams_service.list ();
              var _id = streams_service.create(acct, name, "user_messages");
            }
          }
        }
        return false;
        });

      thumb_box = new Gtk.EventBox ();
      thumb_box.set_visible_window (false);
      thumb_box.set_no_show_all(true);
      var thumb_hbox = new Gtk.HBox (false, 2);
      vbox.pack_start(thumb_hbox, false, false, 4);
      thumb_hbox.pack_start(thumb_box, false, false, 4);
      thumb_box.button_release_event.connect(() =>
      {
        if (img_src != null)
          Gtk.show_uri(Gdk.Screen.get_default(), img_src, 0);
        return false;
      });
     
      thumbnail = new StreamViewAvatar();
      thumbnail.set_no_show_all(true);
      thumb_box.add(thumbnail);
      thumb_box.hide ();

      comments_box = new Gtk.VBox (false, 0);
      comments_box.set_no_show_all (true);
      vbox.pack_start (comments_box, false, false, 4);

      reply_to = new Gtk.Label ("");
      reply_to.set_no_show_all(true);
      reply_to.set_markup ("<span font_weight='light' font_size='small'></span>");
      reply_to.set_alignment (0.0f, 0.5f);
      vbox.pack_start (reply_to, false, false, 2);

      retweeted_by = new Gtk.Label ("");
      retweeted_by.set_markup ("<span font_weight='light' font_size='small'></span>");
      retweeted_by.set_alignment (0.0f, 0.5f);
      retweeted_by.set_no_show_all(true);
      vbox.pack_start (retweeted_by, false, false, 2);

      likes_hbox = new Gtk.HBox (false, 0);
      likes_hbox.set_no_show_all (true);
      // FIXME: Find a theme icon
      var likes_icon = new Gtk.Image.from_file ("/usr/share/gwibber/ui/icons/streams/16x16/favorite.png");
      likes_hbox.pack_start (likes_icon, false, false, 2);
      likes_count = new Gtk.Label ("");
      likes_hbox.pack_start (likes_count, false, false, 0);
      vbox.pack_start (likes_hbox, false, false, 2);

      enter_notify_event.connect (() => {
        update_time_area_wrapper ();
        return false;
      });

      icon_box.enter_notify_event.connect (() => {
        update_time_area_wrapper ();
        return false;
      });

      message.enter_notify_event.connect (() => {
        update_time_area_wrapper ();
        return false;
      });

      leave_notify_event.connect (() => {
        /*
        if (_update_time_area_id != 0)
        {
          Source.remove(_update_time_area_id);
          _update_time_area_id = 0;
        }
        if (!(time.visible))
          update_time_area ();
        */
        update_time_area_wrapper ();
        return false;
      });

      cancellable = new Cancellable();

      Notify.init ("Gwibber");
    }


    private bool on_draw (Cairo.Context cr)
    {
      Gtk.Allocation alloc;
      get_allocation (out alloc);

      cr.set_line_width (1.0);

      cr.set_source_rgba (1.0, 1.0, 1.0, 1.0);
      cr.move_to (alloc.x, 0.5);
      cr.line_to (alloc.x + alloc.width, 0.5);
      cr.stroke ();

      var pat = new Cairo.Pattern.linear (0, 0, 0, alloc.height);
      pat.add_color_stop_rgba (0.0f, 1.0f, 1.0f, 1.0f, 0.2f);
      pat.add_color_stop_rgba (1.0f, 1.0f, 1.0f, 1.0f, 0.0f);

      cr.rectangle (0, 0, alloc.width, alloc.height);
      cr.set_source (pat);
      cr.fill ();

      cr.set_source_rgba (0.0, 0.0, 0.0, 0.1);
      cr.move_to (alloc.x, alloc.height - 0.5);
      cr.line_to (alloc.x + alloc.width, alloc.height - 0.5);
      cr.stroke ();

      propagate_draw (align, cr);

      return true;
    }

    public void reset ()
    {
      hide ();
    }

    [Signal (action=true)]
    public virtual signal void reply (string mid, string account, string sender) 
    {
    }

    private void update_time_area_wrapper ()
    {
      if (_update_time_area_id == 0)
      {
        _update_time_area_id = Timeout.add (200, () => {
          update_time_area ();
          return false;
        });
      }
    }

    private void update_time_area ()
    {
      int x, y;
      get_pointer (out x, out y);
      Gtk.Allocation a;
      get_allocation (out a);
      if (x > 0 && x < a.width && y > 0 && y < a.height)
      {
        time.hide ();
        action_box.set_no_show_all (false);
        action_box.show_all ();
      }
      else
      {
        var str = "<span font_weight='light' font_size='small' color='#999'>";
        str += time_string;
        str += "</span>";
        time.set_markup (str);
        action_box.hide ();
        time.show_all ();
      }
      _update_time_area_id = 0;
    }

    public void set_details (string[] _accounts,
                             string _stream,
                             string _sender,
                             string _sender_nick,
                             bool _from_me,
                             string _timestamp,
                             string _message,
                             string _html,
                             string _icon_uri,
                             string _url,
                             string _source,
                             string _timestring,
                             string _reply_nick,
                             string _reply_name,
                             string _reply_url,
                             double _likes,
                             string _retweet_nick,
                             string _retweet_name,
                             string _retweet_id,
                             string _link_picture,
                             string _link_name,
                             string _link_url,
                             string _link_desc,
                             string _link_caption,
                             string _link_icon,
                             string _img_url,
                             string _img_src,
                             string _img_thumb,
                             string _img_name,
                             string _video_picture,
                             string _video_src,
                             string _video_url,
                             string _video_name,
                             string _comments)
    {
      // hide and reset values to prevent reuse as the tiles change
      icon.hide ();
      action_box.hide ();
      action_box.set_no_show_all (true);
      comments_box.set_no_show_all (true);
      comments_box.hide ();
      thumb_box.set_no_show_all (true);
      thumb_box.hide ();
      thumbnail.hide();
      message.set_markup ("");
      name.set_markup ("");
      icon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
      icon.show ();
      img_uri = null;
      img_src = null;
      // end hide and reset

      /* Cancel all pending async operations */
      if (img_id > 0)
        Source.remove(img_id);
      img_id = 0;
      img_uri = _img_src;
      if (_stream == "images")
      {
        if (_img_thumb != null && _img_thumb.length > 0)
          img_uri = _img_thumb;
        img_src = _img_url;
      }
      else if (_stream == "links")
      {
        img_uri = _link_picture;
        img_src = _link_url;
      }
      else if (_stream == "videos")
      {
        img_uri = _video_picture;
        img_src = _video_src;
      }

      if (_stream == "private")
        private.show ();
      else
        private.hide ();

      foreach (uint _to_disconnect_id in _to_disconnect)
      {
        if (_to_disconnect_id > 0)
          GLib.Source.remove (_to_disconnect_id);
        _to_disconnect.remove (_to_disconnect_id);
      }

      foreach (var w in comments_box.get_children ())
        if (w is Gtk.Widget)
          w.destroy ();

      if (_comments != null)
      {
        //comments_box.set_no_show_all (false);
        var parser = new Json.Parser();
        parser.load_from_data(_comments, -1);
        unowned Json.Node comments_node = null;
        comments_node = parser.get_root();
        if (comments_node != null)
        {
          Json.Object comments_obj = comments_node.get_object ();
          if (comments_obj != null)
          {
            if (comments_obj.has_member ("comments"))
            {
              Json.Array comments = comments_obj.get_array_member ("comments");
              for(int i = 0; i < comments.get_length(); i++) {
                var obj = comments.get_element(i).get_object();
                if (obj != null)
                {
                  var cvbox = new Gtk.VBox (false, 2);
                  var chbox = new Gtk.HBox (false, 2);
                  var clalignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
                  chbox.pack_start (clalignment, false, false, 0);
  
                  var cicon = new StreamViewAvatar();
                  clalignment.add (cicon);
  
                  comments_box.pack_start (chbox, false, false, 2);
                  chbox.pack_start (cvbox, false, false, 2);
                  if (obj.has_member ("text"))
                  {
                    var ctext = obj.get_string_member ("text");
                    var ctext_label = new Gtk.Label ("");
                    ctext_label.set_selectable (true);
                    ctext_label.set_single_line_mode (false);
                    ctext_label.set_line_wrap (true);
                    ctext_label.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
                    ctext_label.set_alignment (0.0f, 0.0f);
                    ctext_label.set_markup ("<span font_size='small'>" + ctext + "</span>");
                    cvbox.pack_end (ctext_label, false, false, 2);
                  }
                  if (obj.has_member ("sender"))
                  {
                    var _sender_obj = obj.get_object_member ("sender");
                    if (_sender_obj != null)
                    {
                      if (_sender_obj.has_member ("name"))
                      {
                        var cname = _sender_obj.get_string_member ("name");
                        var cname_box = new Gtk.HBox (false, 2);
                        var cname_label = new Gtk.Label ("");
                        cname_label.set_markup ("<b><span font_size='small'>" + cname + "</span></b>");
                        cname_box.pack_start (cname_label, false, false, 2);
                        cvbox.pack_end (cname_box, false, false, 2);
                      }  
                      if (_sender_obj.has_member ("image"))
                      {
                        var cimage = _sender_obj.get_string_member ("image");
                        var cached_cicon = utils.avatar_path(cimage);
                        if (cached_cicon != null)
                          cicon.set_from_file(cached_cicon);
                        else
                        {
                          cicon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
                          cicon.show ();
                          uint cid = Idle.add (() => {
                            load_avatar_async.begin (cimage, cicon);
                            return false;
                          });
                          _to_disconnect.append (cid);
                        }
                      }
                    }
                  }
                  comments_box.set_no_show_all (false);
                  comments_box.show_all ();
                }
              }
            }
          }
        }
      }


      string display_name = _sender;

      if (show_fullname && _sender.length > 0)
        display_name = _sender;
      else if (_sender_nick.length > 0)
        display_name = _sender_nick;

      name.set_markup ("<b>" + display_name + "</b>");

      sender = _sender_nick.length > 0 ? "@" + _sender_nick : "";
      /* FIXME: we need access to settings for this"
      if (settings_map["append_colon"])
        sender += ":";
      */

      /* iterate over the accounts, display the service icon and attach a menu 
       * for actions.
       * this is a temporary hack until njpatel finishes the proper actions widget
       */
      string[] icon_displayed = null;

      /* FIXME: _accounts is made up of account:service:mid
       * we need to allow actions depending on each account/service/mid
       */

      foreach (var _p in action_box.get_children ())
      {
        GwibberGtk.ActionBoxItem _i = _p as GwibberGtk.ActionBoxItem;
        foreach (var _w in _i.get_children ())
        {
          if (_w is Gtk.Menu?)
          {
            Gtk.Menu _y = _w as Gtk.Menu;
            foreach (var _x in _y.get_children ())
            {
              _y.remove (_x);
              _x.destroy ();
              _x = null;
            }
          }
          _i.remove (_w);
          _w.destroy ();
          _w = null;
        }
        action_box.remove (_i);
        _i.destroy ();
        _i.dispose ();
        _i = null;
      }

      foreach (var _a in _accounts)
      {
        string _account;
        string _service;
        string _mid;
        (_account, _service, _mid) = _a.split(":");

        if (_account in icon_displayed)
          continue;
        icon_displayed += _account;
        var action_item = new GwibberGtk.ActionBoxItem (_service, _stream, _account, _mid, sender);
        action_box.add(action_item);
        action_item.reply.connect((mid, account, sender) => {
          reply (mid, account, sender);
        });

      }

      if (_from_me)
        icon_box.reparent(ralignment);
      else
        icon_box.reparent(lalignment);

      /* FIXME: We need to make this perform better before enabling it
      var _avatar_cache_image = utils.avatar_path(_icon_uri);
      if (_avatar_cache_image != null)
        icon.set_from_file(_avatar_cache_image);
      else
        icon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
      */
 

      if (_cache_avatar_id > 0)
        GLib.Source.remove(_cache_avatar_id);

      var _avatar_cache_image = utils.avatar_path(_icon_uri);
      if (_avatar_cache_image != null)
      {
        icon.set_from_file(_avatar_cache_image);
      } else
      {
        icon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
        _cache_avatar_id = Idle.add(() => {
          load_avatar_async.begin (_icon_uri, icon);
          return false;
        });
      }

      if (_stream == "user")
      {
        icon_box.set_no_show_all (true);
        icon_box.hide ();
      } else
      {
        icon_box.set_no_show_all (false);
        icon_box.show ();
      }

      url = _url;
      
      icon_box.button_press_event.disconnect(on_avatar_click);
      icon_box.button_press_event.connect(on_avatar_click);

      if ((int)_likes == 0) {
        likes_hbox.hide();
        likes_hbox.set_no_show_all (true);
      } else if ((int)_likes < 0) {
        likes_count.set_markup("");
        likes_hbox.set_no_show_all (false);
        likes_hbox.show_all();
      } else if ((int)_likes == 1) {
        likes_count.set_markup("<span font_size='small'>" + _likes.to_string() + " " + _("person liked this") + "</span>");
        likes_hbox.set_no_show_all (false);
        likes_hbox.show_all();
      } else if ((int)_likes > 1) {
        likes_count.set_markup("<span font_size='small'>" + _likes.to_string() + " " + _("people liked this") + "</span>");
        likes_hbox.set_no_show_all (false);
        likes_hbox.show_all();
      }

      message.set_markup (_html);

      string link_str = null;
      if (_stream == "links")
      {
        link_str = "<a href='" + _link_url + "'>" + _link_name + "</a>\n";
        link_str += _link_desc;
        message.set_markup (link_str);
      }

      string video_str = null;
      if (_stream == "videos")
      {
        video_str = "<a href='" + _video_url + "'>" + _video_name + "</a>\n";
        video_str += _html;
        message.set_markup (video_str);
      }

      if (img_uri.length > 0)
      {
        /* Let's wait a while before loading in case there is some funny
         * business going on or the user is scrolling
         */
        string img_str = null;
        img_id = Timeout.add (200, () =>
        {

          if (_stream == "videos" && video_str.length > 0)
            img_str = video_str;
          else if (_stream == "links" && link_str.length > 0)
            img_str = link_str;
          else
          {
            if (_img_name.length > 0)
              img_str = "<a href='" + _img_url + "'>" + _img_name + "</a>\n";
            else
              img_str = "<a href='" + _img_url + "'>" + _img_url + "</a>\n";
            img_str += _html;
          }

          load_thumbnail_async.begin(img_str);
          return false;
        });
      }

      time_string = utils.generate_time_string ((uint)long.parse(_timestamp));

      update_time_area_wrapper ();

      reply_to.hide ();
      string reply_to_string =  "<span font_weight='light' font_size='small' color='#999'>";
      if (show_fullname && (_reply_name != null && _reply_name.length > 0))
      {
        reply_to_string += " " + _("in reply to") + " <a href='" + _reply_url + "'>" + _reply_name + "</a>";
        reply_to.show ();
      }
      else if (_reply_nick != null && _reply_nick.length > 0)
      {
        reply_to_string += " " + _("in reply to") + " <a href='" + _reply_url + "'>" + _reply_nick + "</a>";
        reply_to.show ();
      }
      reply_to_string  += "</span>";
      reply_to.set_markup (reply_to_string);

      retweeted_by.hide ();
      string retweeted_by_string =  "<span font_weight='light' font_size='small' color='#999'>";
      if (show_fullname && (_retweet_name != null && _retweet_name.length > 0))
      {
        retweeted_by_string += " " + _("shared by") + " " + _retweet_name;
        retweeted_by.show ();
      }
      else if (_retweet_nick != null && _retweet_nick.length > 0)
      {
        retweeted_by_string += " " + _("shared by") + " " + _retweet_nick;
        retweeted_by.show ();
      }
      retweeted_by_string  += "</span>";
     
      retweeted_by.set_markup (retweeted_by_string);

      queue_resize ();
      show ();
    } 

    private async void load_avatar_async(string url, Gtk.Image cicon)
    {
      string t;
      var file = File.new_for_uri(url);
      try {
        var stream = yield file.read_async(Priority.DEFAULT, cancellable);
        Gdk.Pixbuf pixbuf = null;
        pixbuf = new Gdk.Pixbuf.from_stream_at_scale (stream,
                                                      48,
                                                      48,
                                                      false,
                                                      cancellable);
        if (pixbuf is Gdk.Pixbuf)
        {
          cicon.set_from_pixbuf (pixbuf);
          cicon.show();
          var cimage = Path.build_path(Path.DIR_SEPARATOR_S, Environment.get_user_cache_dir(), "gwibber/avatars", url.replace("/",""));
          t = "png";
          if (cimage.has_suffix ("JPG") || cimage.has_suffix ("jpeg") || cimage.has_suffix ("jpg"))
            t = "jpeg";
          pixbuf.save (cimage, t);
        }
      } catch (Error e) {
        debug("Error trying to load %s: %s", url, e.message);
        cicon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
      }
    }

    private async void load_thumbnail_async(string img_markup)
    {
      Gtk.Allocation alloc;
      message.get_allocation (out alloc);

      img_id = 0;
      var file = File.new_for_uri(fix_image_uri(img_uri));
      try {
        var stream = yield file.read_async(Priority.DEFAULT, cancellable);
        Gdk.Pixbuf pixbuf = null;
        pixbuf = new Gdk.Pixbuf.from_stream_at_scale (stream,
                                                        alloc.width - 24,
                                                        100,
                                                        true,
                                                        cancellable);
        if (pixbuf is Gdk.Pixbuf)
        {
          thumbnail.set_from_pixbuf (pixbuf);
          thumbnail.show ();
          thumb_box.set_no_show_all (false);
          thumb_box.show ();
        }
        message.set_markup (img_markup);
      } catch (Error e) {
        debug("Error trying to load %s: %s", img_uri, e.message);
        thumbnail.hide ();
        thumb_box.set_no_show_all (true);
        thumb_box.hide ();
      }
     }

    /* Do any magic we want to here */
    private string fix_image_uri(string uri)
    {
      var ret = uri;

      if (uri.contains("imgur.com") && !uri.has_suffix(".png"))
      {
        /* let's point to the static content in imgur */
        var last = uri.rstr("/").substring(1);
        ret = "http://i.imgur.com/%s.png".printf(last); 
      }
      else if (uri.contains("youtube.com"))
      {
        string id = uri.rstr("/").substring(1);

        if (uri.contains("watch?v="))
        {
          if (uri.contains("&"))
            id = uri.split("v=")[1].split("&")[0];
          else
            id = uri.split("v=")[1];
        }
        ret = "http://img.youtube.com/vi/%s/default.jpg".printf(id);
      }
      return ret;
    }

    private bool on_avatar_click (Gdk.EventButton button)
    {
        /* FIXME: Display a user view with user info and posts instead of 
        launching external browser */
        try {
          AppInfo.launch_default_for_uri (url, null);
        } catch {
          warning ("Unable to launch: " + url);
        }
        return false;
    }
  }
}
