/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

using Dee;
using Gee;
using Config;
using Unity;
using Gwibber;

namespace UnityGwibber {

  /* DBus name for the place. Must match out .place file */
  const string BUS_NAME = "com.canonical.Unity.Lens.Gwibber";

  /**
   * The Daemon class implements all of the logic for the place.
   *
   */
  public class Daemon : GLib.Object
  {
    private Unity.Lens lens;
    private Unity.Scope scope;
    private Gwibber.Streams streams_service;
    private Gwibber.Service service;
    private Gwibber.Utils utils;
    private Dee.Model? _model;
    private Dee.Model? _streams_model;
    private Dee.Filter *_sort_filter;
    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private LensSearch? previous_search;
    private bool is_dirty;
    private unowned Dee.ModelIter _stream_iter_first = null;
    private unowned Dee.ModelIter _stream_iter_last = null;


    construct
    {
      streams_service = new Gwibber.Streams();
      service = new Gwibber.Service();
      utils = new Gwibber.Utils();

      scope = new Unity.Scope ("/com/canonical/unity/scope/gwibber");
      scope.search_in_global = true;

      lens = new Unity.Lens("/com/canonical/unity/lens/gwibber", "gwibber");
      lens.search_in_global = true;
      lens.search_hint = _("Enter name or content you would like to search for");
      lens.visible = true;
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);

      /* Listen for filter changes */
      scope.filters_changed.connect (() => {
        if (scope.active_search != null)
        {
          is_dirty = true;
          scope.notify_property ("active-search");
        }
      });

      lens.notify["active"].connect ((obj, pspec) => {
        if (lens.active && scope.active_search != null)
        {
          if (_stream_iter_first != _model.get_first_iter () || _stream_iter_last != _model.get_last_iter ())
          {
            is_dirty = true;
            scope.notify_property ("active-search");
          }
        }
      });

      /* Listen for changes to the lens entry search */
      scope.notify["active-search"].connect ((obj, pspec) => {
        var search = scope.active_search;
        is_dirty = false;
        update_search (search);
        previous_search = search;
      });
     
      scope.notify["active-global-search"].connect ((obj, pspec) => {
        var search = scope.active_global_search;
        if (search_is_invalid (search))
          return;
        update_global_search (search);
        previous_search = search;
      });

      try
      {
        lens.export ();
      } catch (GLib.IOError e)
      {
        warning ("failed to export lens: %s", e.message);
      }

      _streams_model = streams_service.stream_model;
      Intl.setlocale(LocaleCategory.COLLATE, "C");
      _sort_filter = new Dee.Filter.collator_desc(StreamModelColumn.TIMESTAMP);
      _model = new Dee.FilterModel (_sort_filter, _streams_model);
    }

    private void populate_filters ()
    {
      var filters = new GLib.List<Unity.Filter> ();

      /* Stream filter */
      {
        var filter = new RadioOptionFilter ("stream", _("Stream"));

        filter.add_option ("messages", _("Messages"));
        filter.add_option ("replies", _("Replies"));
        filter.add_option ("images", _("Images"));
        filter.add_option ("videos", _("Videos"));
        filter.add_option ("links", _("Links"));
        filter.add_option ("private", _("Private"));
        filter.add_option ("public", _("Public"));

        filters.append (filter);
      }

      lens.filters = filters;
    }

    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      Icon icon;

      icon = new ThemedIcon ("tray-message");
      var cat = new Unity.Category (_("Messages"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("mail_reply");
      cat =  new Unity.Category (_("Replies"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("gnome-mime-image");
      cat =  new Unity.Category (_("Images"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("video");
      cat =  new Unity.Category (_("Videos"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("web-browser");
      cat =  new Unity.Category (_("Links"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("mail-send-receive");
      cat =  new Unity.Category (_("Private"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new ThemedIcon ("evolution");
      cat =  new Unity.Category (_("Public"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      lens.categories = categories;
    }


    private bool search_is_invalid (LensSearch? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.search_string == null)
        return true;

      return search.search_string.strip() == "";
    }

    private string prepare_search_string (LensSearch? search)
    {
      var s = search.search_string;

      if (s.has_suffix (" "))
        s = s.strip ();

      return s;
    }


    private void update_global_search (LensSearch search)
    {
      var results_model = scope.global_results_model;

      if (search_is_invalid (search))
        {
          return;
        }

      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();

      var search_string = prepare_search_string (search);

      /* FIXME
      try {
      */
        results_model.clear ();
        update_results_model (results_model, null,
                            search_string, null);
      /* FIXME
      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
      }
      */

      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });

      search.finished ();
    }

    private void update_search  (LensSearch search)
    {
      var results_model = scope.results_model;

      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();

      var search_string = prepare_search_string (search);

      string type_id = get_current_type ();
      debug ("type_id is %s", type_id);

      /* FIXME
      try {
      */
        results_model.clear ();
        update_results_model (results_model, type_id,
                            search_string, null);
      /* FIXME
      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
      }
      */

      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });

      search.finished ();
    }

    private string get_current_type ()
    {
      /* Get the current type to filter by */
      var filter = scope.get_filter ("stream") as RadioOptionFilter;
      Unity.FilterOption? option =  filter.get_active_option ();
      return option == null ? "all" : option.id;
    }


    /* Generic method to update a results model. We do it like this to minimize
     * code dup between updating the global- and the entry results model */
    private void update_results_model (Dee.Model results_model, string? type_id,
                                       string? search, Categories? category)
    {
      unowned Dee.ModelIter iter, end;
      string search_string = null;
      Categories group = Categories.MESSAGES;

      results_model.clear ();

      iter = _model.get_first_iter ();
      end = _model.get_last_iter ();

      _stream_iter_first = _model.get_first_iter ();
      _stream_iter_last = end;
      
      if (search != null)
      { 
          search_string = search.down();
      }

      while (iter != end)
      {
        if (type_id == "messages" && _model.get_string(iter, StreamModelColumn.STREAM) != "messages")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "replies" && _model.get_string(iter, StreamModelColumn.STREAM) != "replies")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "images" && _model.get_string(iter, StreamModelColumn.STREAM) != "images")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "videos" && _model.get_string(iter, StreamModelColumn.STREAM) != "videos")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "links" && _model.get_string(iter, StreamModelColumn.STREAM) != "links")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "private" && _model.get_string(iter, StreamModelColumn.STREAM) != "private")
        {
          iter = _model.next (iter);
          continue;
        }

        if (type_id == "public" && _model.get_string(iter, StreamModelColumn.STREAM) != "public")
        {
          iter = _model.next (iter);
          continue;
        }

        if (search_string == null || search_string in _model.get_string(iter, StreamModelColumn.SENDER).down() || search_string in _model.get_string(iter, StreamModelColumn.MESSAGE).down())
          {
          if (_model.get_string(iter, StreamModelColumn.STREAM) == "messages")
            group = Categories.MESSAGES;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "replies")
            group = Categories.REPLIES;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "images")
            group = Categories.IMAGES;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "videos")
            group = Categories.VIDEOS;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "links")
            group = Categories.LINKS;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "private")
            group = Categories.PRIVATE;
          else if (_model.get_string(iter, StreamModelColumn.STREAM) == "public")
            group = Categories.PUBLIC;
          
          
          string _icon_uri = _model.get_string(iter, StreamModelColumn.ICON_URI);
          var _avatar_cache_image = utils.avatar_path(_icon_uri);
          if (_avatar_cache_image == null)
          {
            try
            {
              _avatar_cache_image = service.avatar_path (_icon_uri);
            } catch (GLib.Error e)
            {
            }
            if (_avatar_cache_image == null)
              _avatar_cache_image = _icon_uri;
          }

          results_model.append(_model.get_string(iter, StreamModelColumn.URL), _avatar_cache_image, group, "text/html", _model.get_string(iter, StreamModelColumn.SENDER), _model.get_string(iter, StreamModelColumn.MESSAGE));
          }
        if (iter == end)
          iter = null;
        else
          iter = _model.next (iter);
      }
     
      //debug ("Results has %u rows", results_model.get_n_rows());
    }

  } /* End Daemon class */
} /* end Gwibber namespace */
