/* $Id: gvrender_core_vml.c,v 1.10 2007/10/04 10:38:14 ellson Exp $ $Revision: 1.10 $ */
/* vim:set shiftwidth=4 ts=8: */

/**********************************************************
*      This software is part of the graphviz package      *
*                http://www.graphviz.org/                 *
*                                                         *
*            Copyright (c) 1994-2004 AT&T Corp.           *
*                and is licensed under the                *
*            Common Public License, Version 1.0           *
*                      by AT&T Corp.                      *
*                                                         *
*        Information and Software Systems Research        *
*              AT&T Research, Florham Park NJ             *
**********************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "macros.h"
#include "const.h"

#include "gvplugin_render.h"
#include "graph.h"

typedef enum { FORMAT_VML, FORMAT_VMLZ, } format_type;

extern char *xml_string(char *str);

char graphcoords[256];

#ifdef WIN32
static int
snprintf (char *str, int n, char *fmt, ...)
{
int ret;
va_list a;
va_start (a, fmt);
ret = _vsnprintf (str, n, fmt, a);
va_end (a);
return ret;
}
#endif

static void vml_bzptarray(GVJ_t * job, pointf * A, int n)
{
    int i;
    char *c;

    c = "m ";			/* first point */
    for (i = 0; i < n; i++) {
	gvdevice_printf(job, "%s%.0f,%.0f ", c, A[i].x, -A[i].y);
	if (i == 0)
	    c = "c ";		/* second point */
	else
	    c = "";		/* remaining points */
    }
}

static void vml_print_color(GVJ_t * job, gvcolor_t color)
{
    switch (color.type) {
    case COLOR_STRING:
	gvdevice_fputs(job, color.u.string);
	break;
    case RGBA_BYTE:
	if (color.u.rgba[3] == 0) /* transparent */
	    gvdevice_fputs(job, "none");
	else
	    gvdevice_printf(job, "#%02x%02x%02x",
		color.u.rgba[0], color.u.rgba[1], color.u.rgba[2]);
	break;
    default:
	assert(0);		/* internal error */
    }
}

static void vml_grstroke(GVJ_t * job, int filled)
{
    obj_state_t *obj = job->obj;

    gvdevice_fputs(job, "<v:stroke fillcolor=\"");
    if (filled)
	vml_print_color(job, obj->fillcolor);
    else
	gvdevice_fputs(job, "none");
    gvdevice_fputs(job, "\" strokecolor=\"");
    vml_print_color(job, obj->pencolor);
    if (obj->penwidth != PENWIDTH_NORMAL)
	gvdevice_printf(job, "\" stroke-weight=\"%g", obj->penwidth);
    if (obj->pen == PEN_DASHED) {
	gvdevice_fputs(job, "\" dashstyle=\"dash");
    } else if (obj->pen == PEN_DOTTED) {
	gvdevice_fputs(job, "\" dashstyle=\"dot");
    }
    gvdevice_fputs(job, "\" />");
}

static void vml_grstrokeattr(GVJ_t * job)
{
    obj_state_t *obj = job->obj;

    gvdevice_fputs(job, " strokecolor=\"");
    vml_print_color(job, obj->pencolor);
    if (obj->penwidth != PENWIDTH_NORMAL)
	gvdevice_printf(job, "\" stroke-weight=\"%g", obj->penwidth);
    if (obj->pen == PEN_DASHED) {
	gvdevice_fputs(job, "\" dashstyle=\"dash");
    } else if (obj->pen == PEN_DOTTED) {
	gvdevice_fputs(job, "\" dashstyle=\"dot");
    }
    gvdevice_fputs(job, "\"");
}

static void vml_grfill(GVJ_t * job, int filled)
{
    obj_state_t *obj = job->obj;

    gvdevice_fputs(job, "<v:fill color=\"");
    if (filled)
	vml_print_color(job, obj->fillcolor);
    else
	gvdevice_fputs(job, "none");
    gvdevice_fputs(job, "\" />");
}

static void vml_comment(GVJ_t * job, char *str)
{
    gvdevice_fputs(job, "      <!-- ");
    gvdevice_fputs(job, xml_string(str));
    gvdevice_fputs(job, " -->\n");
}

static void vml_begin_job(GVJ_t * job)
{
    gvdevice_fputs(job, "<?xml version=\"1.1\" encoding=\"UTF-8\" ?>\n");

    gvdevice_fputs(job, "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" ");
    gvdevice_fputs(job, "\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n");
    gvdevice_fputs(job, "<html xml:lang=\"en\" xmlns=\"http://www.w3.org/1999/xhtml\" ");
    gvdevice_fputs(job, "xmlns:v=\"urn:schemas-microsoft-com:vml\""); 
    gvdevice_fputs(job, ">"); 

    gvdevice_fputs(job, "\n<!-- Generated by ");
    gvdevice_fputs(job, xml_string(job->common->info[0]));
    gvdevice_fputs(job, " version ");
    gvdevice_fputs(job, xml_string(job->common->info[1]));
    gvdevice_fputs(job, " (");
    gvdevice_fputs(job, xml_string(job->common->info[2]));
    gvdevice_fputs(job, ")\n     For user: ");
    gvdevice_fputs(job, xml_string(job->common->user));
    gvdevice_fputs(job, " -->\n");
}

static void vml_begin_graph(GVJ_t * job)
{
    obj_state_t *obj = job->obj;

    gvdevice_fputs(job, "<head>");
    if (obj->u.g->name[0]) {
        gvdevice_fputs(job, "<title>");
	gvdevice_fputs(job, xml_string(obj->u.g->name));
        gvdevice_fputs(job, "</title>");
    }
    gvdevice_printf(job, "<!-- Pages: %d -->\n</head>\n", job->pagesArraySize.x * job->pagesArraySize.y);

    snprintf(graphcoords, sizeof(graphcoords), "style=\"width: %.0fpt; height: %.0fpt\" coordsize=\"%.0f,%.0f\" coordorigin=\"-4,-%.0f\"",
	job->width*.75, job->height*.75,
        job->width*.75, job->height*.75,
        job->height*.75 - 4);

    gvdevice_printf(job, "<body>\n<div class=\"graph\" %s>\n", graphcoords);
    gvdevice_fputs(job, "<style type=\"text/css\">\nv\\:* {\nbehavior: url(#default#VML);display:inline-block;position: absolute; left: 0px; top: 0px;\n}\n</style>\n");
/*    graphcoords[0] = '\0'; */

}

static void vml_end_graph(GVJ_t * job)
{
    gvdevice_fputs(job, "</div>\n</body>\n");
}

static void
vml_begin_anchor(GVJ_t * job, char *href, char *tooltip, char *target)
{
    gvdevice_fputs(job, "      <a");
    if (href && href[0])
	gvdevice_printf(job, " href=\"%s\"", xml_string(href));
    if (tooltip && tooltip[0])
	gvdevice_printf(job, " title=\"%s\"", xml_string(tooltip));
    if (target && target[0])
	gvdevice_printf(job, " target=\"%s\"", xml_string(target));
    gvdevice_fputs(job, ">\n");
}

static void vml_end_anchor(GVJ_t * job)
{
    gvdevice_fputs(job, "      </a>\n");
}

static void vml_textpara(GVJ_t * job, pointf p, textpara_t * para)
{
    obj_state_t *obj = job->obj;

    gvdevice_fputs(job, "        <div");
    switch (para->just) {
    case 'l':
	gvdevice_fputs(job, " style=\"text-align: left; ");
	break;
    case 'r':
	gvdevice_fputs(job, " style=\"text-align: right; ");
	break;
    default:
    case 'n':
	gvdevice_fputs(job, " style=\"text-align: center; ");
	break;
    }
    gvdevice_printf(job, "position: absolute; left: %gpx; top: %gpx;", p.x/.75, job->height - p.y/.75 - 14);
    if (para->postscript_alias) {
        gvdevice_printf(job, " font-family: '%s';", para->postscript_alias->family);
        if (para->postscript_alias->weight)
	    gvdevice_printf(job, " font-weight: %s;", para->postscript_alias->weight);
        if (para->postscript_alias->stretch)
	    gvdevice_printf(job, " font-stretch: %s;", para->postscript_alias->stretch);
        if (para->postscript_alias->style)
	    gvdevice_printf(job, " font-style: %s;", para->postscript_alias->style);
    }
    else {
        gvdevice_printf(job, " font-family: \'%s\';", para->fontname);
    }
    /* FIXME - even inkscape requires a magic correction to fontsize.  Why?  */
    gvdevice_printf(job, " font-size: %.2fpt;", para->fontsize * 0.81);
    switch (obj->pencolor.type) {
    case COLOR_STRING:
	if (strcasecmp(obj->pencolor.u.string, "black"))
	    gvdevice_printf(job, "color:%s;", obj->pencolor.u.string);
	break;
    case RGBA_BYTE:
	gvdevice_printf(job, "color:#%02x%02x%02x;",
		obj->pencolor.u.rgba[0], obj->pencolor.u.rgba[1], obj->pencolor.u.rgba[2]);
	break;
    default:
	assert(0);		/* internal error */
    }
    gvdevice_fputs(job, "\">");
    gvdevice_fputs(job, xml_string(para->str));
    gvdevice_fputs(job, "</div>\n");
}

static void vml_ellipse(GVJ_t * job, pointf * A, int filled)
{
    /* A[] contains 2 points: the center and corner. */
    
    gvdevice_fputs(job, "        <v:oval");

    vml_grstrokeattr(job);

    gvdevice_fputs(job, " style=\"position: absolute;");

    gvdevice_printf(job, " left:  %gpt; top:    %gpt;", 2*A[0].x - A[1].x+4, job->height*.75 - A[1].y-4);
    gvdevice_printf(job, " width: %gpt; height: %gpt;", 2*(A[1].x - A[0].x), 2*(A[1].y - A[0].y));
    gvdevice_fputs(job, "\">");
    vml_grstroke(job, filled);
    vml_grfill(job, filled);
    gvdevice_fputs(job, "</v:oval>\n");
}

static void
vml_bezier(GVJ_t * job, pointf * A, int n, int arrow_at_start,
	      int arrow_at_end, int filled)
{
    gvdevice_printf(job, "        <v:shape %s><!-- bezier --><v:path", graphcoords);
    gvdevice_fputs(job, " v=\"");
    vml_bzptarray(job, A, n);
    gvdevice_fputs(job, "\" />");
    vml_grstroke(job, filled);
    gvdevice_fputs(job, "</v:path>");
    vml_grfill(job, filled);
    gvdevice_fputs(job, "</v:shape>\n");
}

static void vml_polygon(GVJ_t * job, pointf * A, int n, int filled)
{
    int i;

    gvdevice_fputs(job, "        <v:shape");
    vml_grstrokeattr(job);
    gvdevice_printf(job, " %s><!-- polygon --><v:path", graphcoords);
    gvdevice_fputs(job, " v=\"");
    for (i = 0; i < n; i++)
    {
        if (i==0) gvdevice_fputs(job, "m ");
	gvdevice_printf(job, "%.0f,%.0f ", A[i].x, -A[i].y);
        if (i==0) gvdevice_fputs(job, "l ");
        if (i==n-1) gvdevice_fputs(job, "x e ");
    }
    gvdevice_fputs(job, "\">");
    vml_grstroke(job, filled);
    gvdevice_fputs(job, "</v:path>");
    vml_grfill(job, filled);
    gvdevice_fputs(job, "</v:shape>\n");
}

static void vml_polyline(GVJ_t * job, pointf * A, int n)
{
    int i;

    gvdevice_printf(job, "        <v:shape %s><!-- polyline --><v:path", graphcoords);
    gvdevice_fputs(job, " v=\"");
    for (i = 0; i < n; i++)
    {
        if (i==0) gvdevice_fputs(job, " m ");
	gvdevice_printf(job, "%.0f,%.0f ", A[i].x, -A[i].y);
        if (i==0) gvdevice_fputs(job, " l ");
        if (i==n-1) gvdevice_fputs(job, " e "); /* no x here for polyline */
    }
    gvdevice_fputs(job, "\">");
    vml_grstroke(job, 0);                 /* no fill here for polyline */
    gvdevice_fputs(job, "</v:path>");
    gvdevice_fputs(job, "</v:shape>\n");

}

/* color names from http://www.w3.org/TR/VML/types.html */
/* NB.  List must be LANG_C sorted */
static char *vml_knowncolors[] = {
    "aliceblue", "antiquewhite", "aqua", "aquamarine", "azure",
    "beige", "bisque", "black", "blanchedalmond", "blue",
    "blueviolet", "brown", "burlywood",
    "cadetblue", "chartreuse", "chocolate", "coral",
    "cornflowerblue", "cornsilk", "crimson", "cyan",
    "darkblue", "darkcyan", "darkgoldenrod", "darkgray",
    "darkgreen", "darkgrey", "darkkhaki", "darkmagenta",
    "darkolivegreen", "darkorange", "darkorchid", "darkred",
    "darksalmon", "darkseagreen", "darkslateblue", "darkslategray",
    "darkslategrey", "darkturquoise", "darkviolet", "deeppink",
    "deepskyblue", "dimgray", "dimgrey", "dodgerblue",
    "firebrick", "floralwhite", "forestgreen", "fuchsia",
    "gainsboro", "ghostwhite", "gold", "goldenrod", "gray",
    "green", "greenyellow", "grey",
    "honeydew", "hotpink", "indianred",
    "indigo", "ivory", "khaki",
    "lavender", "lavenderblush", "lawngreen", "lemonchiffon",
    "lightblue", "lightcoral", "lightcyan", "lightgoldenrodyellow",
    "lightgray", "lightgreen", "lightgrey", "lightpink",
    "lightsalmon", "lightseagreen", "lightskyblue",
    "lightslategray", "lightslategrey", "lightsteelblue",
    "lightyellow", "lime", "limegreen", "linen",
    "magenta", "maroon", "mediumaquamarine", "mediumblue",
    "mediumorchid", "mediumpurple", "mediumseagreen",
    "mediumslateblue", "mediumspringgreen", "mediumturquoise",
    "mediumvioletred", "midnightblue", "mintcream",
    "mistyrose", "moccasin",
    "navajowhite", "navy", "oldlace",
    "olive", "olivedrab", "orange", "orangered", "orchid",
    "palegoldenrod", "palegreen", "paleturquoise",
    "palevioletred", "papayawhip", "peachpuff", "peru", "pink",
    "plum", "powderblue", "purple",
    "red", "rosybrown", "royalblue",
    "saddlebrown", "salmon", "sandybrown", "seagreen", "seashell",
    "sienna", "silver", "skyblue", "slateblue", "slategray",
    "slategrey", "snow", "springgreen", "steelblue",
    "tan", "teal", "thistle", "tomato", "turquoise",
    "violet",
    "wheat", "white", "whitesmoke",
    "yellow", "yellowgreen"
};

gvrender_engine_t vml_engine = {
    vml_begin_job,
    0,				/* vml_end_job */
    vml_begin_graph,
    vml_end_graph,
    0,                          /* vml_begin_layer */
    0,                          /* vml_end_layer */
    0,                          /* vml_begin_page */
    0,                          /* vml_end_page */
    0,                          /* vml_begin_cluster */
    0,                          /* vml_end_cluster */
    0,				/* vml_begin_nodes */
    0,				/* vml_end_nodes */
    0,				/* vml_begin_edges */
    0,				/* vml_end_edges */
    0,                          /* vml_begin_node */
    0,                          /* vml_end_node */
    0,                          /* vml_begin_edge */
    0,                          /* vml_end_edge */
    vml_begin_anchor,
    vml_end_anchor,
    vml_textpara,
    0,				/* vml_resolve_color */
    vml_ellipse,
    vml_polygon,
    vml_bezier,
    vml_polyline,
    vml_comment,
    0,				/* vml_library_shape */
};

gvrender_features_t render_features_vml = {
    GVRENDER_Y_GOES_DOWN
        | GVRENDER_DOES_TRANSFORM
	| GVRENDER_DOES_LABELS
	| GVRENDER_DOES_MAPS
	| GVRENDER_DOES_TARGETS
	| GVRENDER_DOES_TOOLTIPS, /* flags */
    4.,                         /* default pad - graph units */
    vml_knowncolors,		/* knowncolors */
    sizeof(vml_knowncolors) / sizeof(char *),	/* sizeof knowncolors */
    RGBA_BYTE,			/* color_type */
};

gvdevice_features_t device_features_vml = {
    GVDEVICE_DOES_TRUECOLOR,	/* flags */
    {0.,0.},			/* default margin - points */
    {0.,0.},                    /* default page width, height - points */
    {96.,96.},			/* default dpi */
};

gvdevice_features_t device_features_vmlz = {
    GVDEVICE_DOES_TRUECOLOR
      | GVDEVICE_COMPRESSED_FORMAT,	/* flags */
    {0.,0.},			/* default margin - points */
    {0.,0.},                    /* default page width, height - points */
    {96.,96.},			/* default dpi */
};

gvplugin_installed_t gvrender_vml_types[] = {
    {FORMAT_VML, "vml", 1, &vml_engine, &render_features_vml},
    {0, NULL, 0, NULL, NULL}
};

gvplugin_installed_t gvdevice_vml_types[] = {
    {FORMAT_VML, "vml:vml", 1, NULL, &device_features_vml},
#if HAVE_LIBZ
    {FORMAT_VMLZ, "vmlz:vml", 1, NULL, &device_features_vmlz},
#endif
    {0, NULL, 0, NULL, NULL}
};
