/**
 * GMyth Library
 *
 * @file gmyth/gmyth_remote_util.c
 * 
 * @brief <p> This component provides utility functions for accessing remote data.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Hallyson Luiz de Morais Melo <hallyson.melo@indt.org.br>
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_remote_util.h"

#include "gmyth_recorder.h"
#include "gmyth_stringlist.h"
#include "gmyth_debug.h"

/** 
 * Requests the Mythtv backend for a free remote recorder.
 * 
 * @param socket The socket instance where to send the command.
 * @param curr The recorder index, or -1 to consider the first one.
 * 
 * @return the remote encoder instance available, or NULL if any error happens.
 */
GMythRecorder  *
remote_request_next_free_recorder(GMythSocket * socket, gint curr)
{
    GMythRecorder  *recorder = NULL;
    GString        *hostname;
    gint            num,
                    port;

    GMythStringList *strlist = gmyth_string_list_new();

    gmyth_debug("[%s] Request next free recorder in the backend",
                __FUNCTION__);

    gmyth_string_list_append_char_array(strlist, "GET_NEXT_FREE_RECORDER");
    gmyth_string_list_append_int(strlist, curr);

    if (!gmyth_socket_sendreceive_stringlist(socket, strlist)) {
        g_warning("GET_NEXT_FREE_RECORDER request error!\n");
        return NULL;
    }

    num = gmyth_string_list_get_int(strlist, 0);
    hostname = gmyth_string_list_get_string(strlist, 1);
    port = gmyth_string_list_get_int(strlist, 2);

    if (num < 0 || port < 0)
        goto clean_up;

    gmyth_debug
        ("[%s] Free recorder info received: num: %d, hostname: %s, port: %d",
         __FUNCTION__, num, hostname->str, port);

    recorder = gmyth_recorder_new(num, hostname, port);

  clean_up:

    g_string_free(hostname, TRUE);
    g_object_unref(strlist);

    return recorder;
}

/** 
 * Requests the Mythtv backend for the number of free remote recorders.
 * 
 * @param socket The socket instance where to send the command.
 * 
 * @return the number of remote encoders instance available, or 0 if no one is actually free..
 */
gint
gmyth_remote_util_get_free_recorder_count(GMythSocket * socket)
{
    gint            num_recs = 0;

    GMythStringList *strlist = gmyth_string_list_new();

    gmyth_debug("[%s] Request next free recorder in the backend",
                __FUNCTION__);

    gmyth_string_list_append_char_array(strlist,
                                        "GET_FREE_RECORDER_COUNT");

    if (!gmyth_socket_sendreceive_stringlist(socket, strlist)) {
        gmyth_debug("GET_FREE_RECORDER_COUNT request error!");
        return 0;
    }

    num_recs = gmyth_string_list_get_int(strlist, 0);

    if (num_recs < 0)
        goto clean_up;

    gmyth_debug("[%s] Free recorder info received: num recorders: %d",
                __FUNCTION__, num_recs);

  clean_up:

    g_object_unref(strlist);

    return num_recs;
}
