/**
 * GMyth Library
 *
 * @file gmyth/gmyth_jobqueue.c
 * 
 * @brief <p> Library to use JobQueue from mythbackend
 *
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia.
 * @author Artur Duque de Souza <artur.souza@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_jobqueue.h"
#include "gmyth_http.h"
#include "gmyth_debug.h"
#include "gmyth_socket.h"

/** Function to connect
 *
 * @param backend_info the backendinfo
 * @return gboolean - result of connection
 *
 */
static GMythSocket *
backend_connect(GMythBackendInfo * backend_info)
{
    GMythSocket    *socket = gmyth_socket_new();

    if (gmyth_socket_connect_to_backend(socket,
                                        gmyth_backend_info_get_hostname
                                        (backend_info),
                                        gmyth_backend_info_get_port
                                        (backend_info), TRUE) == TRUE) {
        gmyth_debug("Backend socket connection success");
        return socket;
    } else {
        gmyth_debug("Connection failed");
        return NULL;
    }
}


/** Function to send a command to the backend
 *
 * @param socket pointer to a socket
 * @param action the action itself
 * @param job the action itself
 * @param chanid the action itself
 * @param starttime the action itself
 * @param options the action itself
 * @return the value returned by the backend
 *
 */
static gchar   *
send_command(GMythSocket * socket, gchar * action,
             gchar * job, gint chanid, gchar * starttime, gchar * options)
{
    GString        *command = g_string_new("");
    GString        *ret_str;
    gchar          *ret;

    GMythStringList *retlist = gmyth_string_list_new();

    g_string_printf(command, "JOBQUEUE %s %s %d %s %s", action, job,
                    chanid, starttime, options);

    gmyth_string_list_append_string(retlist, command);
    gmyth_socket_write_stringlist(socket, retlist);

    // receive answer
    gmyth_socket_read_stringlist(socket, retlist);
    ret_str = gmyth_string_list_get_string(retlist, 0);

    // ret = ret_str->str;
    ret = g_string_free(ret_str, FALSE);
    g_string_free(command, TRUE);

    gmyth_string_list_clear_all(retlist);
    g_object_unref(retlist);

    return ret;
}


/** Function to analyze the response from the backend
 *
 * @param ret the msg returned by the backend
 * @param value the expected value
 * @return 0 if success and -1 if error
 *
 */
static          gboolean
test_result(gchar * ret, gchar * value)
{
    if (g_ascii_strcasecmp(ret, value) == 0) {
        return TRUE;
    } else {
        gmyth_debug("JobQueue Error: %s", ret);
        return FALSE;
    }
}

/** Function to add a job inside JOBQUEUE
 *
 * @param transcode object holding all the info about the transcoding
 * @param job the job you want to add the action
 * @return TRUE if the job was added, FALSE if not
 *
 */
gboolean
gmyth_jobqueue_add_job(GMythTranscoder * transcode, gchar * job)
{
    GMythSocket    *socket = backend_connect(transcode->backend_info);
    gboolean        res = FALSE;

    if (socket != NULL) {
        GString        *options = g_string_new("");
        gchar          *ret = NULL;

        if (g_ascii_strcasecmp(job, "JOB_TRANSCODE") == 0) {
            if (transcode->cutlist)
                g_string_append(options, " JOB_USE_CUTLIST");

            if (transcode->output)
                g_string_append_printf(options, " JOB_OUTPUT %s",
                                       transcode->output_filename);

            if (transcode->profile != NULL)
                g_string_append_printf(options, " %s", transcode->profile);
        }
        ret = send_command(socket, "ADD", job, transcode->chanid,
                           transcode->starttime, options->str);
        res = test_result(ret, "JOBQUEUE_OK");
        gmyth_socket_close_connection(socket);

        g_object_unref(socket);

        g_string_free(options, TRUE);

        if (ret)
            g_free(ret);

    } else {
        gmyth_debug("JobQueue Connection Failed");
    }

    return res;
}

/** Function to change a job cmd inside JOBQUEUE
 *
 * @param transcode object holding all the info about the transcoding
 * @param action the action (ADD)
 * @param job the job you want to add the action
 * @return the value of the key
 *
 */
gboolean
gmyth_jobqueue_change_cmd(GMythTranscoder * transcode, gchar * action,
                          gchar * job)
{
    GMythSocket    *socket = backend_connect(transcode->backend_info);
    gboolean        res = FALSE;

    if (socket != NULL) {
        gchar          *ret = send_command(socket, action, job,
                                           transcode->chanid,
                                           transcode->starttime, "");

        res = test_result(ret, "JOBQUEUE_CHANGED_CMD_OK");

        gmyth_socket_close_connection(socket);
        g_object_unref(socket);

        g_free(ret);

    } else {
        gmyth_debug("JobQueue Connection Failed");
    }

    return res;
}
