/**
 * GMyth Library
 *
 * @file gmyth/gmyth_file.c
 * 
 * @brief <p> GMythFile deals with the file streaming media remote/local
 * transfering to the MythTV frontend.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Lins Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_file.h"
#include "gmyth_recorder.h"
#include "gmyth_util.h"
#include "gmyth_socket.h"
#include "gmyth_stringlist.h"
#include "gmyth_debug.h"
#include "gmyth_uri.h"
#include "gmyth_marshal.h"

#include <unistd.h>
#include <glib.h>

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>

#define GMYTH_FILE_GET_PRIVATE(obj) \
	(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GMYTH_FILE_TYPE, GMythFilePrivate))

struct _GMythFilePrivate {
    gboolean        disposed;
    gint64          offset;
    guint64         filesize;

    GMythBackendInfo *backend_info;

    /*
     * Myth URI structure 
     */
    gchar          *filename;

    gint            file_id;
};

enum {
    PROP_GMYTH_FILE_DUMMY,
    PROP_GMYTH_FILE_FILENAME,
    PROP_GMYTH_FILE_OFFSET,
    PROP_GMYTH_FILE_FILESIZE,
    PROP_GMYTH_FILE_BACKEND_INFO,
    PROP_GMYTH_FILE_FILEID
};

static void     gmyth_file_set_property(GObject * object, guint prop_id,
                                        const GValue * value,
                                        GParamSpec * pspec);
static void     gmyth_file_get_property(GObject * object, guint prop_id,
                                        GValue * value,
                                        GParamSpec * pspec);

static void     gmyth_file_class_init(GMythFileClass * klass);
static void     gmyth_file_init(GMythFile * object);
static void     gmyth_file_dispose(GObject * object);
static void     gmyth_file_finalize(GObject * object);

G_DEFINE_TYPE(GMythFile, gmyth_file, G_TYPE_OBJECT)
    static void     gmyth_file_class_init(GMythFileClass * klass)
{
    GObjectClass   *gobject_class;
    GMythFileClass *gtransfer_class;

    gobject_class = (GObjectClass *) klass;
    gtransfer_class = (GMythFileClass *) gobject_class;

    gobject_class->dispose = gmyth_file_dispose;
    gobject_class->finalize = gmyth_file_finalize;

    gobject_class->set_property = gmyth_file_set_property;
    gobject_class->get_property = gmyth_file_get_property;

    g_object_class_install_property
        (gobject_class, PROP_GMYTH_FILE_FILENAME,
         g_param_spec_string("filename", "filename",
                             "The file name.",
                             "",
                             G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
                             G_PARAM_STATIC_BLURB | G_PARAM_READABLE |
                             G_PARAM_WRITABLE));

    g_object_class_install_property
        (gobject_class, PROP_GMYTH_FILE_OFFSET,
         g_param_spec_int64("file-offset", "file-offset",
                            "The offset (position) of this file", 0,
                            G_MAXINT64, 0,
                            G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
                            G_PARAM_STATIC_BLURB | G_PARAM_READABLE |
                            G_PARAM_WRITABLE));

    g_object_class_install_property
        (gobject_class, PROP_GMYTH_FILE_FILESIZE,
         g_param_spec_uint64("file-size", "file-size",
                             "The file size in bytes",
                             0, G_MAXUINT64, 0,
                             G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
                             G_PARAM_STATIC_BLURB | G_PARAM_READABLE |
                             G_PARAM_WRITABLE));

    g_object_class_install_property
        (gobject_class, PROP_GMYTH_FILE_BACKEND_INFO,
         g_param_spec_object("backend-info", "backend-info",
                             "The Backend Information about the remote server",
                             G_TYPE_OBJECT,
                             G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
                             G_PARAM_STATIC_BLURB | G_PARAM_READABLE |
                             G_PARAM_WRITABLE));

    g_object_class_install_property
        (gobject_class, PROP_GMYTH_FILE_FILEID,
         g_param_spec_int("file-id", "file-id",
                          "The file ID", 0, G_MAXINT, 0,
                          G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |
                          G_PARAM_STATIC_BLURB | G_PARAM_READABLE |
                          G_PARAM_WRITABLE));

    g_type_class_add_private(gobject_class, sizeof(GMythFilePrivate));

}

static void
gmyth_file_init(GMythFile * file)
{
    g_return_if_fail(file != NULL);

    file->priv = GMYTH_FILE_GET_PRIVATE(file);
}

static void
gmyth_file_dispose(GObject * object)
{
    GMythFilePrivate *priv;
    GMythFile      *file = GMYTH_FILE(object);

    g_return_if_fail(file != NULL);

    priv = GMYTH_FILE_GET_PRIVATE(file);

    if (priv->disposed) {
        /*
         * If dispose did already run, return. 
         */
        return;
    }

    /*
     * Make sure dispose does not run twice. 
     */
    priv->disposed = TRUE;

    if (priv->backend_info != NULL) {
        g_object_unref(priv->backend_info);
        priv->backend_info = NULL;
    }

    if (priv->filename != NULL) {
        g_free(priv->filename);
        priv->filename = NULL;
    }

    G_OBJECT_CLASS(gmyth_file_parent_class)->dispose(object);
}

static void
gmyth_file_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_file_parent_class)->finalize(object);
}

/** 
 * Creates a new instance of GMythFile.
 * 
 * @param backend_info The BackendInfo instance, with all the MythTV network 
 * 										 configuration data.
 * 
 * @return a new instance of the File Transfer. 
 */
GMythFile      *
gmyth_file_new(GMythBackendInfo * backend_info)
{
    GMythFile      *file = NULL;

    g_return_val_if_fail(backend_info != NULL, NULL);

    GParameter     *__params = g_new0(GParameter, 1);
    GParameter     *__params_it = __params;

    (__params_it->name =
     "backend-info", g_value_init(&__params_it->value, G_TYPE_OBJECT),
     g_value_set_object(&__params_it->value, backend_info), __params_it++);
    file =
        g_object_newv(GMYTH_FILE_TYPE, __params_it - __params, __params);

    return file;
}

gchar          *
gmyth_file_get_file_name(GMythFile * file)
{
    GMythFilePrivate *priv = GMYTH_FILE_GET_PRIVATE(file);

    g_return_val_if_fail(file != NULL, NULL);

    return priv->filename;
}

void
gmyth_file_set_file_name(GMythFile * file, const gchar * filename)
{
    g_return_if_fail(file != NULL);
    g_return_if_fail(filename != NULL);

    gchar          *__temp2 = NULL;
    const gchar    *__temp1 = NULL;

    file->priv->filename = (__temp2 =
                            (__temp1 =
                             filename,
                             (__temp1 == NULL ? NULL : g_strdup(__temp1))),
                            (file->priv->filename ==
                             NULL ? NULL : (file->priv->filename =
                                            (g_free(file->priv->filename),
                                             NULL))), __temp2);
}

/** 
 * Creates a new instance of GMythFile.
 * 
 * @param uri_str The URI poiting to the MythTV backend server.
 * 
 * @return a new instance of the File Transfer. 
 */
GMythFile      *
gmyth_file_new_with_uri(const gchar * uri_str)
{
    GMythFile      *file = GMYTH_FILE(g_object_new(GMYTH_FILE_TYPE, NULL));
    GMythFilePrivate *priv = GMYTH_FILE_GET_PRIVATE(file);

    priv->backend_info = gmyth_backend_info_new_with_uri(uri_str);
    return file;
}

/** 
 * Open a File Transfer connection in order to get a remote file.
 * 
 * @param file The actual File Transfer instance. 
 * @param filename The file name of the remote file to be transfered to the client.
 * 
 * @return <code>true</code>, if the connection opening had been done successfully. 
 */
gboolean
gmyth_file_setup(GMythFile * file, const gchar * filename)
{
    gboolean        ret = TRUE;
    GMythFilePrivate *priv;

    g_return_val_if_fail(file != NULL, FALSE);
    g_return_val_if_fail(filename != NULL && strlen(filename) > 0, FALSE);

    priv = GMYTH_FILE_GET_PRIVATE(file);

    if (priv->filename != NULL) {
        gmyth_file_close(file);
    }

    priv->filename = g_strdup(filename);

    return ret;
}

/** 
 * Closes a remote File Transfer connection.
 * 
 * @param file The actual File Transfer instance. 
 */
void
gmyth_file_close(GMythFile * file)
{
    GMythFilePrivate *priv;

    priv = GMYTH_FILE_GET_PRIVATE(file);

    if (priv->filename) {
        g_free(priv->filename);
        priv->filename = NULL;
    }

}

/** 
 * Gets the actual file size of the binary content.
 * 
 * @param file The actual File Transfer instance.
 * 
 * @return The actual file size in bytes. 
 */
guint64
gmyth_file_get_filesize(GMythFile * file)
{
    GMythFilePrivate *priv;

    g_return_val_if_fail(file != NULL, 0);

    priv = GMYTH_FILE_GET_PRIVATE(file);
    return priv->filesize;
}

/** 
 * Sets the actual file size.
 * 
 * @param file The actual File Transfer instance.
 * @param filesize The actual File Transfer size, in bytes.
 */
void
gmyth_file_set_filesize(GMythFile * file, guint64 filesize)
{
    GMythFilePrivate *priv;

    priv = GMYTH_FILE_GET_PRIVATE(file);

    priv->filesize = filesize;
}

/** 
 * Gets the actual offset of the binary content.
 * 
 * @param file The actual File Transfer instance.
 * 
 * @return The actual file offset in bytes. 
 */
gint64
gmyth_file_get_offset(GMythFile * file)
{
    g_return_val_if_fail(file != NULL, 0);

    return file->priv->offset;
}

/**
 * Sets the actual file offset.
 * 
 * @param file The actual File instance.
 * @param filesize The actual File offset, in bytes.
 */
void
gmyth_file_set_offset(GMythFile * file, gint64 offset)
{
    GMythFilePrivate *priv;

    priv = GMYTH_FILE_GET_PRIVATE(file);

    priv->offset = offset;
}

gchar          *
gmyth_file_get_uri(GMythFile * file)
{
    GMythFilePrivate *priv = GMYTH_FILE_GET_PRIVATE(file);
    gchar          *uri = NULL;

    g_return_val_if_fail(file != NULL, NULL);

    if (g_strstr_len(priv->filename, strlen(priv->filename), "://") !=
        NULL)
        uri = g_strdup(priv->filename);
    else
        uri =
            g_strdup_printf("myth://%s:%d/%s",
                            gmyth_backend_info_get_hostname(priv->
                                                            backend_info),
                            gmyth_backend_info_get_port(priv->
                                                        backend_info),
                            priv->filename);

    return uri;
}

static void
gmyth_file_set_property(GObject * object, guint prop_id,
                        const GValue * value, GParamSpec * pspec)
{
    GMythFilePrivate *priv = GMYTH_FILE_GET_PRIVATE(GMYTH_FILE(object));

    switch (prop_id) {
    case PROP_GMYTH_FILE_FILENAME:
        {
            if (!g_value_get_string(value)) {
                break;
            }

            if (priv->filename != NULL) {
                g_free(priv->filename);
                priv->filename = NULL;
            }

            priv->filename = g_value_dup_string(value);
            gmyth_debug("Changed the filename to [%s]!", priv->filename);
            break;
        }
    case PROP_GMYTH_FILE_OFFSET:
        {
            priv->offset = g_value_get_int64(value);
            break;
        }
    case PROP_GMYTH_FILE_FILESIZE:
        {
            priv->filesize = g_value_get_uint64(value);
            break;
        }
    case PROP_GMYTH_FILE_BACKEND_INFO:
        {
            if (!g_value_get_object(value)) {
                break;
            }

            if (priv->backend_info != NULL) {
                g_object_unref(priv->backend_info);
                priv->backend_info = NULL;
            }

            priv->backend_info = g_value_get_object(value);
            gmyth_debug("Changed the backend info to [%s]!",
                        gmyth_backend_info_get_hostname(priv->
                                                        backend_info));
            break;
        }
    case PROP_GMYTH_FILE_FILEID:
        {
            priv->file_id = g_value_get_int(value);
            break;
        }

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }

}

static void
gmyth_file_get_property(GObject * object, guint prop_id,
                        GValue * value, GParamSpec * pspec)
{
    GMythFilePrivate *priv = GMYTH_FILE_GET_PRIVATE(GMYTH_FILE(object));

    switch (prop_id) {
    case PROP_GMYTH_FILE_FILENAME:
        {
            gmyth_debug("Got the filename to [%s]!", priv->filename);
            g_value_set_string(value, priv->filename);
            break;
        }
    case PROP_GMYTH_FILE_OFFSET:
        {
            g_value_set_int64(value, priv->offset);
            break;
        }
    case PROP_GMYTH_FILE_FILESIZE:
        {
            g_value_set_uint64(value, priv->filesize);
            break;
        }
    case PROP_GMYTH_FILE_BACKEND_INFO:
        {
            g_value_set_object(value, priv->backend_info);
            break;
        }
    case PROP_GMYTH_FILE_FILEID:
        {
            g_value_set_int(value, priv->file_id);
            break;
        }
    default:
        {
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
        }
    }

}
