# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from pgm.timing import implicit, controller
from pgm.graph.image import Image
from pgm.widgets import sliced_image, const
from pgm.utils import image as image_utils
import pgm

from twisted.internet import reactor
import math

class Selector(sliced_image.SlicedImage):
    
    time_before_loading = 0.250

    def __init__(self, top, bottom, middle,
                 border_top, border_bottom, border_middle,
                 orientation=const.VERTICAL):
        
        sliced_image.SlicedImage.__init__(self, orientation=orientation)

        self._loading = False
        self._loading_image = None
        self._action_image = None
        self._rotation_matrix = None

        self._create_action_icon()
        self._create_loading_icon()

        self._animated_action = implicit.AnimatedObject(self._action_image)
        self._animated_action.setup_next_animations( duration=800,
                                        repeat_behavior=implicit.REVERSE,
                                        end_behavior=controller.BEGIN,
                                        repeat_count=2)

        self._outside = sliced_image.SlicedImage(orientation=orientation)
        self._outside.top_file = border_top
        self._outside.body_file = border_middle
        self._outside.bottom_file = border_bottom
        
        self._outside.size = self.size
        self.add(self._outside)
        self._outside.z -= 1
        self._outside.visible = True

        self.add(self._loading_image)
        self.add(self._action_image)
        
        # animation support
        self.animated = implicit.AnimatedObject(self._outside, ('opacity','position'))
        # make the cursor glow/unglow via its opacity property

        self.top_file = top
        self.body_file = middle
        self.bottom_file = bottom

        self.start_animation()

    def start_animation(self):
        self.animated.setup_next_animations(duration=850,
                                            transformation=implicit.DECELERATE,
                                            repeat_behavior=implicit.REVERSE,
                                            repeat_count=implicit.INFINITE)
        self.animated.mode = implicit.REPLACE
        self.animated.opacity = 200
        
    def stop_animation(self):
        self.animated.stop_animations()

    def action_done_image_path__set(self, path):
        self._action_image.set_from_file(path, 64)

    def size__set(self, size):
        self._outside.size = size
        super(Selector, self).size__set(size)

    def width__set(self, width):
        self._outside.width = width
        super(Selector, self).width__set(width)
 
    def height__set(self, height):
        self._outside.height = height
        super(Selector, self).height__set(height)
 
    def action_done__set(self, new_value):
        if self._action_image:
            self._action_image.visible = new_value
            self._action_image.opacity = 0

            if new_value:
                self._animated_action.opacity = 255

    def loading_image_path__set(self, path):
        self._loading_image.set_from_file(path, 64)
 
    def _create_action_icon(self):
        #Loading icon is created on demand
        self._action_image = Image()
        self._action_image.bg_color = (0, 0, 0, 0)
        self._action_image.layout = pgm.IMAGE_SCALED
        self._action_image.visible = False
        
    def _create_loading_icon(self):
        #Loading icon is created on demand
        self._loading_image = Image()
        self._loading_image.bg_color = (0, 0, 0, 0)
        self._loading_image.layout = pgm.IMAGE_SCALED
        self._loading_image.visible = False
                    
        self._rotation_matrix = pgm.mat4x4_new_identity()
        self._rotation_matrix.translate(0.5, 0.5, 0.0)
        self._rotation_matrix.rotate_z(math.pi / 30.0)
        self._rotation_matrix.translate(-0.5, -0.5, 0.0)
            
    def loading__set(self, loading):
        self._loading = loading
        reactor.callLater(self.time_before_loading, 
                              self._start_loading_animation)
        
    def loading__get(self):
        return self._loading
 
    def _start_loading_animation(self):
        if self._loading == True and self._loading_image:
            self._loading_image.visible = self.loading
            reactor.callLater(0.017, self._transform_mapping_matrix_cb)
        
    def _transform_mapping_matrix_cb(self):
        if self._loading_image:
            self._loading_image.mapping_matrix *= self._rotation_matrix
            self._loading_image.visible = self.loading
            if self._loading == True:
                reactor.callLater(0.017, self._transform_mapping_matrix_cb)
    
    def _compute_layout(self):
        super(Selector, self)._compute_layout()
        if self._loading_image:
            h = self.height * 0.40
            x = self.width - (2.0 * h)
            y = (self.height - h) / 2.0
            self._loading_image.size = (h, h)
            self._loading_image.position = (x ,y , 0)
        if self._action_image:
            h = self.height * 0.75
            x = self.width * 0.97 - h
            y = (self.height - h) / 2.0
            self._action_image.size = (h, h)
            self._action_image.position = (x ,y , 0)
  
        
if __name__ == "__main__":
    from pgm.graph.image import Image
    from pgm.graph.text import Text
    import pgm
    import os, gobject, sys

    def on_key_pressed(port, event, selector, loop):
        if event.keyval in (pgm.keysyms.Escape, pgm.keysyms.q):
            loop.quit()
        if event.keyval == pgm.keysyms.a:
            max_opacity = 255
            min_opacity = 200
            if selector.animated.opacity == min_opacity:
                selector.animated.opacity = max_opacity
            elif selector.animated.opacity == max_opacity:
                selector.animated.opacity = min_opacity
            
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'bar widget'

    canvas = pgm.Canvas()
    gl.set_canvas(canvas)
    gl.show()

    pictos = ("theme/selector-left.png",
              "theme/selector-right.png",
              "theme/selector-body.png",
              "theme/back-selector-left.png",
              "theme/back-selector-right.png",
              "theme/back-selector-body.png",
              "theme/arrow.png",
              )
    selector = Selector(orientation=const.HORIZONTAL,
                        *pictos)
    selector.canvas = canvas
    selector.position = (0.5, 1, 0)
    selector.size = (canvas.width * 0.5, canvas.height * 0.1)
    selector.visible = True
    
    loop = gobject.MainLoop()
    gl.connect('key-press-event', on_key_pressed, selector, loop)
    loop.run()
