# Copyright 2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
# desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Manuel de la pena <manuel.delapena@canonical.com>
"""Plug-in for pairing desktopcouch to the Ubuntu One sync service."""

import logging
import sys
from desktopcouch.application.pair.couchdb_pairing.couchdb_io import (
    put_static_paired_service)
from desktopcouch.application.server import DesktopDatabase
from ubuntuone.clientdefs import APP_NAME

U1_PAIR_RECORD = "ubuntu_one_pair_record"
MAP_JS = """function(doc) {
    if (doc.service_name == "ubuntuone") {
        emit(doc._id, 0)
            }
    }
"""


def pair_with_ubuntuone(couchdb_port, management_db=None,
                        db_class=DesktopDatabase,
                        put_service_fn=put_static_paired_service):
    """Adds a pairing record with ubuntu one when needed."""
    # Use explicit uri so that we do not access dbus service.
    uri = "http://localhost:%s" % (couchdb_port,)
    if not management_db:
        management_db = db_class("management", uri=uri, create=True, ctx=None)
    # we indeed have credentials to add to the pairing records
    # but first we ensure that the required view is present
    if not management_db.view_exists(U1_PAIR_RECORD, U1_PAIR_RECORD):
        management_db.add_view(
            U1_PAIR_RECORD, MAP_JS, design_doc=U1_PAIR_RECORD)
    view_results = management_db.execute_view(U1_PAIR_RECORD, U1_PAIR_RECORD)
    pairing_found = False
    # Results should contain either one row or no rows
    # If there is one row, its value will be 0, meaning that there is
    #   already an Ubuntu One pairing record, or 1, meaning that there
    #   was an Ubuntu One pairing record but it has since been unpaired
    # Only create a new record if there is not one already. Specifically,
    #   do not add the record if there is a deleted one, as this means
    #   that the user explicitly unpaired it!
    for row in view_results:
        pairing_found = True
        if row.value == 1:
            logging.debug("Not adding desktopcouch pairing since the user "
                "has explicitly unpaired with Ubuntu One")
        else:
            logging.debug("Not adding desktopcouch pairing since we are "
                "already paired")
    if not pairing_found:
        put_service_fn(None, "ubuntuone", uri=uri, ctx=None)
        logging.debug("Pairing desktopcouch with Ubuntu One")


def got_new_credentials(couchdb_port, app_name, credentials):
    """Pair with Ubuntu One when we get the new credentials."""
    if app_name == APP_NAME:
        pair_with_ubuntuone(couchdb_port)


def listen_to_dbus(couchdb_port):
    """Set up the signal handler on D-Bus for Ubuntu One pairing."""
    import dbus
    bus = dbus.SessionBus()

    try:
        import ubuntu_sso

        receiver = lambda *args: got_new_credentials(couchdb_port, *args)

        iface = ubuntu_sso.DBUS_CREDENTIALS_IFACE
        bus.add_signal_receiver(handler_function=receiver,
                                signal_name='CredentialsFound',
                                dbus_interface=iface)

        obj = bus.get_object(ubuntu_sso.DBUS_BUS_NAME,
                             ubuntu_sso.DBUS_CREDENTIALS_PATH,
                             follow_name_owner_changes=True)
        sso_backend = dbus.Interface(obj, dbus_interface=iface)
        sso_backend.find_credentials(APP_NAME, {})
    except ImportError:
        logging.info('Ubuntu SSO is not available.')


def plugin_init(couchdb_port):
    """Set up the signal handler for pairing with Ubuntu One."""
    logging.info('Loaded Ubuntu One extension for desktopcouch.')
    if sys.platform == 'win32':
        logging.warning('Windows support for Ubuntu One is not yet ready.')
    else:
        import gobject
        gobject.idle_add(listen_to_dbus, couchdb_port)
