/*
 * alsaseq a simple midi sequencer based on ALSA
 * for Denemo, a gtk+ frontend to GNU Lilypond.
 * (c) 2006 Benoit Rouits <brouits@free.fr>
 * 
 * Useful code to read to learn the ALSA api:
 * - miniArp.c by Matthias Nagorni
 *   (http://www.suse.de/~mana/alsa090_howto.html)
 *   
 * - midirecord.cc by Tuomas Airaksinen
 *   (http://people.jyu.fi/~tuma/home/progs.php)
 *
 * - midibus.cpp and event.cpp of seq24 by Rob C. Buse
 *   (http://www.filter24.org/seq24/)
 *   
 * - /usr/include/alsa/seqmid.h by the ALSA Team
 *
 */
#ifndef ALSASEQ_H
#define ALSASEQ_H

#include <alsa/asoundlib.h>

typedef struct _plug {
        int id; /* the quick id of the plug */
        int dh; /* the destination host */
        int dp; /* the destination port */
} plug;

typedef struct _alsa_seq {
    snd_seq_t      *seq;	/* the actual ALSA sequencer */
    int             queue;      /* the event queue descriptor */
    int             port[2];	/* my local i/o Midi ports */
    plug           *plugs;      /* Midi host:port destination list */
} alsa_seq;

typedef snd_seq_event_t alsa_ev;

typedef snd_seq_event_type_t alsa_type;

typedef unsigned char alsa_val;
#if 0 /* this is useless since we use midiseq.h */
/*
 * sequencer initialization and termination
 */

alsa_seq      *
midi_seq_new(const char *name);
/*
 * creates a midi sequencer
 */

int
midi_seq_delete(alsa_seq *player);
/*
 * closes and deletes the midi sequencer
 */

int
midi_seq_plug_to(alsa_seq *player, int dh, int dp);
/*
 * plugs to another midi host:port
 */

int
midi_seq_unplug_to(alsa_seq * player, int dh, int dp);
/*
 * disconnect the player to a midi host:port
 */

/*
 * midi sequence creation
 */

int
midi_seq_put_note
(alsa_seq *p, alsa_val channel, alsa_val key, alsa_val press, unsigned int dur, alsa_val rel);
/*
 * puts a note to the midi sequence
 */

extern int
midi_seq_put_control(alsa_seq *p, alsa_val channel, alsa_val c, alsa_val v);
/*
 * puts a midi control to the midi sequence
 * e.g:
 * - full damper pedal is MIDI_PEDAL, 127
 * - off damper pedal is MIDI_PEDAL, 0
 */

int
midi_seq_put_raw(alsa_seq *p, char* data, int len);
/*
 * puts a raw midi event made by hand (experts only)
 */

/*
 * playback controls
 */

int
midi_seq_play(alsa_seq *player);
/*
 * start to play the midi sequence
 */

int
midi_seq_stop(alsa_seq *player);
/*
 * stops the midi sequence
 */

void
midi_seq_flush(alsa_seq *player);
/*
 * flushes buffered events of the sequence
 */

/*
 * recording controls
 */

alsa_ev *
midi_seq_rec_ev(alsa_seq *player);
/*
 * return the next event in the sequence from readable plugs
 */

/*
 * midi event parsing
 */

alsa_type
midi_seq_ev_get_type(alsa_ev *event);
/*
 * returns the midi type of the given event
 */

/* note */

alsa_val
midi_note_get_key(alsa_ev *note);
/*
 * if the event is a note, returns the key
 */

alsa_val
midi_note_get_pressvel(alsa_ev *note);
/*
 * if the event is a note, returns the pressure velocity
 */

alsa_val
midi_note_get_relvel(alsa_ev *note);
/*
 * if the event is a note, returns the release velocity
 */

alsa_val
midi_note_get_channel(alsa_ev *event);
/*
 * if the event is a note, returns the channel (0-15)
 */

/* control */

alsa_val
midi_control_get_val(alsa_ev *control);
/*
 * if the event is a control (e.g: pedal), returns the value (0-127)
 */
#endif
#endif /* ALSASEQ_H */
