# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Application interface for use by plugins.

This primarily makes certain apt functionality available to the plugin.
"""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'Application',
    ]


import apt

import computerjanitorapp

from computerjanitord.errors import (
    MissingLandmarkError, NonDownloadableError)


_ = computerjanitorapp.setup_gettext()

SYNTAPTIC_PREFERENCES_FILE = '/var/lib/synaptic/preferences'
# There really isn't anything special about these packages.  These really just
# represent landmarks in the package namespace that we look for to try to
# judge the sanity of the apt cache.  If these are missing, things are really
# messed up and we can't actually figure out how to continue.
LANDMARK_PACKAGES = [
    'dash',
    'gzip',
    ]


class Application:
    """Interface for plugins requesting apt actions."""

    def __init__(self, apt_cache=None):
        """Create the application interface.

        :param apt_cache: Alternative apt cache for testing purposes.  When
            `None` use the default apt cache.
        """
        if apt_cache is None:
            # Use the real apt cache.
            self.apt_cache = apt.Cache()
        else:
            self.apt_cache = apt_cache
        self.refresh_apt_cache()

    def refresh_apt_cache(self):
        """Refresh the apt cache.

        This API is used by plugins.
        """
        self.apt_cache.open()
        # For historical purposes, Synaptic has a different way of pinning
        # packages than apt, so we have to load its preferences file in order
        # to know what it's pinning.
        self.apt_cache._depcache.ReadPinFile(SYNTAPTIC_PREFERENCES_FILE)

    def verify_apt_cache(self):
        """Verify that essential packages are available in the apt cache.

        This API is used by plugins.

        :raises MissingLandmarkError: When an essential package is not
            available.
        :raises NonDownloadableError: When an essential package cannot be
            downloaded.
        """
        for name in LANDMARK_PACKAGES:
            if name not in self.apt_cache:
                raise MissingLandmarkError(name)
            if not any(v.downloadable for v in self.apt_cache[name].versions):
                raise NonDownloadableError(name)
