/* +-------------------------------------------------------------------+ */
/* | Copyright 1992, 1993, David Koblas (koblas@netcom.com)	       | */
/* |								       | */
/* | Permission to use, copy, modify, and to distribute this software  | */
/* | and its documentation for any purpose is hereby granted without   | */
/* | fee, provided that the above copyright notice appear in all       | */
/* | copies and that both that copyright notice and this permission    | */
/* | notice appear in supporting documentation.	 There is no	       | */
/* | representations about the suitability of this software for	       | */
/* | any purpose.  this software is provided "as is" without express   | */
/* | or implied warranty.					       | */
/* |								       | */
/* +-------------------------------------------------------------------+ */

/* $Id: fontSelect.c,v 1.17 2005/03/20 20:15:32 demailly Exp $ */

#include <math.h>
#include <stdio.h>
#include <locale.h>
#include <ctype.h>
#include <math.h>

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xos.h>
#include <X11/Xft/Xft.h>

#include "xaw_incdir/Form.h"
#include "xaw_incdir/Paned.h"
#include "xaw_incdir/List.h"
#include "xaw_incdir/AsciiText.h"
#include "xaw_incdir/Command.h"
#include "xaw_incdir/Viewport.h"
#include "xaw_incdir/Scrollbar.h"

#include "Paint.h"

#define XftNameUnparseAlloc FcNameUnparse

/*
**  swiped from X11/Xfuncproto.h
**   since qsort() may or may not be defined with a constant sub-function
 */
#ifndef _Xconst
#if __STDC__ || defined(__cplusplus) || defined(c_plusplus) || (FUNCPROTO&4)
#define _Xconst const
#else
#define _Xconst
#endif
#endif				/* _Xconst */

#ifndef NOSTDHDRS
#include <stdlib.h>
#include <unistd.h>
#endif

#include "xpaint.h"
#include "messages.h"
#include "misc.h"
#include "operation.h"
#include "ops.h"
#include "graphic.h"
#include "protocol.h"

#if defined(XAWPLAIN) || defined(XAW3D)
#define BORDERWIDTH 1
#else
#define BORDERWIDTH 0
#endif

typedef struct {
    int num_fonts, num_families, num_weights;
    char *font_select;
    char *weight_select;
    char **font_desc;
    char **font_family;
    char **weight_list;
    XftFont * xft_font;
    char * xft_name;
    double xft_size, xft_height, xft_ascent, xft_descent, xft_maxadv;
    double xft_rot, xft_linespacing;
    Pixmap pixmap;
    Widget shell, form1, vport0, vport1, vport2, subform, text, paint;
    Widget family, familyLabel;
    Widget weight, weightLabel;
    Widget pointBar, pointSelect, pointSelectLabel;
    Widget rotation, rotationLabel;
    Widget dilation, dilationLabel;
    Widget linespacing, linespacingLabel;
    Widget inclination, inclinationLabel;
} arg_t;

static arg_t *theArg = NULL;

static int 
strqsortcmp(char **a, char **b)
{
    if (**a == ':') return 1;
    if (**b == ':') return -1;
    return strcasecmp(*a, *b);
}

void
FreeTypeDrawString(Display *dpy, arg_t * arg,
                   XftDraw *draw, XftColor * color,
                   char *buf, int len, int x, int y, double r)
{
    int j;
    double dx=0.5, dy=0.5, a, u, v;
    XGlyphInfo extents = {};

    u = cos(r);
    v = sin(r);

    for (j=0; j<len; j++) {
        XftDrawString8(draw, color, arg->xft_font,
		       x+((int)dx)+ v*arg->xft_ascent, 
                       y+((int)dy)+ u*arg->xft_ascent, 
                       (XftChar8*)(buf+j), 1);
        XftTextExtents8(dpy, arg->xft_font,
			(XftChar8*)(buf+j), 1, 
                        (XGlyphInfo*)&extents);
        a = sqrt(extents.xOff*extents.xOff+extents.yOff*extents.yOff);
        dx += a * u;
        dy -= a * v;
    }
}

static void 
setSamplePixmap(arg_t * arg)
{
    XtVaSetValues(arg->text, XtNbackgroundPixmap, None, NULL);
    if (arg->pixmap)
        XtVaSetValues(arg->text, XtNbackgroundPixmap, arg->pixmap, NULL);
    /* force refresh  */
    XUnmapWindow(XtDisplay(arg->text), XtWindow(arg->text));
    XMapWindow(XtDisplay(arg->text), XtWindow(arg->text));
}

static void
cleanSamplePixmap(arg_t * arg)
{
    Display * dpy = XtDisplay(arg->shell);
    GC gc;
    int w, h, d;
    if (arg->pixmap) {
        GetPixmapWHD(dpy, arg->pixmap, &w, &h, &d);
        gc = XCreateGC(dpy, arg->pixmap, 0, NULL);
        XSetForeground(dpy, gc, WhitePixelOfScreen(XtScreen(arg->shell)));
        XFillRectangle(dpy, arg->pixmap, gc, 0, 0, w, h);
        XFreeGC(dpy, gc);
    }
    setSamplePixmap(arg);
}

static void 
setXftFont(arg_t * arg)
{
    char * sample[3] = {
      "ABCDEFGHIJKLMNOPQRSTUVWXYZ",
      "abcdefghijlmnopqrstuvwxyz",
      "0123456789-+*/=<>.,:;?!&~#(){}[]\\'`\"^@$"
    };
    Display *dpy = XtDisplay(arg->shell);
    XGlyphInfo extents = {};
    Colormap cmap;
    GC gc;
    Dimension width, height;

    XftColor color;
    XftDraw *draw = NULL;
    XRenderColor xre_color;
    char matrix[80];
    char *ptr;
    double r=0.0, i=0.0, d=1.0, ls=1.0, f = M_PI/180, hd;
    double mat[4];
    int j, x, y, w, x0, x1, x2, y0, y1, y2;

    if (!arg->font_select) {
        cleanSamplePixmap(arg);
        return;
    }

    XtVaGetValues(arg->shell, XtNcolormap, &cmap, NULL);
    XtVaGetValues(arg->vport2, XtNwidth, &width, XtNheight, &height, NULL);
    XtVaGetValues(arg->pointSelect, XtNstring, &ptr, NULL);
    if (ptr) {
        arg->xft_size = atof(ptr);
        if (arg->xft_size<1.0) arg->xft_size = 1.0;
        if (arg->xft_size>300.0) arg->xft_size = 300.0;
    } else
        arg->xft_size = 18.0;
    sprintf(matrix, "%g", arg->xft_size);
    x = strlen(matrix);

    XtVaGetValues(arg->rotation, XtNstring, &ptr, NULL);
    if (ptr) r = atof(ptr)*f;
    arg->xft_rot = r;

    XtVaGetValues(arg->dilation, XtNstring, &ptr, NULL);
    if (ptr) d = atof(ptr);

    XtVaGetValues(arg->linespacing, XtNstring, &ptr, NULL);
    if (ptr) ls = atof(ptr);
    arg->xft_linespacing = ls;

    XtVaGetValues(arg->inclination, XtNstring, &ptr, NULL);
    if (ptr) i = atof(ptr)*d;

    if (r!=0.0 || d!= 1.0 || i!=0.0) {
        mat[0] = cos(r);
        mat[2] = sin(r);
        mat[1] = i*mat[0]-d*mat[2];
        mat[3] = i*mat[2]+d*mat[0];
        sprintf(matrix, ":matrix=%g %g %g %g",
		mat[0], mat[1], mat[2], mat[3]);
    } else {
        *matrix = '\0';
        mat[0] = mat[3] = 1.0;
        mat[1] = mat[2] = 0.0;
    }

    ptr = strchr(arg->font_select, '(');
    if (ptr && ptr>=arg->font_select+2) *(ptr-2) = '\0';

    if (arg->weight_select) { 
        j = strlen(arg->font_select)+strlen(arg->weight_select)+
            strlen(matrix)+x+9;
        arg->xft_name = realloc(arg->xft_name, j);
        sprintf(arg->xft_name, 
                "%s-%g:style=%s", arg->font_select, arg->xft_size, 
		arg->weight_select);
    } else {
        j = strlen(arg->font_select)+strlen(matrix)+x+2;
        arg->xft_name = realloc(arg->xft_name, j);
        sprintf(arg->xft_name, "%s-%g", arg->font_select, arg->xft_size);
    }

    if (arg->xft_font) XftFontClose(dpy, arg->xft_font);
    arg->xft_font = XftFontOpenName(dpy, DefaultScreen(dpy), arg->xft_name);

    if (arg->xft_font) {
        arg->xft_height = arg->xft_font->height * d;
        arg->xft_ascent = arg->xft_font->ascent * d;
        arg->xft_descent = arg->xft_font->descent * d;
        arg->xft_maxadv = arg->xft_font->max_advance_width;     
    }

    if (*matrix) {
        if (arg->weight_select) {
            sprintf(arg->xft_name, 
                    "%s-%g:style=%s%s", arg->font_select, arg->xft_size, 
		    arg->weight_select, matrix);
        } else
            sprintf(arg->xft_name, 
                    "%s-%g%s", arg->font_select, arg->xft_size, matrix);
        if (arg->xft_font) XftFontClose(dpy, arg->xft_font);
        arg->xft_font = 
            XftFontOpenName(dpy, DefaultScreen(dpy), arg->xft_name);
    }

    if (ptr) *(ptr-2) = ' ';

    if (!arg->xft_font) {
        cleanSamplePixmap(arg);
        return;
    }

    w = arg->xft_maxadv;
    hd = arg->xft_height*ls;

    x1 = x2 = y1 = y2 = 0;
    x = (int)(0.5+3*hd*mat[2]);
    y = (int)(0.5+3*hd*mat[0]);
    if (x<=x1) x1 = x; if (x>=x2) x2 = x;
    if (y<=y1) y1 = y; if (y>=y2) y2 = y;

    for (j=0; j<=2; j++) {    
        XftTextExtents8(dpy, arg->xft_font,
		        (XftChar8*)sample[j], strlen(sample[j]), 
                        (XGlyphInfo*)&extents);
        x = extents.xOff + (int)(0.5+(j-0.2)*hd*mat[2]);
        y = extents.yOff + (int)(0.5+(j-0.2)*hd*mat[0]);
	if (x<=x1) x1 = x; if (x>=x2) x2 = x;
	if (y<=y1) y1 = y; if (y>=y2) y2 = y;
        x = extents.xOff + (int)(0.5+(j+1.2)*hd*mat[2]);
        y = extents.yOff + (int)(0.5+(j+1.2)*hd*mat[0]);
	if (x<=x1) x1 = x; if (x>=x2) x2 = x;
	if (y<=y1) y1 = y; if (y>=y2) y2 = y;
    }

    if ((j=x2-x1+6+w/4)>= width) width = j;
    if ((j=y2-y1+6+((int)arg->xft_height*0.2501))>= height) height = j;

    XtResizeWidget(arg->text, width, height, 0);
    XtVaSetValues(arg->text, XtNwidth, width, XtNheight, height, NULL);
    if (arg->pixmap) XFreePixmap(dpy, arg->pixmap);
    arg->pixmap = XCreatePixmap(dpy, XtWindow(arg->text),
                   width, height,    
                   DefaultDepthOfScreen(XtScreen(arg->shell)));
    gc = XCreateGC(dpy, arg->pixmap, 0, NULL);
    XSetForeground(dpy, gc, WhitePixelOfScreen(XtScreen(arg->shell)));
    XFillRectangle(dpy, arg->pixmap, gc, 0, 0, width, height);
    xre_color.red = 0;
    xre_color.green = 0;
    xre_color.blue = 0;
    xre_color.alpha = 255<<8;
    XftColorAllocValue(dpy, DefaultVisual(dpy, DefaultScreen(dpy)),
			       cmap, &xre_color, &color);
    draw = XftDrawCreate(dpy, arg->pixmap,
                         DefaultVisual(dpy, DefaultScreen(dpy)), cmap);

    x0 = 3+w/8;
    y0 = 3+(int) arg->xft_height/8;
    if (x1<0) x0 += -x1;
    if (y1<0) y0 += -y1;

    for (j=0; j<=2; j++) {
        x = x0 + (int)(0.5+j*hd*mat[2]);
        y = y0 + (int)(0.5+j*hd*mat[0]);     
        FreeTypeDrawString(dpy, arg, draw, &color, 
                           sample[j], strlen(sample[j]), x, y, r);
    }

    setSamplePixmap(arg);
    XFreeGC(dpy, gc);
    XftDrawDestroy(draw);
}

static void 
barCallback(Widget w, arg_t * arg, float *percent)
{
    int npts;
    char val[8];
    npts = (int) (428*(*percent));
    if (npts<=100) npts = 1+npts/5;
    else
    if (npts<=200) npts = npts/2-28;
    else
    npts = npts-128;
    sprintf(val, "%d", npts);
    XtVaSetValues(arg->pointSelect, XtNstring, val, NULL);
    setXftFont(arg);
}

static void
setBar(arg_t * arg, double npts)
{
    float percent; 
    char val[8];

    sprintf(val, "%g", npts); 
    XtVaSetValues(arg->pointSelect, XtNstring, val, NULL);

    if (npts<=1.0) npts = 1.0;

    if (npts<=21.0)
        percent = 5*(npts-1)/428.0;
    else
    if (npts<=72.0)
        percent = 2*(npts+28)/428.0;
    else
        percent = (npts+128)/428.0;
    if (percent>=1.0) percent = 1.00001;

    XawScrollbarSetThumb(arg->pointBar, percent, -1.0);
    setXftFont(arg);
}

static void 
scrollCallback(Widget w, arg_t * arg, XtPointer position)
{
    char *ptr;
    double npts;

    XtVaGetValues(arg->pointSelect, XtNstring, &ptr, NULL);
    npts = atof(ptr);
    if ((long)position>0) {
        npts += 1.0;
        if (npts>=300.0) npts = 300.0;
    } else {
        npts -= 1.0;
        if (npts<=1.0) npts = 1.0;
    }
    setBar(arg, npts);
}

static void 
applySetCallback(Widget paint, Arg * argArg)
{
    FontChanged(paint);
}

static void 
applyCallback(Widget w, XtPointer argArg, XtPointer junk)
{
    Display *dpy = XtDisplay(w);
    arg_t *arg = (arg_t *) argArg;
    XftFont * font;

    setXftFont(arg);
    if (!arg->xft_font) return;

    /* reopen same font under different XftFont struct chunk */
    font = XftFontOpenName(dpy, DefaultScreen(dpy), arg->xft_name);
    if (!font) {
	Notice(Global.toplevel, msgText[UNABLE_TO_LOAD_REQUESTED_FONT]);
	return;
    }
    if (Global.xft_font) 
        XftFontClose(dpy, (XftFont *)Global.xft_font);
    Global.xft_font = font;
    if (Global.xft_name) 
        free(Global.xft_name);
    Global.xft_name = strdup(arg->xft_name);
    Global.xft_size = arg->xft_size;
    Global.xft_height = arg->xft_height;
    Global.xft_ascent = arg->xft_ascent;
    Global.xft_descent = arg->xft_descent;
    Global.xft_maxadv = arg->xft_maxadv;
    Global.xft_rot = arg->xft_rot;
    Global.xft_linespacing = arg->xft_linespacing;
    GraphicAll((GraphicAllProc) applySetCallback, NULL);
}

static void 
closeCallback(Widget w, XtPointer argArg, XtPointer junk)
{
    arg_t *arg = (arg_t *) argArg;
    XtPopdown(arg->shell);
}

static void 
clickCallback(Widget w, XtPointer argArg, XEvent * event, Boolean * flg)
{
    arg_t *arg = (arg_t *) argArg;
    XtSetKeyboardFocus(arg->shell, w);
}

static void 
unselectCallback(Widget w, XtPointer argArg, XEvent * event, Boolean * flg)
{
    arg_t *arg = (arg_t *) argArg;

    if (!arg->font_select) return;
    if (event->type != ButtonPress) return;
    if (event->xbutton.button != 1) return;

    if (!strcmp(XtName(w), "font")) {
        if (arg->font_select) free(arg->font_select);
        arg->font_select = NULL;
        if (!arg->pixmap) return;
    }
    if (!strcmp(XtName(w), "weight")) {
        if (arg->weight_select) free(arg->weight_select);
	arg->weight_select = NULL;
    }
    setXftFont(arg);
}

static void 
listCallback(Widget w, XtPointer argArg, XtPointer itemArg)
{
    arg_t *arg = (arg_t *) argArg;
    XawListReturnStruct *item = (XawListReturnStruct *) itemArg;
    char *ptr1, *ptr2;
    int j, k, l;

    if (!strcmp(XtName(w), "font")) {
        if (!item || !item->string) {
	    if (arg->font_select) free(arg->font_select);
	    arg->font_select = NULL;
            cleanSamplePixmap(arg);
            return;       
	}
	if (arg->font_select) free(arg->font_select);
        arg->font_select = strdup(item->string);

        for (j=0; j<arg->num_weights; j++)
	    free(arg->weight_list[j]);
        ptr1 = strchr(arg->font_select, '(');
        if (ptr1) *(ptr1-2) = '\0';
        l = strlen(arg->font_select);
        if (ptr1) *(ptr1-2) = ' ';
        k = 0;
        for (j=0; j<arg->num_fonts; j++) {
	  if (!strncasecmp(arg->font_desc[j], arg->font_select, l)) {
	        ptr2 = strrchr(arg->font_desc[j], ':');
                if (ptr2) {
                    arg->weight_list = 
		       (char **)realloc(arg->weight_list, (k+1)*sizeof(char *));
                    /* hack to put light, medium, normal (et al) on top */
                    if (strncasecmp(ptr2+1, "light", 5) &&
                        strncasecmp(ptr2+1, "medium", 6) &&
                        strncasecmp(ptr2+1, "normal", 6) &&
                        strncasecmp(ptr2+1, "regular", 7))
		        arg->weight_list[k] = strdup(ptr2+1);
                    else {
		        arg->weight_list[k] = strdup(ptr2);
                        arg->weight_list[k][0] = '\1';
		    }
                    k++;
		}
	    }
	}

        qsort(arg->weight_list, k, sizeof(char *),
          (int (*)(_Xconst void *, _Xconst void *)) strqsortcmp);
        j = 1;
        while (j<k) {
	    if (!strcasecmp(arg->weight_list[j], arg->weight_list[j-1])) {
	        free(arg->weight_list[j]);
                --k;
                for (l=j; l<k; l++) 
		    arg->weight_list[l] = arg->weight_list[l+1];
	    } else
	        j++;
	}
        /* remove hack (first character = 1) */
        for (j=0; j<k; j++)
	    if (arg->weight_list[j][0] == '\1')
	        for (l=0; l<strlen(arg->weight_list[j]); l++)
		    arg->weight_list[j][l] = arg->weight_list[j][l+1];
        arg->num_weights = k;
        XawListChange(arg->weight, arg->weight_list, k, 0, True);
	if (arg->weight_select) free(arg->weight_select);
        arg->weight_select = NULL;
    }
    if (!strcmp(XtName(w), "weight")) {
        if (arg->weight_select) free(arg->weight_select);
	arg->weight_select = NULL;
        if (!item || !item->string) {
	    arg->weight_select = NULL;
	} else
            arg->weight_select = strdup(item->string);
      
    }
    setXftFont(arg);
}

void
fontboxResized(Widget w, arg_t * arg, XConfigureEvent * event, Boolean * flg)
{
    Display *dpy = XtDisplay(w);
    Dimension width, height, height1, height2;
    char *val;
    double m = 18.0;

    XtVaGetValues(arg->shell, XtNwidth, &width, XtNheight, &height, NULL);
    XtVaGetValues(arg->form1, XtNheight, &height1, NULL);
    if (width<360) width = 360;
    if (height1<150) height1 = 150;
    XtResizeWidget(arg->shell, width, height, 0);
    XtResizeWidget(arg->form1, width, height1, 0);
    height1 -= 32;
   
    XtResizeWidget(arg->familyLabel, 150, 20, 0);
    XtVaSetValues(arg->familyLabel, XtNwidth, 150, XtNheight, 20, NULL);   
    XtMoveWidget(arg->familyLabel, -2, 4);

    XtMoveWidget(arg->vport0, 4, 27);
    XtResizeWidget(arg->vport0, width-218, height1, BORDERWIDTH);
    XtVaSetValues(arg->vport0, XtNwidth, width-218, NULL);

    XtResizeWidget(arg->weightLabel, 150, 20, 0);
    XtVaSetValues(arg->weightLabel, XtNwidth, 150, XtNheight, 20, NULL);   
    XtMoveWidget(arg->weightLabel, width-220, 4);

    XtMoveWidget(arg->vport1, width-209, 27);
    XtResizeWidget(arg->vport1, 205, height1, BORDERWIDTH);
    XtVaSetValues(arg->vport1, XtNwidth, 205, NULL);   

#ifdef XAW3D   
    height2 = height-height1-162;
#else
    height2 = height-height1-158;
#endif
    if (height2<4) height2 = 4;

    XtResizeWidget(arg->vport2, width-8, height2, BORDERWIDTH);
    XtVaSetValues(arg->vport2, XtNwidth, width-8, NULL);
    XtResizeWidget(arg->text, width-8, height2, 0);

    XtResizeWidget(arg->subform, width, 60, 0);
    XtMoveWidget(arg->pointSelectLabel, 4, 8);

    XtResizeWidget(arg->pointBar, 350, 20, 0);
    XtMoveWidget(arg->pointBar, 150, 8);
    XtMoveWidget(arg->pointSelect, 506, 8);

    XtMoveWidget(arg->rotationLabel, 4, 38);
    XtMoveWidget(arg->rotation, 80, 38);

    XtMoveWidget(arg->inclinationLabel, 159, 38);
    XtMoveWidget(arg->inclination, 235, 38);

    XtMoveWidget(arg->dilationLabel, 314, 38);
    XtMoveWidget(arg->dilation, 390, 38);

    XtMoveWidget(arg->linespacingLabel, 469, 38);
    XtMoveWidget(arg->linespacing, 545, 38);

    XtMoveWidget(arg->vport2, 4, 76);

    XtVaGetValues(arg->pointSelect, XtNstring, &val, NULL);
    if (val) m = atof(val);
    if (m<=2.0001) m = 2.0001;
    if (30*m>=width) width = (int) (30*m);
    if (5*m>=height2-10) height2 = (int) (5*m+10);
    XtVaSetValues(arg->text, XtNwidth, width, XtNheight, height2, NULL);
    if (!arg->pixmap) {
        arg->pixmap = XCreatePixmap(dpy, XtWindow(arg->text),
                           width, height2,    
                           DefaultDepthOfScreen(XtScreen(arg->shell)));
        cleanSamplePixmap(arg);
    }
    if (!XPending(XtDisplay(w)))
        setXftFont(arg);
}

static void 
inputDataAction(Widget w, XEvent * event, String * prms, Cardinal * nprms)
{
    char *val;

    if (theArg == NULL)
	return;

    XtVaGetValues(theArg->pointSelect, XtNstring, &val, NULL);
    if (val && *val) setBar(theArg, atoi(val));

    if (!XtIsManaged(theArg->pointSelect))
	return;
    
    setXftFont(theArg);
    fontboxResized(w, theArg, (XConfigureEvent *)event, NULL);
}

void 
FontSelect(Widget w, Widget paint)
{
    Display *dpy = XtDisplay(w);
    XftFontSet	*fs;
    XftChar8 *str;
    XftChar8 *foundry;
    XftChar8 *style;

    static XtActionsRec dataAct =
      {"input-data-ok", (XtActionProc) inputDataAction};
    static Widget shell = None;
    Widget pane, form1, form2;
    Widget applyButton, doneButton, label;
    arg_t *arg = XtNew(arg_t);
    char *ptr1, *ptr2;
    int j, k, l, lmax;

    PopdownMenusGlobal();

    if (shell != None) {
	XtPopup(shell, XtGrabNone);
	XMapRaised(XtDisplay(shell), XtWindow(shell));
        XtSetMinSizeHints(shell, 360, 480);
	return;
    }

    StateSetBusyWatch(True);

    /*
    **	Init the world
     */

    XftInitFtLibrary();
    fs = XftListFonts(dpy, DefaultScreen(dpy), 0,
                      XFT_FAMILY, XFT_FOUNDRY, XFT_STYLE, (char *)0);
    arg->font_desc = (char **) xmalloc(fs->nfont * sizeof(char *));
    lmax = 0;
    for (j = 0; j <fs->nfont; j++) {
        /* Caution : this allocates str */
        str = XftNameUnparseAlloc(fs->fonts[j]);
        /* Search for first alternative of family name in str */
        ptr1 = strchr(str, ':');
        if (ptr1) *ptr1 = '\0';
        ptr2 = strchr(str, ',');
        if (ptr2) *ptr2 = '\0';
        /* length of family name */
        k = strlen(str);
        if (ptr1) *ptr1 = ':';
        if (ptr2) *ptr2 = ',';
	if (XftPatternGetString(fs->fonts[j], XFT_FOUNDRY, 0, &foundry) !=
            XftResultMatch)
	    foundry = "unknown";
	if (XftPatternGetString(fs->fonts[j], XFT_STYLE, 0, &style) != 
            XftResultMatch)
	    style = "";
        l = k+strlen(foundry)+strlen(style)+3;
        if (l>=lmax) lmax = l;
        arg->font_desc[j] = (char *) xmalloc(l);
        /* copy family name */
        strncpy(arg->font_desc[j], str, k);
        sprintf(arg->font_desc[j]+k, ":%s:%s", foundry, style);
        free(str);
    }

    qsort(arg->font_desc, fs->nfont, sizeof(char *),
          (int (*)(_Xconst void *, _Xconst void *)) strqsortcmp);

    k = 0;
    arg->num_fonts = 0;
    arg->font_family = (char **) xmalloc(fs->nfont * sizeof(char *));

    for (j = 0; j<fs->nfont; j++) {
        if (arg->font_desc[j][0] == ':') break;
        ++arg->num_fonts;
        ptr1 = strchr(arg->font_desc[j], ':');
        if (ptr1) { 
            *ptr1 = '\0';
            ptr2 = strchr(ptr1+1, ':');
	} else
	    ptr2 = NULL;
        if (ptr2) *ptr2 = '\0';
        l = strlen(arg->font_desc[j]);
        if (k==0 ||
            strncasecmp(arg->font_desc[j], arg->font_family[k-1], l)) {
	  if (ptr1 && strcmp(ptr1+1, "unknown")) {
                arg->font_family[k] = 
	            (char *) xmalloc(l+strlen(ptr1+1)+5);
                sprintf(arg->font_family[k], "%s  (%s)", 
                       arg->font_desc[j], ptr1+1);
	    } else
	        arg->font_family[k] = strdup(arg->font_desc[j]);
            ++k;
	}
        if (ptr1) *ptr1 = ':';
        if (ptr2) *ptr2 = ':';
    }

    arg->num_families = k;
    XftFontSetDestroy(fs);

    /*
    **	Init the widgets
     */

    shell = XtVaCreatePopupShell("fontSelect",
				 topLevelShellWidgetClass, Global.toplevel,
				 NULL);
    arg->shell = shell;

    pane = XtVaCreateManagedWidget("pane",
				   panedWidgetClass, arg->shell,
				   XtNborderWidth, 0,
				   NULL);

    label = XtVaCreateManagedWidget("title",
				    labelWidgetClass, pane,
	                            XtNlabel, msgText[FONT_SELECT_DESIRED_PROPERTIES],
				    XtNborderWidth, 0,
				    XtNshowGrip, False,
				    NULL);

    form1 = XtVaCreateManagedWidget("form",
				   formWidgetClass, pane,
				   XtNborderWidth, 0,
				   XtNwidth, 600, XtNheight, 200,
				   NULL);
    arg->form1 = form1;

    /*
    **	lists of items to select
     */

    arg->familyLabel = XtVaCreateManagedWidget("familyLabel",
				    labelWidgetClass, form1,
				    XtNborderWidth, 0,
				    XtNtop, XtChainTop,
				    XtNbottom, XtChainTop,
				    NULL);

    arg->vport0 = XtVaCreateManagedWidget("vport",
				    viewportWidgetClass, form1,
				    XtNwidth, 420, XtNheight, 200,
				    XtNuseBottom, True,
				    XtNuseRight, True,
				    XtNforceBars, False,
				    XtNallowHoriz, False,
				    XtNallowVert, True,
				    XtNfromVert, arg->familyLabel,
				    XtNtop, XtChainTop,
				    NULL);

    arg->family = XtVaCreateManagedWidget("font",
					  listWidgetClass, arg->vport0,
					  XtNverticalList, True,
					  XtNforceColumns, True,
					  XtNdefaultColumns, 1,
					  XtNnumberStrings, 0,
					  NULL);

    arg->weightLabel = XtVaCreateManagedWidget("weightLabel",
				    labelWidgetClass, form1,
				    XtNborderWidth, 0,
				    XtNfromHoriz, arg->vport0,
				    XtNtop, XtChainTop,
				    XtNbottom, XtChainTop,
				    NULL);

    arg->vport1 = XtVaCreateManagedWidget("vport",
				    viewportWidgetClass, form1,
				    XtNwidth, 220, XtNheight, 200,
				    XtNuseBottom, True,
				    XtNuseRight, True,
				    XtNforceBars, False,
				    XtNallowHoriz, False,
				    XtNallowVert, True,
				    XtNfromHoriz, arg->vport0,
				    XtNfromVert, arg->weightLabel,
				    XtNtop, XtChainTop,
				    NULL);

    arg->weight = XtVaCreateManagedWidget("weight",
					  listWidgetClass, arg->vport1,
					  XtNverticalList, True,
					  XtNforceColumns, True,
					  XtNdefaultColumns, 1,
					  XtNnumberStrings, 0,
					  NULL);

    /*
    **	The text area and buttons
     */

    form2 = XtVaCreateManagedWidget("form2",
				   formWidgetClass, pane,
				   XtNborderWidth, 0,
				   NULL);

    arg->subform = XtVaCreateManagedWidget("subForm",
				    formWidgetClass, form2,
				    XtNborderWidth, 0,
				    XtNtop, XtChainTop,
				    XtNbottom, XtChainTop,
				    XtNwidth, 600,
				    XtNheight, 60,
				    NULL);

    XtAppAddActions(XtWidgetToApplicationContext(arg->subform), &dataAct, 1);

    arg->pointSelectLabel = XtVaCreateWidget("pointSelectLabel",
				   labelWidgetClass, arg->subform,
				   XtNborderWidth, 0,
				   XtNleft, XtChainLeft,
				   XtNright, XtChainLeft,
				   NULL);

    arg->pointBar = XtVaCreateManagedWidget("pointBar", 
                                   scrollbarWidgetClass, arg->subform,
				   XtNorientation, XtorientHorizontal,
				   XtNwidth, 328, XtNheight, 20,
				   XtNfromHoriz, arg->pointSelectLabel,
				   XtNleft, XtChainLeft,
				   XtNright, XtChainLeft,
				   NULL);

    arg->pointSelect = XtVaCreateWidget("pointSelect",
					asciiTextWidgetClass, arg->subform,
					XtNleft, XtChainLeft,
					XtNright, XtChainRight,
					XtNfromHoriz, arg->pointBar,
					XtNhorizDistance, 10,
                                        XtNstring, "18",
					XtNeditType, XawtextEdit,
					XtNwrap, XawtextWrapNever,
                                        XtNwidth, 64,
					XtNlength, 8,
					XtNtranslations,
					XtParseTranslationTable("#override\n\
					<Key>Return: input-data-ok()\n\
					<Key>Linefeed: input-data-ok()\n\
					Ctrl<Key>M: input-data-ok()\n\
					Ctrl<Key>J: input-data-ok()\n"),
					NULL);

    arg->rotationLabel = XtVaCreateWidget("rotationLabel",
				          labelWidgetClass, arg->subform,
				          XtNborderWidth, 0,
				          XtNfromVert, arg->pointSelect,
				          XtNleft, XtChainLeft,
				          XtNright, XtChainLeft,
				          NULL);

    arg->rotation = XtVaCreateWidget("rotation",
					asciiTextWidgetClass, arg->subform,
				        XtNfromVert, arg->pointSelect,
	                                XtNfromHoriz, arg->rotationLabel,
					XtNleft, XtChainLeft,
					XtNright, XtChainRight,
					XtNhorizDistance, 10,
                                        XtNstring, "0",
					XtNeditType, XawtextEdit,
					XtNwrap, XawtextWrapNever,
                                        XtNwidth, 64,
					XtNlength, 8,
					XtNtranslations,
					XtParseTranslationTable("#override\n\
					<Key>Return: input-data-ok()\n\
					<Key>Linefeed: input-data-ok()\n\
					Ctrl<Key>M: input-data-ok()\n\
					Ctrl<Key>J: input-data-ok()\n"),
					NULL);

    arg->inclinationLabel = XtVaCreateWidget("inclinationLabel",
				   labelWidgetClass, arg->subform,
				   XtNborderWidth, 0,
				   XtNfromHoriz, arg->rotation,
				   XtNfromVert, arg->pointSelect,
				   XtNhorizDistance, 20,
				   XtNleft, XtChainLeft,
				   XtNright, XtChainLeft,
				   NULL);

    arg->inclination = XtVaCreateWidget("inclination",
					asciiTextWidgetClass, arg->subform,
				        XtNfromVert, arg->pointSelect,
	                                XtNfromHoriz, arg->inclinationLabel,
					XtNleft, XtChainLeft,
					XtNright, XtChainRight,
					XtNhorizDistance, 10,
                                        XtNstring, "0",
					XtNeditType, XawtextEdit,
					XtNwrap, XawtextWrapNever,
                                        XtNwidth, 64,
					XtNlength, 8,
					XtNtranslations,
					XtParseTranslationTable("#override\n\
					<Key>Return: input-data-ok()\n\
					<Key>Linefeed: input-data-ok()\n\
					Ctrl<Key>M: input-data-ok()\n\
					Ctrl<Key>J: input-data-ok()\n"),
					NULL);

    arg->linespacingLabel = XtVaCreateWidget("linespacingLabel",
				   labelWidgetClass, arg->subform,
				   XtNborderWidth, 0,
				   XtNfromHoriz, arg->inclination,
				   XtNfromVert, arg->pointSelect,
				   XtNhorizDistance, 20,
				   XtNleft, XtChainLeft,
				   XtNright, XtChainLeft,
				   NULL);

    arg->linespacing = XtVaCreateWidget("linespacing",
					asciiTextWidgetClass, arg->subform,
				        XtNfromVert, arg->pointSelect,
	                                XtNfromHoriz, arg->linespacingLabel,
					XtNleft, XtChainLeft,
					XtNright, XtChainRight,
					XtNhorizDistance, 10,
                                        XtNstring, "1",
					XtNeditType, XawtextEdit,
					XtNwrap, XawtextWrapNever,
                                        XtNwidth, 64,
					XtNlength, 8,
					XtNtranslations,
					XtParseTranslationTable("#override\n\
					<Key>Return: input-data-ok()\n\
					<Key>Linefeed: input-data-ok()\n\
					Ctrl<Key>M: input-data-ok()\n\
					Ctrl<Key>J: input-data-ok()\n"),
					NULL);

    arg->dilationLabel = XtVaCreateWidget("dilationLabel",
				   labelWidgetClass, arg->subform,
				   XtNborderWidth, 0,
				   XtNfromHoriz, arg->linespacing,
				   XtNfromVert, arg->pointSelect,
				   XtNhorizDistance, 20,
				   XtNleft, XtChainLeft,
				   XtNright, XtChainLeft,
				   NULL);

    arg->dilation = XtVaCreateWidget("dilation",
					asciiTextWidgetClass, arg->subform,
				        XtNfromVert, arg->pointSelect,
	                                XtNfromHoriz, arg->dilationLabel,
					XtNleft, XtChainLeft,
					XtNright, XtChainRight,
					XtNhorizDistance, 10,
                                        XtNstring, "1",
					XtNeditType, XawtextEdit,
					XtNwrap, XawtextWrapNever,
                                        XtNwidth, 64,
					XtNlength, 8,
					XtNtranslations,
					XtParseTranslationTable("#override\n\
					<Key>Return: input-data-ok()\n\
					<Key>Linefeed: input-data-ok()\n\
					Ctrl<Key>M: input-data-ok()\n\
					Ctrl<Key>J: input-data-ok()\n"),
					NULL);

    arg->vport2 = XtVaCreateManagedWidget("vport",
				    viewportWidgetClass, form2,
				    XtNheight, 160,
				    XtNallowResize, True,
				    XtNuseRight, True,
				    XtNforceBars, False,
				    XtNallowHoriz, True,
				    XtNallowVert, True,
				    XtNfromVert, arg->rotation,
				    XtNtop, XtChainTop,
				    XtNbottom, XtChainBottom,
				    NULL);

    arg->text = XtVaCreateManagedWidget("text",
					asciiTextWidgetClass, arg->vport2,
					XtNallowResize, True,
                                     	XtNstring, "",
	                                XtNdisplayCaret, False,
					NULL);

    applyButton = XtVaCreateManagedWidget("apply",
					  commandWidgetClass, form2,
					  XtNfromVert, arg->vport2,
					  XtNtop, XtChainBottom,
					  XtNbottom, XtChainBottom,
					  XtNright, XtChainLeft,
					  XtNleft, XtChainLeft,
					  NULL);

    doneButton = XtVaCreateManagedWidget("done",
					 commandWidgetClass, form2,
					 XtNfromVert, arg->vport2,
					 XtNfromHoriz, applyButton,
					 XtNtop, XtChainBottom,
					 XtNbottom, XtChainBottom,
					 XtNleft, XtChainLeft,
					 XtNright, XtChainLeft,
					 XtNleft, XtChainLeft,
					 NULL);

    arg->paint = paint;
    arg->font_select = NULL;
    arg->xft_name = NULL;
    arg->xft_font = NULL;
    arg->weight_select = NULL;
    arg->pixmap = None;

    arg->weight_list = (char **)xmalloc(sizeof(char*));
    arg->num_weights = 1;
    arg->weight_list[0] = strdup("Regular");
    theArg = arg;
    XawListChange(arg->family, arg->font_family, arg->num_families, 0, True);
    XawListChange(arg->weight, arg->weight_list, 1, 0, True);
    XawScrollbarSetThumb(arg->pointBar, 0.2125, -1.0);

    XtPopup(arg->shell, XtGrabNone);
    XFlush(XtDisplay(arg->shell));
    XtSetMinSizeHints(arg->shell, 360, 480);
    StateSetBusyWatch(False);

    XtUnmanageChild(arg->familyLabel);
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->familyLabel));
    XtUnmanageChild(arg->weightLabel);
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->weightLabel));

    XtUnmanageChild(XtParent(arg->pointSelect));
    XMapWindow(XtDisplay(arg->shell), XtWindow(XtParent(arg->pointSelect)));

    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->pointSelectLabel));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->pointBar));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->pointSelect));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->rotationLabel));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->rotation));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->inclinationLabel));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->inclination));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->dilationLabel));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->dilation));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->linespacingLabel));
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->linespacing));

    XtUnmanageChild(arg->vport2);
    XMapWindow(XtDisplay(arg->shell), XtWindow(arg->vport2));
    XFlush(XtDisplay(arg->shell));

    XtAddCallback(arg->family, XtNcallback, listCallback, (XtPointer) arg);
    XtAddCallback(arg->weight, XtNcallback, listCallback, (XtPointer) arg);
 
    XtAddEventHandler(arg->family, ButtonPressMask, False,
		      (XtEventHandler) mousewheelScroll, (XtPointer) 1);
    XtAddEventHandler(arg->weight, ButtonPressMask, False,
		      (XtEventHandler) mousewheelScroll, (XtPointer) 1);
    XtAddEventHandler(arg->text, ButtonPressMask, False,
		      (XtEventHandler) mousewheelScroll, (XtPointer) 1);

    XtInsertRawEventHandler(arg->family, ButtonPressMask, False,
			    (XtEventHandler) unselectCallback, (XtPointer) arg,
                            XtListHead);
    XtInsertRawEventHandler(arg->weight, ButtonPressMask, False,
			    (XtEventHandler) unselectCallback, (XtPointer) arg,
                            XtListHead);

    XtAddCallback(arg->pointBar, XtNjumpProc,
		  (XtCallbackProc) barCallback, (XtPointer) arg);
    XtAddCallback(arg->pointBar, XtNscrollProc,
		  (XtCallbackProc) scrollCallback, (XtPointer) arg);

    XtAddEventHandler(arg->pointSelect, ButtonPressMask, False,
                      (XtEventHandler) clickCallback, (XtPointer) arg);
    XtAddEventHandler(arg->rotation, ButtonPressMask, False,
                      (XtEventHandler) clickCallback, (XtPointer) arg);
    XtAddEventHandler(arg->dilation, ButtonPressMask, False,
                      (XtEventHandler) clickCallback, (XtPointer) arg);
    XtAddEventHandler(arg->inclination, ButtonPressMask, False,
                      (XtEventHandler) clickCallback, (XtPointer) arg);
    XtAddEventHandler(arg->linespacing, ButtonPressMask, False,
                      (XtEventHandler) clickCallback, (XtPointer) arg);

    XtAddCallback(applyButton, XtNcallback, applyCallback, (XtPointer) arg);
    XtAddCallback(doneButton, XtNcallback, closeCallback, (XtPointer) arg);
    AddDestroyCallback(arg->shell, (DestroyCallbackFunc) closeCallback, arg);
    XtAddEventHandler(arg->shell, StructureNotifyMask, False,
		      (XtEventHandler) fontboxResized, (XtPointer) arg);
}

void *
setDefaultGlobalFont(Display *dpy, char *name)
{
    char *ptr1, *ptr2, *str;
    XftFont *font;
    Global.xft_size = 18;
    font = XftFontOpenName(dpy, DefaultScreen(dpy), name);
    if (!font) {
	Notice(Global.toplevel, msgText[UNABLE_TO_LOAD_REQUESTED_FONT]);
	return NULL;
    }
    if (Global.xft_name)
        free(Global.xft_name);
    Global.xft_name = strdup(name);
    if (Global.xft_font)
        XftFontClose(dpy, (XftFont *)Global.xft_font);
    Global.xft_font = (void *) font;
    Global.xft_height = font->height;
    Global.xft_ascent = font->ascent;
    Global.xft_descent = font->descent;
    Global.xft_maxadv = font->max_advance_width;
    Global.xft_rot = 0;
    Global.xft_linespacing = 1.0;
    str = strdup(name);
    ptr1 = strchr(str, '-');
    if (ptr1) {
        ++ptr1;
        ptr2 = strchr(ptr1, ':');
        if (ptr2) *ptr2 = '\0';
        Global.xft_size = atoi(ptr1);
    } else
        /* guess size from height */
        Global.xft_size = (int)(Global.xft_height * 0.643);
    free(str);
    return (void *) font;
}
