/*-
 * Copyright (c) 2003-2007 Tim Kientzle
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "test.h"
__FBSDID("$FreeBSD$");

/*
 * Exercise symlink recreation.
 */
DEFINE_TEST(test_write_disk_symlink)
{
    static const char data[]="abcdefghijklmnopqrstuvwxyz";
    struct archive *ad;
    struct archive_entry *ae;
    int r;

    if (!canSymlink()) {
        skipping("Symlinks not supported");
        return;
    }

    /* Write entries to disk. */
    assert((ad = archive_write_disk_new()) != NULL);

    /*
     * First, create a regular file then a symlink to that file.
     */

    /* Regular file: link1a */
    assert((ae = archive_entry_new()) != NULL);
    archive_entry_copy_pathname(ae, "link1a");
    archive_entry_set_mode(ae, AE_IFREG | 0755);
    archive_entry_set_size(ae, sizeof(data));
    assertEqualIntA(ad, 0, archive_write_header(ad, ae));
    assertEqualInt(sizeof(data),
        archive_write_data(ad, data, sizeof(data)));
    assertEqualIntA(ad, 0, archive_write_finish_entry(ad));
    archive_entry_free(ae);

    /* Symbolic Link: link1b -> link1a */
    assert((ae = archive_entry_new()) != NULL);
    archive_entry_copy_pathname(ae, "link1b");
    archive_entry_set_mode(ae, AE_IFLNK | 0642);
    archive_entry_set_size(ae, 0);
    archive_entry_copy_symlink(ae, "link1a");
    assertEqualIntA(ad, 0, r = archive_write_header(ad, ae));
    if (r >= ARCHIVE_WARN)
        assertEqualIntA(ad, 0, archive_write_finish_entry(ad));
    archive_entry_free(ae);

    /*
     * We should be able to do this in the other order as well,
     * of course.
     */

    /* Symbolic link: link2b -> link2a */
    assert((ae = archive_entry_new()) != NULL);
    archive_entry_copy_pathname(ae, "link2b");
    archive_entry_set_mode(ae, AE_IFLNK | 0642);
    archive_entry_unset_size(ae);
    archive_entry_copy_symlink(ae, "link2a");
    assertEqualIntA(ad, 0, r = archive_write_header(ad, ae));
    if (r >= ARCHIVE_WARN) {
        assertEqualInt(ARCHIVE_WARN,
            archive_write_data(ad, data, sizeof(data)));
        assertEqualIntA(ad, 0, archive_write_finish_entry(ad));
    }
    archive_entry_free(ae);

    /* File: link2a */
    assert((ae = archive_entry_new()) != NULL);
    archive_entry_copy_pathname(ae, "link2a");
    archive_entry_set_mode(ae, AE_IFREG | 0755);
    archive_entry_set_size(ae, sizeof(data));
    assertEqualIntA(ad, 0, archive_write_header(ad, ae));
    assertEqualInt(sizeof(data),
        archive_write_data(ad, data, sizeof(data)));
    assertEqualIntA(ad, 0, archive_write_finish_entry(ad));
    archive_entry_free(ae);

    assertEqualInt(ARCHIVE_OK, archive_write_finish(ad));

    /* Test the entries on disk. */

    /* Test #1 */
    assertIsReg("link1a", -1);
    assertFileSize("link1a", sizeof(data));
    assertFileNLinks("link1a", 1);
    assertIsSymlink("link1b", "link1a");

    /* Test #2: Should produce identical results to test #1 */
    assertIsReg("link2a", -1);
    assertFileSize("link2a", sizeof(data));
    assertFileNLinks("link2a", 1);
    assertIsSymlink("link2b", "link2a");
}
