/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.SequenceInputStream;
import java.util.ArrayList;
import java.util.Vector;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.mapred.JobConf;

class TaskLog {
    private static final Log LOG = LogFactory.getLog((String)TaskLog.class.getName());
    private static final File LOG_DIR = new File(System.getProperty("hadoop.log.dir"), "userlogs");
    private static final String SPLIT_INDEX_NAME = "split.idx";

    TaskLog() {
    }

    private static File getTaskLogDir(String taskid, LogFilter filter) {
        return new File(new File(LOG_DIR, taskid), filter.getPrefix());
    }

    static {
        if (!LOG_DIR.exists()) {
            LOG_DIR.mkdirs();
        }
    }

    static class Reader {
        private String taskId;
        private LogFilter filter;
        private File taskLogDir;
        private boolean initialized = false;
        private IndexRecord[] indexRecords = null;
        private BufferedReader splitIndex;
        private long logFileSize = 0L;

        public Reader(String taskId, LogFilter filter) {
            this.taskId = taskId;
            this.filter = filter;
            this.taskLogDir = TaskLog.getTaskLogDir(this.taskId, this.filter);
        }

        private synchronized void init() throws IOException {
            String line;
            this.splitIndex = new BufferedReader(new InputStreamReader(new FileInputStream(new File(this.taskLogDir, TaskLog.SPLIT_INDEX_NAME))));
            ArrayList<IndexRecord> records = new ArrayList<IndexRecord>();
            while ((line = this.splitIndex.readLine()) != null) {
                String[] fields = line.split("\\|");
                if (fields.length != 3) {
                    throw new IOException("Malformed split-index with " + fields.length + " fields");
                }
                IndexRecord record = new IndexRecord(fields[0], Long.valueOf(fields[1]), Long.valueOf(fields[2]));
                LOG.debug((Object)("Split: <" + record.splitName + ", " + record.splitOffset + ", " + record.splitLength + ">"));
                records.add(record);
                this.logFileSize += record.splitLength;
            }
            this.indexRecords = new IndexRecord[records.size()];
            this.indexRecords = records.toArray(this.indexRecords);
            this.initialized = true;
            LOG.debug((Object)("Log size: " + this.logFileSize));
        }

        public long getTotalLogSize() throws IOException {
            if (!this.initialized) {
                this.init();
            }
            return this.logFileSize;
        }

        public byte[] fetchAll() throws IOException {
            if (!this.initialized) {
                this.init();
            }
            Vector<InputStream> streams = new Vector<InputStream>();
            int totalLogSize = 0;
            for (int i = 0; i < this.indexRecords.length; ++i) {
                InputStream stream = this.getLogSplit(i);
                if (stream == null) continue;
                streams.add(stream);
                totalLogSize = (int)((long)totalLogSize + this.indexRecords[i].splitLength);
                LOG.debug((Object)("Added split: " + i));
            }
            LOG.debug((Object)("Total log-size on disk: " + totalLogSize + "; actual log-size: " + this.logFileSize));
            byte[] b = new byte[totalLogSize];
            SequenceInputStream in = new SequenceInputStream(streams.elements());
            int bytesRead = 0;
            int totalBytesRead = 0;
            int off = 0;
            int len = totalLogSize;
            LOG.debug((Object)("Attempting to read " + len + " bytes from logs"));
            while ((bytesRead = in.read(b, off, len)) > 0) {
                LOG.debug((Object)("Got " + bytesRead + " bytes"));
                off += bytesRead;
                len -= bytesRead;
                totalBytesRead += bytesRead;
            }
            if (totalBytesRead != totalLogSize) {
                LOG.debug((Object)"Didn't not read all requisite data in logs!");
            }
            return b;
        }

        public synchronized int tail(byte[] b, int off, int len, long tailSize, int tailWindow) throws IOException {
            if (!this.initialized) {
                this.init();
            }
            LOG.debug((Object)("tailSize: " + tailSize + " - tailWindow: " + tailWindow));
            if (tailSize * (long)tailWindow > this.logFileSize) {
                tailSize = this.logFileSize;
                tailWindow = 1;
            }
            return this.read(b, off, len, this.logFileSize - tailSize * (long)tailWindow, tailSize);
        }

        public synchronized int read(byte[] b, int off, int len, long logOffset, long logLength) throws IOException {
            LOG.debug((Object)("TaskLog.Reader.read: logOffset: " + logOffset + " - logLength: " + logLength));
            if (logLength == 0L) {
                return 0;
            }
            if (!this.initialized) {
                this.init();
            }
            Vector<InputStream> streams = new Vector<InputStream>();
            long offset = logOffset;
            int startIndex = -1;
            int stopIndex = -1;
            boolean inRange = false;
            for (int i = 0; i < this.indexRecords.length; ++i) {
                InputStream stream;
                LOG.debug((Object)("offset: " + offset + " - (split, splitOffset) : (" + i + ", " + this.indexRecords[i].splitOffset + ")"));
                if (offset <= this.indexRecords[i].splitOffset) {
                    if (!inRange) {
                        startIndex = i - (i > 0 ? 1 : 0);
                        LOG.debug((Object)("Starting at split: " + startIndex));
                        offset += logLength;
                        stream = this.getLogSplit(startIndex);
                        if (stream != null) {
                            streams.add(stream);
                        }
                        LOG.debug((Object)("Added split: " + startIndex));
                        inRange = true;
                    } else {
                        stopIndex = i - 1;
                        LOG.debug((Object)("Stop at split: " + stopIndex));
                        break;
                    }
                }
                if (!inRange) continue;
                stream = this.getLogSplit(i);
                if (stream != null) {
                    streams.add(stream);
                }
                LOG.debug((Object)("Added split: " + i));
            }
            if (startIndex == -1) {
                throw new IOException("Illegal logOffset/logLength");
            }
            if (stopIndex == -1) {
                stopIndex = this.indexRecords.length - 1;
                LOG.debug((Object)("Stop at split: " + stopIndex));
                if (logOffset + logLength > this.logFileSize) {
                    LOG.debug((Object)"logOffset+logLength exceeds log-file size");
                    logLength = this.logFileSize - logOffset;
                }
            }
            SequenceInputStream in = new SequenceInputStream(streams.elements());
            if (streams.size() == stopIndex - startIndex + 1) {
                long skipBytes = in.skip(logOffset - this.indexRecords[startIndex].splitOffset);
                LOG.debug((Object)("Skipped " + skipBytes + " bytes from " + startIndex + " stream"));
            }
            int bytesRead = 0;
            int totalBytesRead = 0;
            len = Math.min((int)logLength, len);
            LOG.debug((Object)("Attempting to read " + len + " bytes from logs"));
            while ((bytesRead = in.read(b, off, len)) > 0) {
                off += bytesRead;
                len -= bytesRead;
                totalBytesRead += bytesRead;
            }
            return totalBytesRead;
        }

        private synchronized InputStream getLogSplit(int split) throws IOException {
            String splitName = this.indexRecords[split].splitName;
            LOG.debug((Object)("About to open the split: " + splitName));
            BufferedInputStream in = null;
            try {
                in = new BufferedInputStream(new FileInputStream(new File(splitName)));
            }
            catch (FileNotFoundException fnfe) {
                in = null;
                LOG.debug((Object)("Split " + splitName + " not found... probably purged!"));
            }
            return in;
        }

        private static class IndexRecord {
            String splitName;
            long splitOffset;
            long splitLength;

            IndexRecord(String splitName, long splitOffset, long splitLength) {
                this.splitName = splitName;
                this.splitOffset = splitOffset;
                this.splitLength = splitLength;
            }
        }
    }

    static class Writer {
        private String taskId;
        private JobConf conf;
        private LogFilter filter;
        private final File taskLogDir;
        private final int noKeepSplits;
        private final long splitFileSize;
        private final boolean purgeLogSplits;
        private final int logsRetainHours;
        private boolean initialized = false;
        private long splitOffset = 0L;
        private long splitLength = 0L;
        private int noSplits = 0;
        private File currentSplit;
        private OutputStream out;
        private OutputStream splitIndex;
        private int flushCtr = 0;
        private static final int FLUSH_BYTES = 256;

        Writer(String taskId, LogFilter filter, int noKeepSplits, long totalLogSize, boolean purgeLogSplits, int logsRetainHours) {
            this.taskId = taskId;
            this.filter = filter;
            this.taskLogDir = TaskLog.getTaskLogDir(this.taskId, this.filter);
            this.noKeepSplits = noKeepSplits;
            this.splitFileSize = totalLogSize / (long)noKeepSplits;
            this.purgeLogSplits = purgeLogSplits;
            this.logsRetainHours = logsRetainHours;
        }

        private File getLogSplit(int split) {
            String splitName = "part-" + String.format("%1$06d", split);
            return new File(this.taskLogDir, splitName);
        }

        private void deleteDir(File dir) throws IOException {
            File[] files = dir.listFiles();
            if (files != null) {
                for (int i = 0; i < files.length; ++i) {
                    if (files[i].isDirectory()) {
                        this.deleteDir(files[i]);
                    }
                    files[i].delete();
                }
            }
            boolean del = dir.delete();
            LOG.debug((Object)("Deleted " + dir + ": " + del));
        }

        public synchronized void init() throws IOException {
            if (!this.initialized) {
                long purgeTimeStamp = System.currentTimeMillis() - (long)(this.logsRetainHours * 60 * 60 * 1000);
                File[] oldTaskLogs = LOG_DIR.listFiles(new TaskLogsPurgeFilter(purgeTimeStamp));
                if (oldTaskLogs != null) {
                    for (int i = 0; i < oldTaskLogs.length; ++i) {
                        this.deleteDir(oldTaskLogs[i]);
                    }
                }
                if (this.taskLogDir.exists()) {
                    this.deleteDir(this.taskLogDir);
                }
                this.taskLogDir.mkdirs();
                this.splitIndex = new BufferedOutputStream(new FileOutputStream(new File(this.taskLogDir, TaskLog.SPLIT_INDEX_NAME)));
                this.out = this.createLogSplit(this.noSplits);
                this.initialized = true;
            }
        }

        public synchronized void write(byte[] b, int off, int len) throws IOException {
            if (this.splitLength > this.splitFileSize) {
                LOG.debug((Object)("Total no. of bytes written to split#" + this.noSplits + " -> " + this.splitLength));
                this.logRotate();
            }
            if (this.flushCtr > 256) {
                this.out.flush();
                this.flushCtr = 0;
            }
            this.out.write(b, off, len);
            this.splitLength += (long)len;
            this.flushCtr += len;
        }

        public synchronized void close() throws IOException {
            if (this.out != null) {
                this.out.close();
            }
            if (this.splitIndex != null) {
                this.writeIndexRecord();
                this.splitIndex.close();
            }
        }

        private synchronized OutputStream createLogSplit(int split) throws IOException {
            this.currentSplit = this.getLogSplit(split);
            LOG.debug((Object)("About to create the split: " + this.currentSplit));
            return new BufferedOutputStream(new FileOutputStream(this.currentSplit));
        }

        private synchronized void writeIndexRecord() throws IOException {
            String indexRecord = new String(this.currentSplit + "|" + this.splitOffset + "|" + this.splitLength + "\n");
            this.splitIndex.write(indexRecord.getBytes());
            this.splitIndex.flush();
        }

        private synchronized void logRotate() throws IOException {
            LOG.debug((Object)("About to rotate-out the split: " + this.noSplits));
            this.out.close();
            this.writeIndexRecord();
            this.splitOffset += this.splitLength;
            this.splitLength = 0L;
            this.flushCtr = 0;
            ++this.noSplits;
            if (this.purgeLogSplits && this.noSplits >= this.noKeepSplits) {
                File purgeLogSplit = this.getLogSplit(this.noSplits - this.noKeepSplits);
                purgeLogSplit.delete();
                LOG.debug((Object)("Purged log-split #" + (this.noSplits - this.noKeepSplits) + " - " + purgeLogSplit));
            }
            this.out = this.createLogSplit(this.noSplits);
        }

        private static class TaskLogsPurgeFilter
        implements FileFilter {
            long purgeTimeStamp;

            TaskLogsPurgeFilter(long purgeTimeStamp) {
                this.purgeTimeStamp = purgeTimeStamp;
            }

            public boolean accept(File file) {
                LOG.debug((Object)("PurgeFilter - file: " + file + ", mtime: " + file.lastModified() + ", purge: " + this.purgeTimeStamp));
                return file.lastModified() < this.purgeTimeStamp;
            }
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum LogFilter {
        STDOUT("stdout"),
        STDERR("stderr"),
        SYSLOG("syslog");

        private String prefix;

        private LogFilter(String prefix) {
            this.prefix = prefix;
        }

        String getPrefix() {
            return this.prefix;
        }
    }
}

