#!/usr/bin/env python

# Copyright (c) 2013. The EFIDIR team. All right reserved.
#
# This file is part of EFIDIR tools.
#
# EFIDIR tool(s) is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# EFIDIR tool(s) is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public licence
# along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
#
# author: Matthieu Volat / ISTerre
#

import datetime, fnmatch, os, xml.etree.ElementTree, glob
#import datetime, glob, os, xml.etree.ElementTree

def infos_from_xml(xmlpath):
	infos = {}
	tree = xml.etree.ElementTree.ElementTree()
	tree.parse(xmlpath)
	time = tree.find("*/sceneInfo/start/timeUTC")
	orbit_direction = tree.find("*/missionInfo/orbitDirection")
	cos_dirname = tree.find("*/imageData/file/location/path")
	cos_filename = tree.find("*/imageData/file/location/filename")
	if (time == None or orbit_direction == None or cos_filename == None):
		return None
	infos["time"] = datetime.datetime.strptime(time.text,
	                                           "%Y-%m-%dT%H:%M:%S.%fZ")
	infos["orbit_direction"] = orbit_direction.text
	infos["cos"] = os.path.join(os.path.dirname(xmlpath),
                                cos_dirname.text,
                                cos_filename.text)
	return infos

def findinfos(date, where):
	# Read date
	time = datetime.datetime.strptime(date, "%Y%m%d")
	# Find *ALL* XML files in directory
	xmls = []
	for dirpath, dirnames, filenames in os.walk(where):
		# Filter a few dir/files that are assured to *NOT* host the xml file
		# but whom reading increase a bit too much the search time
		if "ANNOTATION" in dirnames:
			dirnames.remove("ANNOTATION")
		# Append wathever is left 
		for filename in fnmatch.filter(filenames, '*.xml'):
			xmls.append(os.path.join(dirpath, filename))
	# Filter out XML files that are not acquisitions
	for xml in xmls:
		infos = infos_from_xml(xml)
		if infos == None:
			continue
		infos["xml"] = xml
		if (infos["time"] - time).days == 0:
			return infos
	raise Exception("Information not found for date "+date+" in "+where)


def readinfos(path):
    infos = {}
    # Try to find the XML file
    xmlpath = glob.glob(path+"/TSX-1.SAR.L1B/T[DS]X*/*.xml")[0]
    # Read data
    tree = xml.etree.ElementTree.ElementTree()
    tree.parse(xmlpath)
    time = tree.find("*/sceneInfo/start/timeUTC")
    orbit_direction = tree.find("*/missionInfo/orbitDirection")
    cos_dirname = tree.find("*/imageData/file/location/path")
    cos_filename = tree.find("*/imageData/file/location/filename")
    if (time == None or orbit_direction == None or cos_filename == None):
        return None
    infos["xml"] = xmlpath
    infos["time"] = datetime.datetime.strptime(time.text,
                                               "%Y-%m-%dT%H:%M:%S.%fZ")
    infos["orbit_direction"] = orbit_direction.text
    infos["cos"] = os.path.join(os.path.dirname(xmlpath),
                                cos_dirname.text,
                                cos_filename.text)
    return infos
