/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup sarlut_operators
 * \defgroup simu_tools Simulations tools
 */
/**
 * \ingroup simu_tools
 * \file slt_simu.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief operators of simulation
 * 
 * file includes (stucture, prototype de fonction) of the operator simu_tools
 * 
 * 
 */

#ifndef SLT_SIMU_H
#define SLT_SIMU_H

#include "efidir_image.h"
#include "slt_calculations_geodetic.h"
#include "slt_orbit_calculations.h"

#define NOM_FICH 256
#define MXI 32767

/**
 * \ingroup simu_tools
 * \struct simu_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct
{
	int verbose; 	// mode bavard :
			// 0=non, 1=verbose (on affiche tous les résultats intermédiaires interprétables), 2=debug
	int inut_read, inut_write; // pixels inutilisables
	int type_traj; // format de la trajectoire à lire
/*	double lat, lon, pas_lat, pas_lon;
	double rot, rot_lat, rot_lon;*/
	float rotdeg; // rotation en degrés
	TMapInfo mapinfoInit, mapinfo; // lat,long,size_lat,size_long,rot. Init pour le MNT avant transformation.
	TMapFile mapfileInit, mapfile; // lat,long,pas_lat,pas_long,rot_lat, rot_long. Init pour le MNT avant transformation.
	int nblig, nbcol, nbligInit, nbcolInit; // nombre de lignes et colonnes des images de travail avant et après rééchantillonnage
	matrice_3x3 matrice_RST_donnees;
	char lat_orientation[2]; // Est E ou Ouest W
	char *rit_fichier_georef; // rit pour read_ima_t(nom et extension) du fichier de georeferencement
	char *Nom_fichier_traj; // nom du fichier de la trajectoire
	char *Nom_fichier_log; // nom du fichier d'info complémentaires
	coord_3d *traj; // points de la trajectoire initiale issus du fichier de la trajectoire
	coord_3d *source; // points 
	droite_par_points_3D droite_traj;
	int taille_traj; // nombre de point de la trajectoire (brut, avant interpolation ou rééchantillonnage sur le MNT)
	char *passe; 
	float rt, hmax; //hmax est l'altitude maximale du MNT initial
	double hS; // altitude du satellite (utile dans le cas satellitaire sans trajectoire)
	double Xs, Zs; // coordonnées du satellite (utile dans le cas satellitaire sans trajectoire)
	double *px;
	double py;
	double beta; // beta est l'angle en radian entre l'axe des Z et (centre de la terre, point courant)
	double alpha_min;	// angle d'émission minimale du satellite (utile dans le cas satellitaire sans trajectoire)
	EFIDIRImageFloat tan_gamma;	//dans T1
	EFIDIRImageFloat dzdx,dzdy;	//dans T6
	EFIDIRImageFloat theta;
	EFIDIRImageFloat w;		//dans T1
	EFIDIRImageFloat wx, wy;		//dans T6
	EFIDIRImageFloat ro, romoins, roplus;
	float romin, romax;  
	float *romi;		//dans T7
	float *roma;		//dans T7
//	imafl romi, roma, buf;
	float pas_sl, pas_azimuth;
	char  *interpol;	//dans T4
	char  *mask_fold;	//dans T4 : rep "N" ou la valeur de remplacement 
	float v_fold;		//dans T4 : valeur de remplacement pour les zones de mask en foldover
	char  *mask_ombre;	//dans T4 : rep "N" ou la valeur de remplacement 
	float v_ombre;		//dans T4 : valeur de remplacement pour les zones de mask en  ombre
	EFIDIRImageFloat buf2;		//dans T4
	EFIDIRImageInt ind;		//dans T4
	TLatLong lloffset;	//dans T8
	int taille_imRadar;	//dans T8

	/* Used in order to compute a precise distance to the orbit
	 * using the polynomial interpolation */
	Estimators estim;
	/* This is the number of points on the orbit that will be used in interpolation process*/
	int nof_estims;

	/* This is the polynomial degree used for interpolation in orbit emulation process */
	int polynomial_degree;

} simu_param, *Simu_param;


/* ***********************************  FONCTIONS ***********************************/

// trajectoire
/**
 * \ingroup simu_tools
 * \fn int init_trajectoire(Simu_param des, EFIDIRImageFloat imMNT);
 * \brief function of initialisation of track of radar
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMNT is DEM used for simulation
 * \return 0 if no error occurs
 *
 */
int init_trajectoire(Simu_param des, EFIDIRImageFloat imMNT);

/**
 * \ingroup simu_tools
 * \fn int lire_traj_carac(Simu_param des);
 * \brief function of lecture of characteristic of  track file of radar
 * This file contains two parameters: angle of emission and average altitude of radar 
 *
 * \param des is the structure of the parameters for the operator 
 * \return 0 if no error occurs
 *
 */
int lire_traj_carac(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int lire_traj_coord(Simu_param des);
 * \brief function of lecture of precise coordonate of radar 
 *
 * \param des is the structure of the parameters for the operator 
 * \return 0 if no error occurs
 *
 */
int lire_traj_coord(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int lire_traj_dlr(Simu_param des);
 * \brief function of lecture of binary file including the coordonates of DLR aircraft
 *
 * \param des is the structure of the parameters for the operator 
 * \return 0 if no error occurs
 *
 */
int lire_traj_dlr(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int lire_traj_orbite(Simu_param des);
 * \brief function of lecture of ASCII file of satellite orbit
 *
 * \param des is the structure of the parameters for the operator 
 * \return 0 if no error occurs
 *
 */
int lire_traj_orbite(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int lire_traj_xml_orbite_fine(Simu_param des);
 * \brief This is the function used in order to compute the estimators of the trajectory. It updates the estim field of the Simu_param operator considering the state vectors and some other informations extracted from the xml file.
 *
 * \param des is the structure of the parameters for the operator
 * \return 0 if no error occurs
 *
 */
int lire_traj_xml_orbite_fine(Simu_param des);


/**
 * \ingroup simu_tools
 * \fn int lire_traj_roi_pac_orbite_fine(Simu_param des);
 * \brief This is the function used in order to compute the estimators of the trajectory It updates the estim field of the Simu_param operator considering the values extracted from the ROI-PAC file
 *
 * \param des is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int lire_traj_roi_pac_orbite_fine(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn coord_4d *lire_traj_local_dlr(Simu_param des);
 * \brief function of lecture of local cartesian coordonate of aircraft
 *
 * \param des is the structure of the parameters for the operator 
 * \return coord_4d A local cartesian coordonate of trajectory
 *
 */
coord_4d *lire_traj_local_dlr(Simu_param des);

//géoréférencement
/**
 * \ingroup simu_tools
 * \fn int read_map_lect(Simu_param des);
 * \brief function of lecture of geocode file
 *
 * \param des is the structure of the parameters for the operator 
 * \return 0 if no error occurs
 *
 */
int read_map_lect(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int read_MapInfo(char *nom_fichier_georef, TMapInfo *map_info);
 * \brief function of lecture of MapInfo
 *
 * \param nom_fichier_georef is the geocode file
 * \param map_info is the geocode information
 * \return 0 if no error occurs
 *
 */
int read_MapInfo(char *nom_fichier_georef, TMapInfo *map_info);

/**
 * \ingroup simu_tools
 * \fn int read_MapFile(char *nom_fichier_georef, TMapFile *map_file);
 * \brief function of lecture of Map file
 *
 * \param nom_fichier_georef is the geocode file
 * \param map_file is the geocode information
 * \return 0 if no error occurs
 *
 */
int read_MapFile(char *nom_fichier_georef, TMapFile *map_file);

/**
 * \ingroup simu_tools
 * \fn int write_MapInfo(TMapInfo map_info, char *nom_mapinfo);
 * \brief function of writing of MapInfo
 *
 * \param map_info is the geocode information
 * \param nom_mapinfo is the MapInfo file to write
 * \return 0 if no error occurs
 *
 */
int write_MapInfo(TMapInfo map_info, char *nom_mapinfo);

/**
 * \ingroup simu_tools
 * \fn int write_MapFile(TMapFile map_file, char *nom_mapfile);
 * \brief function of writing of Mapfile
 *
 * \param map_file is the geocode information
 * \param nom_mapfile is the Mapfile to write
 * \return 0 if no error occurs
 *
 */
int write_MapFile(TMapFile map_file, char *nom_mapfile);

/**
 * \ingroup simu_tools
 * \fn int hdr2georef(char *nom_georef, char *nom_hdr);
 * \brief function of generation of geocode file from ENVI image header
 *
 * \param nom_georef is the file containing geocode information to write
 * \param nom_hdr is the ENVI image header
 * \return 0 if no error occurs
 *
 */
int hdr2georef(char *nom_georef, char *nom_hdr);

/**
 * \ingroup simu_tools
 * \fn void affiche_MapInfo(TMapInfo map_info);
 * \brief display function of MapInfo
 *
 * \param map_info is the geocode information
 *
 */
void affiche_MapInfo(TMapInfo map_info);

/**
 * \ingroup simu_tools
 * \fn void affiche_MapFile(TMapFile map_file);
 * \brief display function of Mapfile
 *
 * \param map_file is the geocode information
 *
 */
void affiche_MapFile(TMapFile map_file);
// int chmap(simu_param *des, char *nom_georef, double new_rot, double reechX, double reechY, int origX, int origY);
/**
 * \ingroup simu_tools
 * \fn int chmap(Simu_param des, int mode, float info);
 * \brief function of modification of geocode information following image processing
 *
 * \param des is the structure of the parameters for the operator 
 * \param mode 0 - rotation 1- flip (left/right)
 * \param info is the rotation angle or number of column of image to flip
 * \return 0 if no error occurs
 *
 */
int chmap(Simu_param des, int mode, float info);

/**
 * \ingroup simu_tools
 * \fn void write_map_info(EFIDIRImage image, TMapInfo map_info);
 * \brief writing geocode information of an image in .georef file
 *
 * \param image is the image whose header including geocode inforamtion
 * \param map_info is the name of .georef file to write
 *
 */
void write_map_info(EFIDIRImage image, TMapInfo map_info);

// positionnement par rapport au georeferencement
/**
 * \ingroup simu_tools
 * \fn TLatLong ij2lldeg(TMapFile mapfile, float coordi, float coordj);
 * \brief function of calculation of coordonate LAT/LON after rotation
 *
 * \param mapfile is the geocode information file
 * \param coordi is number of line
 * \param coordj is number of column
 * \return A coordonate LAT/LON 
 *
 */
TLatLong ij2lldeg(TMapFile mapfile, float coordi, float coordj);

/**
 * \ingroup simu_tools
 * \fn point_2d lldeg2ij(TMapFile mapfile, TLatLong llA);
 * \brief function of calculation of coordonate LAT/LON after from mapfile
 *
 * \param mapfile is the geocode information file
 * \param llA is the LAT/LON structure
 * \return A coordonate LAT/LON 
 *
 */
point_2d lldeg2ij(TMapFile mapfile, TLatLong llA);

/**
 * \ingroup simu_tools
 * \fn void init_pas(Simu_param des, EFIDIRImageFloat imMNT);
 * \brief function of initialisation of pixel spacing of DEM
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMNT is the DEM image
 *
 */
void init_pas(Simu_param des, EFIDIRImageFloat imMNT);

// calcul dans le cas aéroporté ou satellitaire avec trajectoire
/**
 * \ingroup simu_tools
 * \fn void calcul_coord_source_3D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of coordonates for each line of acquisition of DEM
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_coord_source_3D(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void compute_coord_3D_source_with_precision(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief This function is used in order to compute the coordinates of the CPA for each first pixel of each line.
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator
 *
 */
void compute_coord_3D_source_with_precision(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void calcul_theta_3D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of incidence angle
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_theta_3D(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void distances_source_3D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of distance of sensor
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void distances_source_3D(EFIDIRImageFloat imMNT, Simu_param des);

// divers
/**
 * \ingroup simu_tools
 * \fn void calcul_gamma(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of tan of local slope
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_gamma(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void calcul_gamma(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of tan of local slope in surface on Ox
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_gammasurf(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void calcul_orientation_traj_naive(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation orientation of trajectory
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_orientation_traj_naive(EFIDIRImageFloat imMNT, Simu_param des);

// calcul dans le cas satellitaire sans infos de trajectoire
/**
 * \ingroup simu_tools
 * \fn void calcul_coord_source_2D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of the coordonate of satellite
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void calcul_coord_source_2D(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void calcul_theta_2D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of incidence angle (with respecto to the vertical)
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */

void calcul_theta_2D(EFIDIRImageFloat imMNT, Simu_param des);

/**
 * \ingroup simu_tools
 * \fn void distances_source_2D(EFIDIRImageFloat imMNT, Simu_param des);
 * \brief function of calculation of the distance between the sensor and the ground
 *
 * \param imMNT is the DEM image
 * \param des is the structure of the parameters for the operator 
 *
 */
void distances_source_2D(EFIDIRImageFloat imMNT, Simu_param des);

// fonctions principales pour la simulation
/**
 * \ingroup simu_tools
 * \fn int param2mapInfo(Simu_param des, EFIDIRImageFloat imMNT);
 * \brief function of determination of new parameters for image simulation 
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMNT is the DEM image
 * \return 0 if no error occurs
 *
 */
int param2mapInfo(Simu_param des, EFIDIRImageFloat imMNT);

/**
 * \ingroup simu_tools
 * \fn int simu_init(Simu_param des, EFIDIRImageFloat imMNT);
 * \brief function of pre-calculation
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMNT is the DEM image
 * \return 0 if no error occurs
 *
 */
int simu_init(Simu_param des, EFIDIRImageFloat imMNT);

/**
 * \ingroup simu_tools
 * \fn void latlong_ground_init(TMapFile mapfile, EFIDIRImageFloat imLAT, EFIDIRImageFloat imLON);
 * \brief function of initialisation of latitude and longitude images from geocode information
 *
 * \param mapfile is the geocode information file
 * \param imLAT is the latitude image
 * \param imLON is the longitude image
 *
 */
void latlong_ground_init(TMapFile mapfile, EFIDIRImageFloat imLAT, EFIDIRImageFloat imLON);

// gestion de la memoire
/**
 * \ingroup simu_tools
 * \fn void free_3D_t(Simu_param des);
 * \brief free allocated memories
 *
 * \param des is the structure of the parameters for the operator
 *
 */
void free_3D_t(Simu_param des);

// transformations des données
/**
 * \ingroup simu_tools
 * \fn int donnees_init_par_RST(Simu_param des);
 * \brief function of pre-calculation of parameters for RST
 *
 * \param des is the structure of the parameters for the operator
 * \return 0 if no error occurs
 *
 */

int donnees_init_par_RST(Simu_param des);

/**
 * \ingroup simu_tools
 * \fn int donnees_calc_par_RST_u1(Simu_param des, EFIDIRImageByte image);
 * \brief function of application of RST transformation
 *
 * \param des is the structure of the parameters for the operator
 * \param image is the original 8 bytes image to be transformated
 * \return 0 if no error occurs
 *
 */
int donnees_calc_par_RST_u1(Simu_param des, EFIDIRImageByte image);

/**
 * \ingroup simu_tools
 * \fn int  donnees_calc_par_RST_fl(Simu_param des, EFIDIRImageFloat image);
 * \brief function of application of RST transformation
 *
 * \param des is the structure of the parameters for the operator
 * \param image is the original float image to be transformated
 * \return 0 if no error occurs
 *
 */
int donnees_calc_par_RST_fl(Simu_param des, EFIDIRImageFloat image);

/**
 * \ingroup simu_tools
 * \fn int donnees_ferm_par_RST_fl(Simu_param des, EFIDIRImageFloat imRes, EFIDIRImageFloat imResTmp);
 * \brief function of application of RST transformation
 *
 * \param des is the structure of the parameters for the operator
 * \param imRes is the resultant float image to be transformated
 * \param imResTmp is a buffer image
 * \return 0 if no error occurs
 *
 */
int donnees_ferm_par_RST_fl(Simu_param des, EFIDIRImageFloat imRes, EFIDIRImageFloat imResTmp);

/**
 * \ingroup simu_tools
 * \fn int donnees_ferm_par_RST_u1(Simu_param des, EFIDIRImageByte imRes, EFIDIRImageByte imResTmp);
 * \brief function of application of RST transformation
 *
 * \param des is the structure of the parameters for the operator
 * \param imRes is the resultant 8 bytes image to be transformated
 * \param imResTmp is a buffer image
 * \return 0 if no error occurs
 *
 */
int donnees_ferm_par_RST_u1(Simu_param des, EFIDIRImageByte imRes, EFIDIRImageByte imResTmp);

#endif
