/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * \ingroup sarlut_operators
 * \defgroup registration Registration of 2 images by correlation
 */
/**
 * \ingroup registration
 * \file slt_recal_correl.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief operator of registration of 2 images by correlation
 * 
 * file includes (stucture, prototype de fonction) of the operator registration
 * 
 * 
 */

#ifndef SLT_RECAL_CORREL_H
#define SLT_RECAL_CORREL_H

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include "slt_geom2D.h"

/* modif Yajing YAN */
//#define NOM_FICH 50
#define NOM_FICH 256
#define MAX_NB_GCP 50
/* modif Yajing YAN */

/**
 * \ingroup registration
 * \struct zone_recalage_param
 *
 * \brief structure that contains the parameters of the points used for image registration
 *
 */
typedef struct
{
	int baseX, baseY, baseTailleX, baseTailleY; // sur l'image de référence : 4 coordonnées: X,Y du point en haut à gauche, et tailles en X et Y de la fenêtre
	int warpX, warpY, warpTailleX, warpTailleY; // sur l'image à recaler : 4 coordonnées: X,Y du point en haut à gauche, et tailles en X et Y de la fenêtre
	
} zone_recalage_param, *Zone_recalage_param;

/**
 * \ingroup registration
 * \struct appariement_param
 *
 * \brief structure that contains the parameters of the points of reference image
 *
 */

typedef struct
{
	float baseX, baseY, warpX, warpY; // sur l'image de référence : 4 coordonnées: X,Y du point en haut à gauche, et tailles en X et Y de la fenêtre
	double valeur_correl;
} appariement_param, *Appariement_param;


/**
 * \ingroup registration
 * \struct recal_correl_param
 *
 * \brief structure that contains the parameters of operator for image registration
 *
 */
typedef struct
{
	int verbose;
	float val_inut_read, val_inut_write; // valeurs inutilisables à lire et à écrire
	char *nom_fichier_zones;
	int nombre_zones;
	zone_recalage_param *def_zones_recal;		// "nombre_zones" zones, 4 coordonnées: X,Y du point en haut à gauche, et tailles en X et Y de la fenêtre
	char *nom_fichier_GCP;
	char nom_fichier_appariements[NOM_FICH-10];
	int imcorrel_nr, imcorrel_nc;	// taille de la zone resultat à parcourir...
	appariement_param *appariement;	// coordonnées des points de correspondance de meilleure corrélation dans chaque image et leur valeur de corrélation
	float matrice_RST[3][3];
} recal_correl_param, *Recal_correl_param;


// calcul des points de recalage
/**
 * \ingroup registration
 * \fn Recal_correl_param new_recal_correl_param();
 *
 * \return A new reference on allocated recal_correl_param structure
 *
 * \brief Create a new Recal_correl_param reference 
 *
 */
Recal_correl_param new_recal_correl_param();

/**
 * \ingroup registration
 * \fn void free_recal_correl_param(Recal_correl_param trecal);
 *
 * \param trecal A reference on a recal_correl_param structure
 *
 * \brief Free an allocated Recal_correl_param reference 
 *
 */
void free_recal_correl_param(Recal_correl_param trecal);

/**
 * \ingroup registration
 * \fn void define_recal_correl_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_recal_correl_param(char *extra_description);

/**
 * \ingroup registration
 * \fn  Recal_correl_param get_recal_correl_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Recal_correl_param that contains the parameters.
 */
Recal_correl_param get_recal_correl_param();

//void select_ima_recal(select_ima_t *des, int c0, int c1, int l0, int l1, int b0);
/**
 * \ingroup registration
 * \fn void recal_correl_calc_fl(Recal_correl_param trecal, EFIDIRImageFloat imRef, EFIDIRImageFloat imRecal);
 * \brief function of search of GCP in using a sliding window
 * Function allowing searching the GCP points by maximizing the normalised
 * inter-correlation in using a sliding window. 
 *
 * \param trecal is the structure of the parameters for the operator
 * \param imRef is the reference image
 * \param imRecal is the image to register 
 *
 */
void recal_correl_calc_fl(Recal_correl_param trecal, EFIDIRImageFloat imRef, EFIDIRImageFloat imRecal);

/**
 * \ingroup registration
 * \fn int lect_fichier_zones(Recal_correl_param trecal);
 * \brief function of lecture of GCP file
 *
 * \param trecal is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int lect_fichier_zones(Recal_correl_param trecal);

/**
 * \ingroup registration
 * \fn int suppr_mauvaises_correl(Recal_correl_param trecal, int new_nombre_zones);
 * \brief function of suppression of poor quality GCP
 *
 * \param trecal is the structure of the parameters for the operator
 * \param new_nombre_zones is the number of matches to keep
 * \return 0 if no error occurs
 */
int suppr_mauvaises_correl(Recal_correl_param trecal, int new_nombre_zones);

/**
 * \ingroup registration
 * \fn void free_correl_t(Recal_correl_param trecal);
 * \brief memory free function
 *
 * \param trecal is the structure of the parameters for the operator
 */
void free_correl_t(Recal_correl_param trecal);

// calcul de la transformation RST
/**
 * \ingroup registration
 * \fn void points_corresp_calc(Recal_correl_param trecal, EFIDIRImageFloat imRef, EFIDIRImageFloat imRecal);
 * \brief function of calculation of GCP points
 *
 * \param trecal is the structure of the parameters for the operator
 * \param imRef is the reference image
 * \param imRecal is the image to register
 */
void points_corresp_calc(Recal_correl_param trecal, EFIDIRImageFloat imRef, EFIDIRImageFloat imRecal);

/**
 * \ingroup registration
 * \fn int matrice_RST_calc(Recal_correl_param trecal);
 * \brief function of calculation of RST matrix
 *
 * \param trecal is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int matrice_RST_calc(Recal_correl_param trecal);

/**
 * \ingroup registration
 * \fn int matrice_RST_affiche(Recal_correl_param trecal);
 * \brief function of displaying the RST matrix
 *
 * \param trecal is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int matrice_RST_affiche(Recal_correl_param trecal);
/* modif Yajing YAN */
//void resoudre(float A[9][3],float B[9],float X[3]);

/**
 * \ingroup registration
 * \fn void resoudre(float A[MAX_NB_GCP][3],float B[MAX_NB_GCP],float X[3], int nombre_zones);
 * \brief function of regroundving a linear system of equations
 *
 * \param A is  the system matrix
 * \param B is a vector
 * \param X is the searched vector
 * \param nombre_zones is the number of GCP
 */
void resoudre(float A[MAX_NB_GCP][3],float B[MAX_NB_GCP],float X[3], int nombre_zones);
/* modif Yajing YAN */

// mtest_calcul_RST : en partant directement des points d'amer
/**
 * \ingroup registration
 * \fn Recal_correl_param new_test_calcul_RST_param();
 *
 * \return A new reference on allocated recal_correl_param structure
 *
 * \brief Create a new Recal_correl_param reference 
 *
 */
Recal_correl_param new_test_calcul_RST_param();

/**
 * \ingroup registration
 * \fn void free_test_calcul_RST_param(Recal_correl_param des);
 *
 * \param des A reference on a recal_correl_param structure
 *
 * \brief Free an allocated Recal_correl_param reference 
 *
 */
void free_test_calcul_RST_param(Recal_correl_param des);

/**
 * \ingroup registration
 * \fn void define_test_calcul_RST_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_test_calcul_RST_param(char *extra_description);

/**
 * \ingroup registration
 * \fn  Recal_correl_param get_test_calcul_RST_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Recal_correl_param that contains the parameters.
 */
Recal_correl_param get_test_calcul_RST_param();

/**
 * \ingroup registration
 * \fn int test_calcul_RST_init(Recal_correl_param trecal);
 * \brief function of initialisation of parameters
 *
 * \param trecal is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int test_calcul_RST_init(Recal_correl_param trecal);

/**
 * \ingroup registration
 * \fn int GCP_ENVI_lect(Recal_correl_param trecal);
 * \brief function of lecture of GCP file of ENVI
 *
 * \param trecal is the structure of the parameters for the operator
 * \return 0 if no error occurs
 */
int GCP_ENVI_lect(Recal_correl_param trecal);

/**
 * \ingroup registration
 * \fn int GCP_ENVI_write(Recal_correl_param des, char *nom_fichier_base, char *nom_fichier_wrap);
 * \brief function of writing GCP file in format of ENVI
 *
 * \param des is the structure of the parameters for the operator
 * \param nom_fichier_base is the name of reference file
 * \param nom_fichier_wrap is the name of transformated file
 * \return 0 if no error occurs
 */

int GCP_ENVI_write(Recal_correl_param des, char *nom_fichier_base, char *nom_fichier_wrap);

/**
 * \ingroup registration
 * \fn int mtransform_RST_init(EFIDIRImageFloat imRadarBase, EFIDIRImageFloat imIntensSl2, EFIDIRImageFloat imXsr2, EFIDIRImageFloat imYsr2);
 * \brief function of allocation of memory for images to output
 *
 * \param imRadarBase is the real intensity image in radar geometry
 * \param imIntensSl2 is the simulated intensity image in radar geometry
 * \param imXsr2 is the LUT (ground->radar) in X
 * \param imYsr2 is the LUT (ground->radar) in Y
 * \return 0 if no error occurs
 */
int mtransform_RST_init(EFIDIRImageFloat imRadarBase, EFIDIRImageFloat imIntensSl2, EFIDIRImageFloat imXsr2, EFIDIRImageFloat imYsr2);

#endif
