/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLT_ORBIT_EMULATION_H
#define SLT_ORBIT_EMULATION_H

#include "efidir_allocation.h"
#include "efidir_matrix_op.h"
#include "slt_calculations_geodetic.h"
#include "slt_parser_tools.h"
#include "efidir_math.h"
#include "slt_geom2D.h"
#include "efidir_assert.h"
#include "slt_info_extraction.h"

/**
 * \ingroup sarlut_operators
 * \defgroup orbit_emulation Some tools for orbit propagation
 */

/**
 * \ingroup orbit_emulation
 * \file slt_orbit_emulation.h
 * \author  Diana Rosu
 * \brief Some tools that should be used for orbit propagation
 * file includes (stucture, prototype de fonction) of sarlut package
 */

/**
 * \ingroup orbit_emulation
 * \struct Estimators_s
 * \brief estimators_s structure contains the x, y and z position and also the time of each point of the orbit
 */
typedef struct Estimators_s{

	double **x_estim;
	double **y_estim;
	double **z_estim;

	double **time;

}estimators, *Estimators;

/**
 * \ingroup orbit_emulation
 * \fn Estimators new_Estimators(int nof_estim);
 * \brief Function used in order to allocate memory for an estimator structure
 * \param nof_estim is the number of estimators i.e. the number of points on the orbit that will be considered
 * \return a pointer to a structure of estimators_s
 */
Estimators new_Estimators(int nof_estim);

/**
 * \ingroup orbit_emulation
 * \fn void free_Estimators(int nof_estim, Estimators est);
 * \brief Function used in order to deallocate the memory occupied by an estimator structure
 * \param nof_estim is the number of estimators i.e. the number of points on the orbit that are considered
 * \param est is the estimator vector for which we want to deallocate the memory
 */
void free_Estimators(int nof_estim, Estimators est);

/**
 * \ingroup orbit_emulation
 * \fn void compute_time_series (double t_start, double t_end, double t_pitch, double **time_computed, double t_range_start, double t_range_stop);
 * \brief Function used in order compute the function variable or the polynomial fitting function. It computes it considering a time start, a time end, the frequency but also a specific time of acquisition start and a specific time of acquisition stop. This is in order to support both MNT and RSO images
 *
 * \param t_start is the time start of the time_gps (usually the time_gps of the first state vector)
 * \param t_end is the time end of the time_gps (usually is the time_gps of the last state vector)
 * \param t_pitch is the frequency or the time step
 * \param time_computed is the output of the function being the set of values for the function variable in f(t) = a0 + a1*t + ... + a7*t^7
 * \param t_range_start is the start of the acquisition. This is important if we use the RSO image instead MNT. If the image used is a MNT image then this parameter should take the same value as t_start
 * \param t_range_stop is the end of the acquisition. This is important if we use the RSO image instead MNT. If the image used is a MNT image then this parameter should take the same value as t_end.
 */
void compute_time_series (double t_start, double t_end, double t_pitch, double **time_computed, double t_range_start, double t_range_stop);

/**
 * \ingroup orbit_emulation
 * \fn double **get_polynomial_variable(State_vectors st_vectors, int nof_estim, int nof_st_vect);
 * \brief This function is used in order to obtain the values for the time which is the variable used in the polynomial interpolation function in order to obtain the values of the moments on the orbit (time_d).
 * \param st_vectors is the structure which holds the state vectors extracted from the xml file
 * \param nof_estim is the number of estimators on the orbit
 * \param nof_st_vect is the number of state vectors extracted from the xml file.
 * \return a double pointer to the values of time_variable used in polynomial interpolation function
 */
double **get_polynomial_variable(State_vectors st_vectors, int nof_estim, int nof_st_vect);

/**
 * \ingroup orbit_emulation
 * \fn double **get_polynomial_variable_4_SAR(State_vectors st_vectors, General_info gen_infos);
 * \brief If we want to get the time of acquisition for each raw of SAR image we need to use some specific information which resides in the XML file and which is extracted using get_general_xml_info. So, in order to compute the number of estimators we use this information and not a user specific information.
 * \param st_vectors is the structure which holds the state vectors extracted from the xml file
 * \param gen_infos is the general xml information
 * \return a double pointer to the values of time_variable used in polynomial interpolation function
 */
double **get_polynomial_variable_4_SAR(State_vectors st_vectors, General_info gen_infos);

/**
 * \ingroup orbit_emulation
 * \fn double **get_polynomial_variable_4_coeffs(State_vectors st_vectors, int nof_st_vect);
 * \brief This function is used in order to obtain the values for the time which is the variable used in the polynomial interpolation function in order to obtain the values of the moments on the orbit (time_d).
 *
 * \param st_vectors is the structure which holds the state vectors extracted from the xml file
 * \param nof_st_vect is the number of state vectors extracted from the xml file.
 * \return a double pointer to the values of time_variable used in polynomial fitting function
 */
double **get_polynomial_variable_4_coeffs(State_vectors st_vectors, int nof_st_vect);

/**
 * \ingroup orbit_emulation
 * \fn void orbit_propagation(int polinomial_degree, int nof_estim, int nof_st_vectors, State_vectors st_vectors, double **time_e, double **time_d, Estimators estim);
 * \brief Function used in order to get the orbit for a specific image for a given file
 * \param polinomial_degree is the polynomial degree used for interpolation
 * \param nof_estim is the number of estimators that will be computed
 * \param nof_st_vectors is the number of state vectors that were extracted from a given file in a previous stage of computation
 * \param st_vectors is the state vector structure and holds the values extracted from a specified file
 * \param time_e is the polynomial variable used in order to compute the coefficients of the polynomial
 * \param time_d is the polynomial variable used for polynomial interpolation
 * \param estim is a pointer to a structure which holds the emultaion of the orbit
 */
void orbit_propagation(int polinomial_degree, int nof_estim, int nof_st_vectors, State_vectors st_vectors, double **time_e, double **time_d, Estimators estim);

/**
 * \ingroup orbit_emulation
 * \fn void polyfit(double **x, double **y, int pol_deg, int nof_stat_vec, double *coefs);
 * \brief This function may be used in order to compute the coefficients for a polynomial as a y(x) = a0 + a1x + .. anx^n for a 8 degree polynomial and for 12 pairs of data points
 *
 * \param x is the function variable
 * \param y is the function that should be estimated
 * \param pol_deg is the polynomial degree
 * \param nof_stat_vec is the number of paires (y, x)
 * \param coefs are the coefficients that should be computed
 */
void polyfit(double **x, double **y, int pol_deg, int nof_stat_vec, double *coefs);

/**
 * \ingroup orbit_emulation
 * \fn void create_vandermonde(double **in_vec, int n, int m, double **mat);
 * \brief This function may be used in order to build a Vandermonde matrix
 * \param in_vec is the vector which generates the matrix. The length of the vector will be equal with the number of lines of the matrix
 * \param n is the number of columns of the resulted matrix
 * \param m is the number of lines of the resulted matrix
 * \param mat is the Vandermonde matrix
 */
void create_vandermonde(double **in_vec, int n, int m, double **mat);

/**
 * \ingroup orbit_emulation
 * \fn void matrix_prod_db(double **mat1, double **mat2, int l1, int c1, int l2, int c2, double **mat_prod);
 * \brief This function may be used in order to compute the product of two matrix
 * \param mat1 is the first matrix in the product
 * \param mat2 is the second matrix in the product
 * \param l1 is the number of lines of the first matrix
 * \param c1 is the number of columns of the first matrix
 * \param l2 is the number of lines of the second matrix
 * \param c2 is the number of columns of the second matrix
 * \param mat_prod is the product matrix
 */
void matrix_prod_db(double **mat1, double **mat2, int l1, int c1, int l2, int c2, double **mat_prod);

/**
 * \ingroup orbit_emulation
 * \fn void polyval(double *coefs, double **func_variable, int pol_deg, int nof_pairs, double **func_value);
 * \brief This function is used in order to compute the values for the polynomial in some specified points (the points are specified by func_variable) knowing the coefficients of the polynomial.
 * \param coefs are the coefficients of the polynomial
 * \param func_variable gives the points where the function must be computed
 * \param pol_deg is the degree of the polynomial used
 * \param nof_pairs is the number of points where the function should be computed
 * \param func_value gives the values for the function in the specified points
 */
void polyval(double *coefs, double **func_variable, int pol_deg, int nof_pairs, double **func_value);

/**
 * \ingroup orbit_emulation
 *
 * \fn int compute_nof_points_on_orbit(float dist_between_points, int nof_st_vect, State_vectors st_vectors);
 * \brief This function is used in order to compute the number of points on the trajectory, that is
 * \ the number of estimators that will be computed further using the polynomial. In order to do this
 * \ it must be known the distance between two points on the trajectory in meters.
 *
 * \param dist_between_points is the distance between two points on the orbit (input parameter)
 * \param nof_st_vect is the number of state vectors
 * \param st_vectors is a pointer to the structure which stores the state vectors
 * \return an integer which specifies the number of points on the trajectory, that is the number of estimators that will be computed.
 */
int compute_nof_points_on_orbit(float dist_between_points, int nof_st_vect, State_vectors st_vectors);

#endif /* SLT_ORBIT_PROPAG_H */
