/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLT_ORBIT_CALCULATIONS_H
#define SLT_ORBIT_CALCULATIONS_H

#include "slt_orbit_emulation.h"

/**
 * \ingroup sarlut_operators
 * \defgroup orbit_calculations Some tools which may be used to perform some calculations considering an orbit already emulated
 */

/**
 * \ingroup orbit_calculations
 * \file slt_orbit_calculations.h
 * \author  Diana Rosu
 * \brief Some tools that should be used for computations which implies the usage of orbit a priori emulated
 * file includes (structure, functions prototype) of sarlut package
 */

/**
 * \ingroup orbit_calculations
 * \struct Traj_dist_in_time_s
 * \brief This structure holds the pair (distance, acquisition_time, index_in_estim_v) for a point on the MNT to the trajectory
 */
typedef struct Traj_dist_in_time_s{

	double dist;
	double time;
	int index_in_estim_v;

} traj_dist_in_time, *Traj_dist_in_time;

/**
 * \ingroup orbit_calculations
 * \fn Traj_dist_in_time new_Traj_dist_in_time();
 * \brief Function used in order to allocate memory for a traj_dist_in_time_s structure
 * \return a pointer to a structure of traj_dist_in_time_s
 */
Traj_dist_in_time new_Traj_dist_in_time();

/**
 * \ingroup orbit_calculations
 * \fn void free_Traj_dist_in_time(Traj_dist_in_time t_d);
 * \brief Function used in order to deallocate the memory occupied by a traj_dist_in_time structure
 * \param t_d is a pointer to the structure for which we want to deallocate the memory
 */
void free_Traj_dist_in_time(Traj_dist_in_time t_d);

/**
 * \ingroup orbit_calculations
 * \struct SAR_Coordinates_s
 * \brief This structure holds the position in SAR image, this is the line and the column in a sub-pixel sense.
 */
typedef struct SAR_Coordinates_s{

	double line;
	double column;

} SAR_coordinates, *SAR_Coordinates;

/**
 * \ingroup orbit_calculations
 * \fn SAR_Coordinates new_SAR_Coordinates();
 * \brief Function used in order to allocate memory for a SAR_Coordinates_s structure
 * \return a pointer to a structure SAR_Coordinates_s
 */
SAR_Coordinates new_SAR_Coordinates();

/**
 * \ingroup orbit_calculations
 * \fn void free_SAR_Coordinates(SAR_Coordinates sar_coord);
 * \param sar_coord is a pointer to the structure for which we want to deallocate the memory
 * \brief Function used in order to deallocate the memory for a SAR_Coordinates_s structure
 */
void free_SAR_Coordinates(SAR_Coordinates sar_coord);

/**
 * \ingroup orbit_calculations
 * \fn void generate_lat_long_imgs(EFIDIRImage imMNT, EFIDIRImage imLAT, EFIDIRImage imLONG, char *info_map);
 * \brief This function is used in order to generate the longitude and the latitude images considering a DEM
 * \param imMNT is the given DEM
 * \param imLAT is the resulting latitude image
 * \param imLONG is the resulting longitude image
 * \param info_map is the info_map retrieved from the MNT header
 */
void generate_lat_long_imgs(EFIDIRImage imMNT, EFIDIRImage imLAT, EFIDIRImage imLONG, char *info_map);

/**
 * \ingroup orbit_calculations
 * \fn void gen_latitude_img(int nblig, int nbcol, EFIDIRImage imLAT, char *info_map);
 * \brief This function is used in order to generate the latitude image considering the latitude origin of the most upper left pixel and the pitch in altitude read from the map info field of ENVI header.
 * \ If the latitude is for a southern region then the value will be negative. There is no need to check for the orientation, this is managed by the minus sign for the origin latitude and for the pitch.
 * \ This function is a refactorization of gen_igroundatitudes function which resides in geodutils group (slt_calculations_geodetic.c)
 * \param nblig number of lines of the image
 * \param nbcol number of columns of the image
 * \param imLAT is the resulting image of latitude
 * \param info_map is the info map which is retrieved from the original image (ex: MNT)
 */
void gen_latitude_img(int nblig, int nbcol, EFIDIRImage imLAT, char *info_map);

/**
 * \ingroup orbit_calculations
 * \fn void generate_cartesian_images(EFIDIRImage imLAT, EFIDIRImage imLONG, EFIDIRImage imMNT, EFIDIRImage xImg, EFIDIRImage yImg, EFIDIRImage zImg, int ignored_val);
 * \brief This function generates a separate image for each cartesian coordinate. Thus, the input consists in 3 images while the output consists also in 3 images, one for each dimension.
 * \param imLAT is the image in latitude
 * \param imLONG is the image in longitude
 * \param imMNT is the MNT model
 * \param xImg is the resulting image for the x coordinate
 * \param yImg is the resulting image for the y coordinate
 * \param zImg is the resulting image for the z coordinate
 * \param ignored_val is the value that should be ignored.
 */
void generate_cartesian_images(EFIDIRImage imLAT, EFIDIRImage imLONG, EFIDIRImage imMNT, EFIDIRImage xImg, EFIDIRImage yImg, EFIDIRImage zImg, int ignored_val);

/**
 * \ingroup orbit_calculations
 * \fn void distance_MNT_orbit(int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage distImage);
 * \brief This function is used in order to compute the distance from each pixel of the MNT to the orbit.
 * \param ignored_val is the value of the MNT that should be ignored. The value of distance and time images for that location will be also ignored_val.
 * \param nof_estim is the number of estimators for the orbit
 * \param estim is the structure of estimators for a specific orbit
 * \param imMNT is the MNT image
 * \param distImage is the resulting distance image
 */
void distance_MNT_orbit(int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage distImage);

/**
 * \ingroup orbit_calculations
 * \fn void time_of_acquisition_MNT_orbit(int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage acq_timeImage);
 * \brief This function is used in order to compute the time of acquisition for each pixel of the MNT on the orbit.
 * \param ignored_val is the value of the MNT that should be ignored. The value of distance and time images for that location will be also ignored_val.
 * \param nof_estim is the number of estimators for the orbit
 * \param estim is the structure of estimators for a specific orbit
 * \param imMNT is the MNT image
 * \param acq_timeImage is the resulting time_of_acquisition image
 */
void time_of_acquisition_MNT_orbit(int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage acq_timeImage);

/**
 * \ingroup orbit_calculations
 * \fn void azimuth_SAR_approximative_ping_step(General_info gen_info, int ignored_val, Estimators estim, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to compute the image index in azimuth in an approximative manner for a SAR image mapped on the DEM.
 * \param gen_info is the structure which contains the general information extracted from the metadata file
 * \param ignored_val is the value of the MNT that should be ignored. The value of azimuth in the corresponding locations will be also ignored_val.
 * \param estim is the structure which holds the orbit emulation
 * \param imMNT is the DEM image
 * \param azimuthImage is the resulting azimuth image
 */
void azimuth_SAR_approximative_ping_step(General_info gen_info, int ignored_val, Estimators estim, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 * \fn void azimuth_SAR_full_precision_ping_step(int ignored_val, State_vectors st_vect, General_info gen_info, Estimators estim, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to compute the image of index of in azimuth in a precise manner (sub-pixelic) for a SAR image mapped on the DEM.
 * \param ignored_val is the value of the MNT that should be ignored. azimuth in the corresponding locations will be also ignored_val.
 * \param st_vect is a pointer to a state vector structure
 * \param gen_info is a pointer to a general information structure
 * \param estim is the structure of estimators for a specific orbit
 * \param imMNT is the MNT image
 * \param azimuthImage is the resulting azimuth image
 */
void azimuth_SAR_full_precision_ping_step(int ignored_val, State_vectors st_vect, General_info gen_info, Estimators estim, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 * \fn void range_SAR_full_precision(General_info gen_info, int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to compute in a precise manner the image of index in range for a SAR image mapped on the DEM
 * \param gen_info is a pointer to a general information structure
 * \param ignored_val is the value of the DEM that should be ignored. The value of range in the corresponding locations will be also ignored_val.
 * \param nof_estim is the number of estimators for the orbit
 * \param estim is the structure of estimators for a specific orbit
 * \param imMNT is the MNT image
 * \param rangeImage is the image of range computed on the MNT
 */
void range_SAR_full_precision(General_info gen_info, int ignored_val, int nof_estim, Estimators estim, EFIDIRImage imMNT, EFIDIRImage rangeImage);

void azimuth_range_SAR_full_precision(General_info gen_info, int ignored_val, int nof_estim, Estimators estim, State_vectors st_vect, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 * \fn traj_dist_in_time min_distance_point_traj(int nof_estim, coord_3d mnt_point, Estimators estim);
 * \brief This function returns the minimum distance to the orbit for a given point and the associated time of the point where is obtained the minimum distance (the CPA).
 * \param nof_estim is the number of estimators of the orbit
 * \param mnt_point is a point on the MNT
 * \param estim is the pointer to a estimator structure computed in a previous step
 * \return a structure which contains both the distance and the moment on the orbit
 */
traj_dist_in_time min_distance_point_traj(int nof_estim, coord_3d mnt_point, Estimators estim);

/**
 * \ingroup orbit_calculations
 * \fn void gen_MNT_img_plat (int nof_l, int nof_c, double d_val, EFIDIRImage res_mnt);
 * \brief This function Generates an image (which should be used as a MNT for a plat tarain
 * \param nof_l is the number of lines of the resulted image
 * \param nof_c is the number of columns of the resulted image
 * \param d_val is the value that should be used to fill the image. Pay attention do not use 0;
 * \param res_mnt is the resulted image
 */
void gen_MNT_img_plat (int nof_l, int nof_c, double d_val, EFIDIRImage res_mnt);

/**
 * \ingroup orbit_calculations
 * \fn void gen_longitude_img(int nblig, int nbcol, EFIDIRImage imLON, char *info_map);
 * \brief This function is used in order to generate the longitude image considering the longitude origin of the most upper left pixel and the pitch in longitude read from the map info field of ENVI header.
 * \ If the longitude is for a southern region then the value will be negative.
 * \ This function is a refactorization of gen_igroundongitudes function which resides in geodutils group (slt_calculations_geodetic.c)
 * \param nblig number of lines of the image
 * \param nbcol number of columns of the image
 * \param imLON is the resulting image of longitude
 * \param info_map is the info map which is retrieved from the original image (ex: MNT)
 */
void gen_longitude_img(int nblig, int nbcol, EFIDIRImage imLON, char *info_map);

/**
 * \ingroup orbit_calculations
 * \fn void linear_combination_4_images(int reading_mode, int ignored_val, double coef1, double coef2, double coef3, EFIDIRImage master_img, EFIDIRImage slave_img, EFIDIRImage result_img);
 * \brief This function is used in order to compute an image as a result for coef1*master_img + coef2 * slave_img + coef3
 * \param reading_mode is the value that specifies if the reading process for master and slave image is line by line or all at once. Thus, put zero if the reading mode is all at once and 1 if the reading mode is line by line.
 * \param ignored_val is the value in the master and slave images that should be ignored.
 * \param coef1 is the constant which multiplies the values of pixels of the first image (master image)
 * \param coef2 is the constant which multiplies the values of pixels of the second image (slave image)
 * \param coef3 is the constant which is added to the result
 * \param master_img is the master image (or the first image)
 * \param slave_img is the slave image (or the second image)
 * \param result_img is the resulted image
 */
void linear_combination_4_images(int reading_mode, int ignored_val, double coef1, double coef2, double coef3, EFIDIRImage master_img, EFIDIRImage slave_img, EFIDIRImage result_img);

/**
 * \ingroup orbit_calculations
 * \fn void generate_dist_and_time_imgs_for_orbit_from_xml_tsx(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);
 * \brief This function is used in order to generate the time and distance images for a given TerraSAR-X xml file.
 * \param polynomial_degree is the polynomial degree used for interpolation
 * \param input_file is the xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param distImage is the distance image that will be generated
 * \param timeImage is the time image that will be generated
 */
void generate_dist_and_time_imgs_for_orbit_from_xml_tsx(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_dist_and_time_imgs_for_orbit_from_xml_rsat2(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);
 * \brief This function is used in order to generate the time and distance images for a given RADARSAT-2 xml file.
 * \param polynomial_degree is the polynomial degree used for interpolation
 * \param input_file is the xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param distImage is the distance image that will be generated
 * \param timeImage is the time image that will be generated
 */
void generate_dist_and_time_imgs_for_orbit_from_xml_rsat2(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_dist_and_time_imgs_for_orbit_from_gamma_output_file(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);
 * \brief This function is used in order to generate the time and distance images for a given gamma output file.
 * \param polynomial_degree is the polynomial degree used for the interpolation
 * \param input_file is the xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param distImage is the distance image that will be generated
 * \param timeImage is the time image that will be generated
 */
void generate_dist_and_time_imgs_for_orbit_from_gamma_output_file(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage distImage, EFIDIRImage timeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_xml_tsx_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image aquired by TerraSAR-X satellite on the MNT used.
 * \param polynomial_degree is the polynomial degree used for the interpolation in orbit emulation process
 * \param input_file is the TerraSAR-X xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param low_level_precision specifies if the output azimuth image will be in full precision or an approximation for the azimuth. This parameter will be 0 if there is required a full precision and 1 if the low precision is satisfactory.
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_xml_tsx_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage)
 * \brief This function is used in order to generate the image of line acquisition of SAR image aquired by TerraSAR-X satellite on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for the interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the TerraSAR-X xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_xml_rsat2_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image acquired by RADARSAT-2 satellite on the MNT used.
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param input_file is the RADARSAT2 xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param low_level_precision specifies if the output azimuth image will be in full precision or an approximation for the azimuth. This parameter will be 0 if there is required a full precision and 1 if the low precision is satisfactory.
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_xml_rsat2_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage)
 * \brief This function is used in order to generate the image of line acquisition of SAR image acquired by RADARSAT-2 satellite on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the RADARSAT2 xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_gamma_output_file_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image for which on dispose of a gamma output file on the MNT used.
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param input_file is the gamma output file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param low_level_precision specifies if the output azimuth image will be in full precision or an approximation for the azimuth. This parameter will be 0 if there is required a full precision and 1 if the low precision is satisfactory.
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_gamma_output_file_ping_step(int polynomial_degree, char *input_file, float ignored_value, int low_level_precision, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_azimuth_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image for which on dispose of a gamma output file on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the gamma output file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param azimuthImage is the image of the line of the SAR image on the MNT
 */
void generate_azimuth_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_SAR_img_for_orbit_from_xml_tsx(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used.
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param input_file is the TerraSAR-X xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generate_range_SAR_img_for_orbit_from_xml_tsx(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the TerraSAR-X xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generate_range_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 * \fn void generate_range_SAR_img_for_orbit_from_xml_rsat2(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used.
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param input_file is the RADARSAT-2 xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generate_range_SAR_img_for_orbit_from_xml_rsat2(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 * \fn void generate_range_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the RADARSAT-2 xml file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generate_range_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generaterange_SAR_img_for_orbit_from_gamma_output_file(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used.
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param input_file is the Gamma output file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generaterange_SAR_img_for_orbit_from_gamma_output_file(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
 * \brief This function is used in order to generate the image of line acquisition of SAR image on the MNT used. This function does not use the sampling frequency step but a parameter entered by the user
 * \param polynomial_degree is the polynomial degree used for interpolation in orbit emulation process
 * \param dist_between_points specifies the number of estimators of the orbit
 * \param input_file is the Gamma output file from which the state vectors and the needed informations will be extracted
 * \param ignored_value is the value of the MNT image that will be ignored
 * \param imMNT is the MNT image that will be used to generate the distance and time images
 * \param rangeImage is the image of the SAR range on the MNT
 */
void generate_range_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn SAR_Coordinates get_SAR_coord_on_MNT (coord_3d mnt_point, int nof_estim, Estimators estim, General_info gen_info);
 * \brief This function is used in order to compute the coordinates from SAR perspective on the DEM that is used. (VALIDATION PURPOSES -> convey the coordinates of a known point on the earth and check if it is the same in the SAR image.)
 * \param mnt_point are the coordinates of a point on the DEM
 * \param nof_estim is the number of estimators of the orbit
 * \param estim is a pointer to a structure which holds the estimators of the orbit
 * \param gen_info is a pointer to a structure which holds the general information as prf, lastPixel and first pixel needed in order to compute the lines and columns indexes in the SAR image.
 * \return a structure SAR_coordinates which consists the line and column index in the SAR image of the observed point.
 */
SAR_Coordinates get_SAR_coord_on_MNT (coord_3d mnt_point, int nof_estim, Estimators estim, General_info gen_info);

void generate_range_azimuth_SAR_img_for_orbit_from_image_user_step(int polynomial_degree, float dist_between_points, EFIDIRImage imInput, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_azimuth_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate both the range and azimuth images for a TSX product starting from the metadata file
 * \param polynomial_degree is the degree of the polynomial used for orbit estimation.
 * \param dist_between_points is the estimated distance between two successive points on the orbit
 * \param input_file is the complete path to the metadata xml file for a TSX product
 * \param ignored_value is the value that should be ignored in a DEM image. The correspondent pixel values in the resulting images will have the same value
 * \param imMNT is the full path to the DEM
 * \param rangeImage is the resulting range image
 * \param azimuthImage is the resulting azimuth image
 */
void generate_range_azimuth_SAR_img_for_orbit_from_xml_tsx_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_azimuth_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate both the range and azimuth images for a RSAT2 product starting from the metadata file
 * \param polynomial_degree is the degree of the polynomial used for orbit estimation.
 * \param dist_between_points is the estimated distance between two successive points on the orbit
 * \param input_file is the complete path to the metadata xml file for a TSX product
 * \param ignored_value is the value that should be ignored in a DEM image. The correspondent pixel values in the resulting images will have the same value
 * \param imMNT is the full path to the DEM
 * \param rangeImage is the resulting range image
 * \param azimuthImage is the resulting azimuth image
 */
void generate_range_azimuth_SAR_img_for_orbit_from_xml_rsat2_user_step(int polynomial_degree, float dist_between_points, char *input_file, float ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);

/**
 * \ingroup orbit_calculations
 *
 * \fn void generate_range_azimuth_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);
 * \brief This function is used in order to generate both the range and azimuth images for a metadata given by the gamma output file
 * \param polynomial_degree is the degree of the polynomial used for orbit estimation.
 * \param dist_between_points is the estimated distance between two successive points on the orbit
 * \param input_file is the complete path to the metadata xml file for a TSX product
 * \param ignored_value is the value that should be ignored in a DEM image. The correspondent pixel values in the resulting images will have the same value
 * \param imMNT is the full path to the DEM
 * \param rangeImage is the resulting range image
 * \param azimuthImage is the resulting azimuth image
 */
void generate_range_azimuth_SAR_img_for_orbit_from_gamma_output_file_user_step(int polynomial_degree, float dist_between_points, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage, EFIDIRImage azimuthImage);


void generate_range_SAR_img_for_orbit_from_gamma_output_file(int polynomial_degree, char *input_file, int ignored_value, EFIDIRImage imMNT, EFIDIRImage rangeImage);
#endif /* SLT_ORBIT_CALCULATIONS_H_ */
