/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup sarlut_operators
 * \defgroup basic_operations Some basic operations for sarlut package
 */
/**
 * \ingroup basic_operations
 * \file slt_operations_basic.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief some basic operators 
 * 
 * file includes (stucture, prototype de fonction) of the operator basic_operations
 * 
 * 
 */

#ifndef SLT_OPERATIONS_H
#define SLT_OPERATIONS_H

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include "slt_geom2D.h"

// STRUCTURES
/**
 * \ingroup basic_operations
 * \struct stat_visu_msq_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct{
	int val_obj;
} stat_visu_msq_param, *Stat_visu_msq_param;

/**
 * \ingroup basic_operations
 * \struct comp_images_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct{
	float val_inut, val_inut2, val_inut_res;
	int type_de_comparaison;
} comp_images_param, *Comp_images_param;

/**
 * \ingroup basic_operations
 * \struct change_values_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct{
	float seuil_bas;
	float seuil_haut;
} change_values_param, *Change_values_param;

/**
 * \ingroup basic_operations
 * \struct gen_images_test_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct{
	float offset_x, offset_y, pas_x, pas_y;
	int nblig, nbcol;
} gen_images_test_param, *Gen_images_test_param;

/**
 * \ingroup basic_operations
 * \struct enlarge_image_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct{
	float background_fl;
	int background_u1;
	int offset_x, offset_y;
	int nblig, nbcol;
} enlarge_image_param, *Enlarge_image_param;


// LUT_FLIP
/**
 * \ingroup basic_operations
 * \fn void define_lut_flip_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_lut_flip_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn  get_lut_flip_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A float...
 */
float get_lut_flip_param();

/**
 * \ingroup basic_operations
 * \fn int lut_flip_init(EFIDIRImageFloat imLUT, EFIDIRImageFloat imLUTflip);
 * \brief function of flip
 *
 * \param  imLUT is the original LUT (radar->ground)
 * \param imLUTflip is the flipped LUT (radar->ground)
 * \return 0 if no error occurs
 *
 */
int lut_flip_init(EFIDIRImageFloat imLUT, EFIDIRImageFloat imLUTflip);

/**
 * \ingroup basic_operations
 * \fn int lut_flip(EFIDIRImageFloat imLUT, EFIDIRImageFloat imLUTflip, float offset);
 * \brief function of flip
 *
 * \param  imLUT is the original LUT (radar->ground)
 * \param imLUTflip is the flipped LUT (radar->ground)
 * \param offset is number of columns of reference image in radar
 * \return 0 if no error occurs
 *
 */
int lut_flip(EFIDIRImageFloat imLUT, EFIDIRImageFloat imLUTflip, float offset);

/**
 * \ingroup basic_operations
 * \fn int lut_flip_bis(EFIDIRImageFloat imLUT, int offset);
 * \brief second function of flip
 *
 * \param  imLUT is the original LUT (radar->ground)
 * \param offset is number of columns of reference image in radar
 * \return 0 if no error occurs
 *
 */
int lut_flip_bis(EFIDIRImageFloat imLUT, int offset);

// STAT_VISU_MSQ
/**
 * \ingroup basic_operations
 * \fn Stat_visu_msq_param new_stat_visu_msq_param ( );
 *
 * \return A new reference on allocated stat_visu_msq_param structure
 *
 * \brief Create a new Stat_visu_msq_param reference 
 *
 */
static Stat_visu_msq_param new_stat_visu_msq_param ( );

/**
 * \ingroup basic_operations
 * \fn void free_stat_visu_msq_param(Stat_visu_msq_param des);
 *
 * \param des A reference on a stat_visu_msq_param structure
 *
 * \brief Free an allocated Stat_visu_msq_param reference 
 *
 */
static void free_stat_visu_msq_param(Stat_visu_msq_param des);

/**
 * \ingroup basic_operations
 * \fn void define_visu_msq_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_visu_msq_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn  Stat_visu_msq_param get_stat_visu_msq_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Stat_visu_msq_param that contains the parameters.
 */
 Stat_visu_msq_param get_stat_visu_msq_param();

/**
 * \ingroup basic_operations
 * \fn int stat_visu_msq_init(Stat_visu_msq_param des, EFIDIRImageByte imMsqVisu, EFIDIRImageByte imMsqObj, EFIDIRImageByte imMsqObjVisu);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMsqVisu is the mask
 * \param imMsqObj is the mask
 * \param imMsqObjVisu is the mask
 * \return 0 if no error occurs
 *
 */

int stat_visu_msq_init(Stat_visu_msq_param des, EFIDIRImageByte imMsqVisu, EFIDIRImageByte imMsqObj, EFIDIRImageByte imMsqObjVisu);

/**
 * \ingroup basic_operations
 * \fn int stat_visu_msq_calc(Stat_visu_msq_param des, EFIDIRImageByte imMsqVisu, EFIDIRImageByte imMsqObj, EFIDIRImageByte imMsqObjVisu);
 * \brief function of calculation of operator stat_visu_msq
 *
 * \param des is the structure of the parameters for the operator 
 * \param imMsqVisu is the mask
 * \param imMsqObj is the mask
 * \param imMsqObjVisu is the mask
 * \return 0 if no error occurs
 *
 */

int stat_visu_msq_calc(Stat_visu_msq_param des, EFIDIRImageByte imMsqVisu, EFIDIRImageByte imMsqObj, EFIDIRImageByte imMsqObjVisu);

// COMP_IMAGES
/**
 * \ingroup basic_operations
 * \fn Comp_images_param new_comp_images_param ( );
 *
 * \return A new reference on allocated comp_images_param structure
 *
 * \brief Create a new Comp_images_param reference 
 *
 */
static Comp_images_param new_comp_images_param ( );

/**
 * \ingroup basic_operations
 * \fn void free_comp_images_param(Comp_images_param des);
 *
 * \param des A reference on a comp_images_param structure
 *
 * \brief Free an allocated Comp_images_param reference 
 *
 */
static void free_comp_images_param(Comp_images_param des);

/**
 * \ingroup basic_operations
 * \fn void define_comp_images_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_comp_images_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn  Comp_images_param get_comp_images_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Comp_images_param that contains the parameters.
 */
 Comp_images_param get_comp_images_param( );

/**
 * \ingroup basic_operations
 * \fn int comp_images_init(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imRef
 * \param imComp
 * \param imResult
 * \return 0 if no error occurs
 *
 */

int comp_images_init(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);

/**
 * \ingroup basic_operations
 * \fn int comp_images_calc(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);
 * \brief function of calculation of operator comp_images
 *
 * \param des is the structure of the parameters for the operator 
 * \param imRef
 * \param imComp
 * \param imResult
 * \return 0 if no error occurs
 *
 */

int comp_images_calc(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);

// OP_SIMPLES2
/**
 * \ingroup basic_operations
 * \fn void define_op_simples2_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_op_simples2_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn  Comp_images_param get_op_simples2_param( );
 * \brief Default efidir function that gets parameters
 *
 * \return A Comp_images_param that contains the parameters.
 */
Comp_images_param get_op_simples2_param( );

/**
 * \ingroup basic_operations
 * \fn int op_simples2_calc(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imRef is the reference image
 * \param imComp is the image to compare
 * \param imResult is the image of result
 * \return 0 if no error occurs
 *
 */
int op_simples2_calc(Comp_images_param des, EFIDIRImageFloat imRef, EFIDIRImageFloat imComp, EFIDIRImageFloat imResult);

// CHANGE_VALUES
/**
 * \ingroup basic_operations
 * \fn Change_values_param new_change_values_param();
 *
 * \return A new reference on allocated change_values_param structure
 *
 * \brief Create a new Change_values_param reference 
 *
 */

Change_values_param new_change_values_param();

/**
 * \ingroup basic_operations
 * \fn void free_change_values_param(Change_values_param des);
 *
 * \param des A reference on a change_values_param structure
 *
 * \brief Free an allocated Change_values_param reference 
 *
 */
void free_change_values_param(Change_values_param des);

/**
 * \ingroup basic_operations
 * \fn void define_change_values_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_change_values_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn  Change_values_param get_change_values_param( );
 * \brief Default efidir function that gets parameters
 *
 * \return A Change_values_param that contains the parameters.
 */
Change_values_param get_change_values_param( );

/**
 * \ingroup basic_operations
 * \fn int change_values_init(Change_values_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original image
 * \param imRes is the image of result
 * \return 0 if no error occurs
 *
 */
int change_values_init(Change_values_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);

/**
 * \ingroup basic_operations
 * \fn int change_values_calc(Change_values_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);
 * \brief function of changing value of an image
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original image
 * \param imRes is the image of result
 * \return 0 if no error occurs
 *
 */
int change_values_calc(Change_values_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);

// GEN_IMAGES_TEST
/**
 * \ingroup basic_operations
 * \fn Gen_images_test_param new_gen_images_test_param();
 *
 * \return A new reference on allocated gen_images_test_param structure
 *
 * \brief Create a new Gen_images_test_param reference 
 *
 */
Gen_images_test_param new_gen_images_test_param();

/**
 * \ingroup basic_operations
 * \fn void free_gen_images_test_param(Gen_images_test_param des);
 *
 * \param des A reference on a gen_images_test_param structure
 *
 * \brief Free an allocated Gen_images_test_param reference 
 *
 */
void free_gen_images_test_param(Gen_images_test_param des);

/**
 * \ingroup basic_operations
 * \fn void define_gen_images_test_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_gen_images_test_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn Gen_images_test_param get_gen_images_test_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Gen_images_test_param that contains the parameters.
 */
Gen_images_test_param get_gen_images_test_param();

/**
 * \ingroup basic_operations
 * \fn int gen_images_test_init(Gen_images_test_param des, EFIDIRImageFloat imX, EFIDIRImageFloat imY);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imX is the image of X
 * \param imY is the image of Y
 * \return 0 if no error occurs
 *
 */
int gen_images_test_init(Gen_images_test_param des, EFIDIRImageFloat imX, EFIDIRImageFloat imY);


/**
 * \ingroup basic_operations
 * \fn int gen_images_test_calc(Gen_images_test_param des, EFIDIRImageFloat imX, EFIDIRImageFloat imY);
 * \brief function of allocation of images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imX is the image of X
 * \param imY is the image of Y
 * \return 0 if no error occurs
 *
 */
int gen_images_test_calc(Gen_images_test_param des, EFIDIRImageFloat imX, EFIDIRImageFloat imY);

// ENLARGE_IMAGE_FL
/**
 * \ingroup basic_operations
 * \fn Enlarge_image_param new_enlarge_image_param();
 *
 * \return A new reference on allocated enlarge_image_param structure
 *
 * \brief Create a new Enlarge_image_param reference 
 *
 */
Enlarge_image_param new_enlarge_image_param();

/**
 * \ingroup basic_operations
 * \fn void free_enlarge_image_param(Enlarge_image_param des);
 *
 * \param des A reference on a enlarge_image_param structure
 *
 * \brief Free an allocated Enlarge_image_param reference 
 *
 */
void free_enlarge_image_param(Enlarge_image_param des);

/**
 * \ingroup basic_operations
 * \fn void define_enlarge_image_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_enlarge_image_param(char *extra_description);

/**
 * \ingroup basic_operations
 * \fn Enlarge_image_param get_enlarge_image_param( );
 * \brief Default efidir function that gets parameters
 *
 * \return A Enlarge_image_param that contains the parameters.
 */
Enlarge_image_param get_enlarge_image_param( );

/**
 * \ingroup basic_operations
 * \fn int enlarge_image_init_fl(Enlarge_image_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);
 * \brief function of allocation of float images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original float image
 * \param imRes is the float image of result
 * \return 0 if no error occurs
 *
 */
int enlarge_image_init_fl(Enlarge_image_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);

/**
 * \ingroup basic_operations
 * \fn int enlarge_image_calc_fl(Enlarge_image_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);
 * \brief function of enlarging a float image
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original float image
 * \param imRes is the float image of result
 * \return 0 if no error occurs
 *
 */
int enlarge_image_calc_fl(Enlarge_image_param des, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes);

/**
 * \ingroup basic_operations
 * \fn int enlarge_image_init_u1(Enlarge_image_param des, EFIDIRImageByte imInit, EFIDIRImageByte imRes);
 * \brief function of allocation of 8bytes images to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original 8 bytes image
 * \param imRes is the 8 bytes image of result
 * \return 0 if no error occurs
 *
 */
int enlarge_image_init_u1(Enlarge_image_param des, EFIDIRImageByte imInit, EFIDIRImageByte imRes);

/**
 * \ingroup basic_operations
 * \fn int enlarge_image_calc_u1(Enlarge_image_param des, EFIDIRImageByte imInit, EFIDIRImageByte imRes);
 * \brief function of enlarging a 8 bytes image
 *
 * \param des is the structure of the parameters for the operator 
 * \param imInit is the original 8 bytes image
 * \param imRes is the 8 bytes image of result
 * \return 0 if no error occurs
 *
 */
int enlarge_image_calc_u1(Enlarge_image_param des, EFIDIRImageByte imInit, EFIDIRImageByte imRes);


#endif
