/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup sarlut_operators
 * \defgroup utilisation_lut_radar2ground Transformation from radar geometry to ground geometry
 */
/**
 * \ingroup utilisation_lut_radar2ground
 * \file slt_interpol_lut_radar2ground.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief operator of transformation from radar geometry to ground geometry
 * 
 * file includes (stucture, prototype de fonction) of the operator utilisation_lut_radar2ground
 * 
 * 
 */

#ifndef SLT_INTERPOL_LUT_GROUND2RADAR_H
#define SLT_INTERPOL_LUT_GROUND2RADAR_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include "slt_calculations_geodetic.h"


/**
 * \ingroup utilisation_lut_radar2ground
 * \struct interpol_lut_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct
{
    int interpol;
    float val_inut, val_msq;
    char *pass; // modif Yajing YAN
} interpol_lut_param, *Interpol_lut_param;


/**
 * \ingroup utilisation_lut_radar2ground
 * \fn Interpol_lut_param new_interpol_lut_param ( );
 *
 * \return A new reference on allocated interpol_lut_param structure
 *
 * \brief Create a new Interpol_lut_param reference 
 *
 */
static Interpol_lut_param new_interpol_lut_param ( );

/**
 * \ingroup utilisation_lut_radar2ground
 * \fn void free_interpol_lut_param(Interpol_lut_param des);
 *
 * \param des A reference on a interpol_lut_param structure
 *
 * \brief Free an allocated Interpol_lut_param reference 
 *
 */
void free_interpol_lut_param(Interpol_lut_param des);

/**
 * \ingroup utilisation_lut_radar2ground
 * \fn void define_interpol_lut_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_interpol_lut_param(char *extra_description);

/**
 * \ingroup utilisation_lut_radar2ground
 * \fn  Interpol_lut_param Interpol_lut_param get_interpol_lut_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Interpol_lut_param that contains the parameters.
 */
 Interpol_lut_param get_interpol_lut_param( );

// interpol_lut_fl
/**
 * \ingroup utilisation_lut_radar2ground
 * \fn int interpol_lut_init_fl(Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageFloat imres);
 * \brief function of initialisation of operator interpol_lut_init_fl
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the image to interpolate
 * \param imRange is the LUT in X
 * \param imAzimut is the LUT in Y
 * \param imres is the image after interpolation
 * \return 0 if no error occurs
 *
 */
int interpol_lut_init_fl(Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageFloat imres);

/**
 * \ingroup utilisation_lut_radar2ground
 * \fn int interpol_lut_calc_fl(Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageFloat imres);
 * \brief function of utilisation of LUT for float image
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the image to interpolate
 * \param imRange is the LUT in X
 * \param imAzimut is the LUT in Y
 * \param imres is the image after interpolation
 * \return 0 if no error occurs
 *
 */

int interpol_lut_calc_fl(Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageFloat imres);


// interpol_lut_u1
/**
 * \ingroup utilisation_lut_radar2ground
 * \fn int interpol_lut_init_u1(Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageByte imres);
 * \brief function of initialisation of operator interpol_lut_init_u1
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the image to interpolate
 * \param imRange is the LUT in X
 * \param imAzimut is the LUT in Y
 * \param imres is the image after interpolation
 * \return 0 if no error occurs
 *
 */
int interpol_lut_init_u1(Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageByte imres);

/**
 * \ingroup utilisation_lut_radar2ground
 * \fn int interpol_lut_calc_u1(Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageByte imres);
 * \brief function of utilisation of LUT for 8bytes image
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the image to interpolate
 * \param imRange is the LUT in X
 * \param imAzimut is the LUT in Y
 * \param imres is the image after interpolation
 * \return 0 if no error occurs
 *
 */
int interpol_lut_calc_u1(Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imRange, EFIDIRImageFloat imAzimut, EFIDIRImageByte imres);

#endif
