/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup sarlut_operators
 * \defgroup utilisation_lut_ground2radar Transformation from ground geometry to radar geometry
 */
/**
 * \ingroup utilisation_lut_ground2radar
 * \file slt_interpol_lut_ground2radar.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief operator of transformation from ground geometry to radar geometry with LUT (look-up table)
 * 
 * file includes (stucture, prototype de fonction) of the operator utilisation_lut_ground2radar
 * 
 * 
 */

#ifndef SLT_INTERPOL_LUT_GROUND2RADAR_H
#define SLT_INTERPOL_LUT_GROUND2RADAR_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include "slt_simu.h"
#include "slt_intensity_ground.h"

/**
 * \ingroup utilisation_lut_ground2radar
 * \struct interpol_lut_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct
{
	int interpol;
	float val_inut, val_msq;
} interpol_lut_param, *Interpol_lut_param;

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn Interpol_lut_param new_interpol_lut_georef_param();
 *
 * \return A new reference on allocated interpol_lut_param structure
 *
 * \brief Create a new Interpol_lut_param reference 
 *
 */
Interpol_lut_param new_interpol_lut_georef_param();

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn Simu_param new_sim_param();
 *
 * \return A new reference on allocated simu_param structure
 *
 * \brief Create a new Simu_param reference 
 *
 */
Simu_param new_sim_param();

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn void free_interpol_lut_georef_param(Interpol_lut_param des);
 *
 * \param des A reference on a interpol_lut_param structure
 *
 * \brief Free an allocated Interpol_lut_param reference 
 *
 */
void free_interpol_lut_georef_param(Interpol_lut_param des);

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn void free_sim_param(Simu_param sim);
 *
 * \param sim A reference on a simu_param structure
 *
 * \brief Free an allocated Simu_param reference 
 *
 */
void free_sim_param(Simu_param sim);

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn void define_interpol_lut_georef_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_interpol_lut_georef_param(char *extra_description);
//void define_sim_param(char *extra_description);

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn  Interpol_lut_param get_interpol_lut_georef_param( );
 * \brief Default efidir function that gets parameters
 *
 * \return A Interpol_lut_param that contains the parameters.
 */
Interpol_lut_param get_interpol_lut_georef_param( );
//Simu_param get_sim_param();

// INTERPOL_LUT_GEOREF_FL
/**
 * \ingroup utilisation_lut_ground2radar
 * \fn int interpol_lut_georef_init_fl(Simu_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageFloat imres);
 * \brief function of allocation of images (float) to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the original image (ground geometry)
 * \param imLong is the LUT in longitude
 * \param imLat is the LUT in latitude
 * \param imres is the result of transformation with LUT (radar geometry)
 * \return 0 if no error occurs
 *
 */
int interpol_lut_georef_init_fl(Simu_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageFloat imres);

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn int interpol_lut_georef_calc_fl(Simu_param sim, Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageFloat imres);
 * \brief function of transformation of geometry (ground->radar) of images (float) with LUT
 *
 * \param sim is the structure of the parameters for the operator
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the original image (ground geometry)
 * \param imLong is the LUT in longitude
 * \param imLat is the LUT in latitude
 * \param imres is the result of transformation with LUT (radar geometry)
 * \return 0 if no error occurs
 *
 */
int interpol_lut_georef_calc_fl(Simu_param sim, Interpol_lut_param des, EFIDIRImageFloat iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageFloat imres);


// INTERPOL_LUT_GEOREF_U1
/**
 * \ingroup utilisation_lut_ground2radar
 * \fn int interpol_lut_georef_init_u1(Simu_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageByte imres);
 * \brief function of allocation of images (8 bytes) to output
 *
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the original image (ground geometry)
 * \param imLong is the LUT in longitude
 * \param imLat is the LUT in latitude
 * \param imres is the result of transformation with LUT (radar geometry)
 * \return 0 if no error occurs
 *
 */
int interpol_lut_georef_init_u1(Simu_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageByte imres);

/**
 * \ingroup utilisation_lut_ground2radar
 * \fn int interpol_lut_georef_calc_u1(Simu_param sim,Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageByte imres);
 * \brief function of transformation of geometry (ground->radar) of images (8 bytes) with LUT
 *
 * \param sim is the structure of the parameters for the operator
 * \param des is the structure of the parameters for the operator 
 * \param iminit is the original image (ground geometry)
 * \param imLong is the LUT in longitude
 * \param imLat is the LUT in latitude
 * \param imres is the result of transformation with LUT (radar geometry)
 * \return 0 if no error occurs
 *
 */
int interpol_lut_georef_calc_u1(Simu_param sim,Interpol_lut_param des, EFIDIRImageByte iminit, EFIDIRImageFloat imLong, EFIDIRImageFloat imLat, EFIDIRImageByte imres);

#endif
