/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLT_INFO_EXTRACTION_H_
#define SLT_INFO_EXTRACTION_H_

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <string.h>
#include "efidir_allocation.h"
#include "efidir_matrix_op.h"
#include "slt_calculations_geodetic.h"
#include "slt_parser_tools.h"
#include "efidir_math.h"
#include "slt_geom2D.h"
#include <sys/times.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <time.h>
#include <unistd.h>

#define LIGHT_SPEED 299792458 /*The speed of light in [m]*/
#define UTC_AHEAD_GPS -15 /* The deviation between GPS and UTC */

/**
 * \ingroup sarlut_operators
 * \defgroup information_extractor Some tools which may be used in order to extract the needed information from some types of metadata files: TerraSAR-X xml, RADARSAT-2 sml and GAMMA output files
 */

/**
 * \ingroup information_extractor
 * \file slt_info_extraction.h
 * \author  Diana Rosu
 * \brief Some tools that should be used in order to extract the needed information for orbit propagation.
 * file includes (stucture, prototype de fonction) of sarlut package
 */

/**
 * \ingroup information_extractor
 * \enum SatellitePassDirection
 * \brief The direction of the satellite track. It can be ASCENDING or DESCENDING
 */
enum SatellitePassDirection {
	DESCENDING = 0, /* Descending pass */
	ASCENDING = 1 /* Ascending pass */
};

/**
 * \ingroup information_extractor
 * \enum MetadataType
 * \brief The type of metadata file that is given as input. Pay attention, sometimes the GAMMA output files do not specify if the acquired image is flipped, this is the case of RADARSAT2 with gamma output files.
 */
enum MetadataType {
	TSX = 0, /* TerraSAR-X xml */
	GAMMA_Output_File = 1, /* An output file of GAMMA software */
	RADARSAT2 = 2 /* A RADARSAT-2 xml */
};

/**
 * \ingroup information_extractor
 * \struct State_vectors_s
 * \brief State_vectors_s structure contains the state vectors extracted from either a xml file (TSX or RADARSAT2), from ROI-PAC file or from GAMMA output file
 */
typedef struct State_vectors_s{
	double **x_pos; /* x cartesian dimension */
	double **y_pos; /* y cartesian dimension */
	double **z_pos; /* z cartesian dimension */

	double **time_gps; /* Time dimension */

	double **x_vel; /* The x speed component */
	double **y_vel; /* The y speed component */
	double **z_vel; /* The z speed component */
}state_vectors, *State_vectors;

/**
 * \ingroup information_extractor
 * \struct General_info_s
 * \brief general_info_s structure contains all the informations needed in processing but not the state vectors
 */
typedef struct General_info_s{

	int nof_state_vectors; /*This is the number of state vectors specified in the xml file*/
	int nof_rows; /*This is the number of rows of the RSO image */
	int nof_columns; /*This is the number of columns of the RSO image */

	double time_stamp_first_row; /* Time stamp of first image row */
	double time_stamp_last_row; /* Time stamp of last image row */

	double time_stamp_first_row_fraction; /* This is the start of the step time */
	double time_stamp_last_row_fraction; /* This is the stop of the step time */

	double x_pos_antenna; /* This is the x position of the antenna */
	double y_pos_antenna; /* This is the y position of the antenna */
	double z_pos_antenna; /* This is the z position of the antenna */

	double time_stamp_first_sample; /* Minimum range time of the entire scene */
	double time_stamp_last_sample; /* Maximum range time of the entire scene */

	double rsf; /* The range sampling frequency */
	double prf; /* The pulse repetition frequency */

	double slant_range; /* Pitch in slant */
	double spacing_azimuth; /* Pitch in azimuth */

	int satellite_pass_direction; /* The direction of the satellite may be ascending or descending and it can be indicated by SatellitePassDirection */
	int metadata_type; /* Specifies the metadata from which the information was extracted and it can be indicated by MetadataType*/

} general_info, *General_info;

/**
 * \ingroup information_extractor
 * \fn State_vectors new_State_vectors(int nof_st_vect);
 * \brief Function used in order to allocate memory for a State_vectors_s structure
 * \return a pointer to a structure of state vectors
 */

State_vectors new_State_vectors(int nof_st_vect);

/**
 * \ingroup information_extractor
 * \fn void free_State_vectors(int nof_st_vect, State_vectors st);
 * \param nof_st_vect is the number of memory locations occupied by the structure
 * \param st is the pointer to a State_vectors_s structure which occupies the memory
 * \brief Function used in order to free the memory occupied by a State_vectors_s structure
 */
void free_State_vectors(int nof_st_vect, State_vectors st);

/**
 * \ingroup information_extractor
 * \fn General_info new_General_info();
 * \brief Function used in order to allocate memory for a general_info_s structure
 * \return a pointer to a structure of general information
 */
General_info new_General_info();

/**
 * \ingroup information_extractor
 * \fn void free_General_info(General_info gen_info);
 * \brief Function used in order to free the memory occupied by a General_info_s structure
 * \return a pointer to a structure of state vectors
 */
void free_General_info(General_info gen_info);

/**
 * \ingroup information_extractor
 * \brief Transition function that fetch information from an EFIDIRImage
 *        structure
 *
 * Instead of reading metadata files directly, this function will retrieve
 * them from an EFIDIRImage structure, where metadata was filled by the
 * respective drivers.
 * 
 * The idea is that later, modules should access directly the EFIDIRImage,
 * but meanwhile we can plug the driver-provided metadata in the current
 * code.
 *
 * \param image      Image storing the metadata
 * \param gen_infos  Structure that will be filled with the information
 */
void get_general_info_image(EFIDIRImage image, General_info gen_infos);

/**
 * \ingroup information_extractor
 * \fn void get_general_info_xml_tsx (char *data_in_file, General_info gen_info);
 * \brief Function used in order to extract the general information from the xml file for
 * TerraSAR-X product. This approach is valid only for xml of TerraSAR-X files. It fills
 * a General_info_s structure with the information founded between some specific xml tags.
 * \param data_in_file is the path to the xml file from where we will extract the information
 * \param gen_info is the structure that will be filled with the needed information
 */
void get_general_info_xml_tsx (char *data_in_file, General_info gen_info);

/**
 * \ingroup information_extractor
 * \fn void get_general_info_gamma_file (char *data_in_file, General_info gen_infos);
 * \brief Function used in order to extract the general information from a Gamma
 * output file. This approach is valid only for GAMMA output files.
 * It fills the General_info_s with the values founded in the data_in_file file.
 * \param data_in_file is the path to the gamma output file from where we will extract the information
 * \param gen_infos is the structure that will be filled with the needed information
 */
void get_general_info_gamma_file (char *data_in_file, General_info gen_infos);

/**
 * \ingroup information_extractor
 * \fn void get_general_info_xml_rsat2 (char *data_in_file, General_info gen_infos);
 * \brief Function used in order to extract the general information from a RADARSAT2 xml file.
 * This approach is valid only for RADARSAT-2 xml. It fills the General_info_s with
 * the values found in the data_in_file file.
 * \param data_in_file is the path to the gamma output file from where we will extract the information
 * \param gen_infos is the structure that will be filled with the needed information
 */
void get_general_info_xml_rsat2 (char *data_in_file, General_info gen_infos);

/**
 * \ingroup information_extractor
 * \brief Transition function that fetch state vectors from an EFIDIRImage
 *        structure
 *
 * Instead of reading metadata files directly, this function will retrieve
 * the state vectors from an EFIDIRImage structure, where metadata was 
 * filled by the respective drivers.
 * 
 * The idea is that later, modules should access directly the EFIDIRImage,
 * but meanwhile we can plug the driver-provided state vectors in the
 * current code.
 *
 * \param image       Image storing the metadata
 * \param st_vectors  Structure that will be filled with the information
 */
void get_state_vectors_from_image(EFIDIRImage image, State_vectors st_vectors);

/**
 * \ingroup information_extractor
 * \fn void get_state_vectors_from_xml_tsx(char *data_in_file, General_info gen_infos, State_vectors st_vect);
 * \brief Function used in order to extract the state vectors from the xml file
 * for a TerraSAR-X product. Pay attention in this case the position is computed
 * as the position extracted from the xml file + the position of the antenna (stored in gen_infos structure).
 * \param data_in_file is the TerraSAR-X xml file that should be provided in order to extract the state vectors.
 * \param gen_infos is a pointer to a structure that contains the number of state vectors and the position
 * of the antenna which should be used in order to compute the state vectors.
 * \param st_vect is a pointer to a structure that contains the state vectors for the current orbit
 * \return a pointer to a structure which contains the state vectors.
 */
void get_state_vectors_from_xml_tsx(char *data_in_file, General_info gen_infos, State_vectors st_vect);

/**
 * \ingroup information_extractor
 * \fn void get_state_vectors_from_Gamma_output_file(char *data_in_file, int nof_state_vectors, State_vectors st_vectors);
 * \brief Function used in order to extract the state vectors from Gamma output file.
 * \param data_in_file is the path to the gamma output file that should be provided in order to extract the state vectors.
 * \param nof_state_vectors is the number of state vectors that should be extracted
 * \param st_vectors is a pointer to a structure that contains the state vectors and which should be filled
 */
void get_state_vectors_from_Gamma_output_file(char *data_in_file, int nof_state_vectors, State_vectors st_vectors);

/**
 * \ingroup information_extractor
 * \fn void get_state_vectors_from_xml_rsat2(char *data_in_file, int nof_state_vectors, State_vectors st_vectors);
 * \brief Function used in order to extract the state vectors from a xml file for RADARSAT2 product.
 * \param data_in_file is the path to the xml file of RADARSAT-2 satellite that should be provided in order to extract the state vectors.
 * \param nof_state_vectors is the number of state vectors that should be extracted
 * \param st_vectors is a pointer to a structure that contains the state vectors and which should be filled
 */
void get_state_vectors_from_xml_rsat2(char *data_in_file, int nof_state_vectors, State_vectors st_vectors);

/**
 * \ingroup information_extractor
 * \fn State_vectors get_state_vectors_from_roi_pac(char *data_in_file);
 * \brief Function used in order to extract the state vectors from the roi_pac file.
 *
 * \param data_in_file is the roi_pac file that should be provided in order to extract the state vectors.
 * \return a pointer to a structure which contains the state vectors.
 */
State_vectors get_state_vectors_from_roi_pac(char *data_in_file);

/**
 * \ingroup information_extractor
 * \fn double utc2gps_time_conversion(char *time_utc);
 * \brief Function used in order to convert a time given in a UTC format to a time in a GPS format
 * \param time_utc is time in a UTC format
 * \return a double value which is the time GPS
 */
double utc2gps_time_conversion(char *time_utc);

/**
 * \ingroup information_extractor
 *
 * \fn void write_general_informations_2_xml_file(char *data_in_file, int nof_state_vectors, General_info gen_infos, State_vectors st_vect);
 * \brief This function generates a xml file which contains the informations holden by the
 * General_info and State_vector structures. It may be also used if there is no General_info structure.
 * \param data_in_file is the name of the file used to extract the information needed. The name of the output file is that of the input file followed by "general_info.xml"
 * \param nof_state_vectors is the number of state vectors that will be used. This is because this function may be used with other types of files than xml.
 * \param gen_infos is the pointer to a structure which contains the general informations. If it is not available than use NULL.
 * \param st_vect is a pointer to a structure which holds the state vectors
 */
void write_general_informations_2_xml_file(char *data_in_file, int nof_state_vectors, General_info gen_infos, State_vectors st_vect);


#endif /* SLT_INFO_EXTRACTION_H_ */
