/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "slt_orbit_calculations.h"
#include "slt_interpol_lut_ground2radar.h"

#ifndef SLT_GROUND_2_SAR_TRANSFORM_H
#define SLT_GROUND_2_SAR_TRANSFORM_H

/**
 * \ingroup sarlut_operators
 * \defgroup ground_2_SAR_transformations Some tools which may be used in order to adapt the SAR image on available DEM
 */

/**
 * \ingroup ground_2_SAR_transformations
 * \file slt_ground_2_SAR_transform.h
 * \author  Diana Rosu
 * \brief Some tools which may be used in order to adapt the SAR image on available DEM
 * file includes (stucture, prototype de fonction) of sarlut package
 */

/**
 * \ingroup ground_2_SAR_transformations
 * \struct SAR_geolocalisation_matrix_in_master_slave_s
 * \brief This structure holds the mapping of the master and slave images obtained over the DEM on the SAR image (only range and azimuth)
 */
typedef struct SAR_geolocalisation_matrix_in_master_slave_s {

	double **azimuth_on_master; /* The corresponding azimuth for the current position on the master image */
	double **azimuth_on_slave; /* The corresponding azimuth for the current position in the master image on the slave image obtained over the DEM */
	double **range_on_master; /* The corresponding range of SAR image on the master image obtained over DEM */
	double **range_on_slave; /* The corresponding range of SAR image on the slave image obtained over DEM */

} sar_geolocalisation_matrix_MS, *SAR_geolocalisation_matrix_MS;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct Master_lat_long_geolocalisation_matrix_s
 * \brief This structure holds the mapping of the master image obtained over the DEM (lat, long, alt and range)
 */
typedef struct Master_lat_long_geolocalisation_matrix_s {

	double **DEM_lat; /* The latitude of the DEM */
	double **DEM_long; /* The longitude of the DEM */
	double **DEM_h; /* The altitude of the DEM*/
	double **range_on_master; /* The corresponding range of SAR image on the master image obtained over DEM */

} lat_long_geolocalisation_4_master, *Lat_Long_geolocalisation_4_master;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct Azimuth_image_statistics
 * \brief This structure holds some statistics of the azimuth image. The statistics are: the minimum value, the maximum value and the histogram.
 */
typedef struct Azimuth_image_statistics_s {
	double max_azimuth; /* The rounded minimum value in the azimuth image */
	double min_azimuth; /* The rounded maximum value in the azimuth image */
	int *hist_azimuth; /* The histogram of the rounded values in the azimuth image */
} azimuth_image_statistics, *Azimuth_image_statistics;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct range_image_statistics
 * \brief This structure holds some statistics of the range image. The statistics are: the minimum value and the maximum value
 */
typedef struct range_image_statistics_s{
	double max_range; /* The rounded minimum value in the range image */
	double min_range; /* The rounded maximum value in the range image */
} range_image_statistics;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct cropping_infos
 * \brief This structure holds some informations about the cropping process
 */
typedef struct Cropping_info_s{
	int min_line;
	int max_line;
	int min_column;
	int max_column;
	int additional_nof_lines;
	int additional_nof_columns;
} cropping_info;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct interpolation_lat_long_sar2ground_s
 * \brief This structure holds some informations about the kind of interpolation that will be used in order to bring an image
 * in ground geometry in SAR geometry and also the values that should be ignored or should be added as ignored values.
 * This should work for a multiband image.
 */
typedef struct interpolation_lat_long_sar2ground_s{
	int interpolation_type;
	float *input_ignored_val, *output_ignored_val;
} interpolation_lat_long_sar2ground, *Interpolation_lat_long_sar2ground;

/**
 * \ingroup ground_2_SAR_transformations
 * \struct interpolation_range_azimuth_ground2sar_s
 * \brief This structure holds some informations about the kind of interpolation that will be used in order to bring an image
 *  in SAR geometry in ground geometry and also the values that should be ignored or should be added as ignored values.
 */
typedef struct interpolation_range_azimuth_ground2sar_s {
	int interpolation_type;
	float input_ignored_val, output_ignored_val;
} interpolation_range_azimuth_ground2sar, *Interpolation_range_azimuth_ground2sar;

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Interpolation_lat_long_sar2ground_s new_interpolation_lat_long_sar2ground_s(int nof_bands);
 * \brief Function used in order to allocate memory for an Interpolation_lat_long_sar2ground_s structure.
 * \param nof_bands is the number of bands of a certain image. We need to know this value in order to allocate enough memory for the vectors which holds the ignored values.
 * \return a pointer to a structure which contains ignored values and the type of the interpolation
 */
Interpolation_lat_long_sar2ground new_interpolation_lat_long_sar2ground_s(int nof_bands);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Interpolation_range_azimuth_ground2sar new_interpolation_range_azimuth_ground2sar();
 * \brief Function used in order to allocate memory for an Interpolation_range_azimuth_ground2sar structure.
 * \return a pointer to a structure which contains ignored values and the type of the interpolation
 */
Interpolation_range_azimuth_ground2sar new_interpolation_range_azimuth_ground2sar();

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void define_interpolation_lat_long_sar2ground_param(char *extra_description);
 * \brief Function used in order to obtain from the user the values of the Interpolation_lat_long_sar2ground operator when it is used in a program
 * \param extra_description is the extra description
 */
void define_interpolation_lat_long_sar2ground_param(char *extra_description);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void define_interpolation_range_azimuth_ground2sar_param(char *extra_description);
 * \brief Function used in order to obtain from the user the values of the Interpolation_range_azimuth_ground2sar operator when it is used in a program
 * \param extra_description is the extra description
 */
void define_interpolation_range_azimuth_ground2sar_param(char *extra_description);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Interpolation_lat_long_sar2ground get_interpolation_lat_long_sar2ground_param(int nof_bands);
 * \brief Function used in order to retrieve the values provided by the user in the Interpolation_lat_long_sar2ground parameter structure
 * \param nof_bands specifies the number of bands for the image that should be interpolated
 * \return a pointer to a structure which contains ignored values and the type of the interpolation
 */
Interpolation_lat_long_sar2ground get_interpolation_lat_long_sar2ground_param(int nof_bands);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Interpolation_range_azimuth_ground2sar get_interpolation_range_azimuth_ground2sar_param();
 * \brief Function used in order to retrieve the values provided by the user in the Interpolation_range_azimuth_ground2sar parameter structure
 * \return a pointer to a structure which contains ignored values and the type of the interpolation
 */
Interpolation_range_azimuth_ground2sar get_interpolation_range_azimuth_ground2sar_param();

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void free_Interpolation_lat_long_sar2ground(Interpolation_lat_long_sar2ground param);
 * \param param A pointer to the structure for which we want to free the memory
 * \brief Free an allocated Interpolation_lat_long_sar2ground reference
 *
 */
void free_Interpolation_lat_long_sar2ground(Interpolation_lat_long_sar2ground param);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void free_Interpolation_range_azimuth_ground2sar(Interpolation_range_azimuth_ground2sar param);
 *
 * \param param A pointer to the structure for which we want to free the memory
 *
 * \brief Free an allocated Interpolation_range_azimuth_ground2sar reference
 *
 */
void free_Interpolation_range_azimuth_ground2sar(Interpolation_range_azimuth_ground2sar param);


/**
 * \ingroup ground_2_SAR_transformations
 * \fn Azimuth_image_statistics new_azimuth_image_statistics(int nof_his_elems);
 * \brief Function used in order to allocate memory for a Azimuth_image_statistics structure
 * \param nof_his_elems is the number of elements for each element in the histogram
 * \return a pointer to a structure which contains the statistics on an azimuth image
 */
Azimuth_image_statistics new_azimuth_image_statistics(int nof_his_elems);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void free_Azimuth_image_statistics_s (Azimuth_image_statistics azim_stats);
 * \param azim_stats is the Azimuth_image_statistics structure that should be freed
 * \brief Function used in order to deallocate the memory for an Azimuth_image_statistics_s structure
 */
void free_Azimuth_image_statistics_s (Azimuth_image_statistics azim_stats);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Azimuth_image_statistics get_azimuth_image_statistics(EFIDIRImage imAzimuth, EFIDIRImage imRange, int ignored_value);
 * \brief Function used in order to compute some statistics on the azimuth image
 * \param imAzimuth is the Azimuth image on which we compute the statistics
 * \param imRange is the Range image for the same image which should be used in order to compute the
 * statistics just for those regions of the image where there are valid values in both azimuth and range images
 * \param ignored_value is the value of the Azimuth image that should not be considered when we make the statistics
 * \return a pointer to a structure which contains the statistics on an azimuth image
 */
Azimuth_image_statistics get_azimuth_image_statistics(EFIDIRImage imAzimuth, EFIDIRImage imRange, int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn range_image_statistics get_range_image_statistics(EFIDIRImage imRange, int ignored_value);
 * \brief Function used in order to compute some statistics on the range image
 * \param imRange is the Range image on which we compute the statistics
 * \param ignored_value is the value of the Range image that should not be considered when we make the statistics
 * \return a structure which contains the statistics on an range image, those are the maximum and the minimum value
 */
range_image_statistics get_range_image_statistics(EFIDIRImage imRange,
		int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn SAR_geolocalisation_matrix_MS new_SAR_geolocalisation_matrix_MS(int nof_lines, int *nof_columns);
 * \brief Function used in order to allocate memory for a SAR_geolocalisation_matrix_MS structure
 * \param nof_lines is the number of lines of the future matrix
 * \param nof_columns is a pointer to a vector which specifies the number of columns for each line in the future matrix
 * \return a pointer to a structure which defines the range/azimuth in both the master and the slave images
 */
SAR_geolocalisation_matrix_MS new_SAR_geolocalisation_matrix_MS(int nof_lines, int *nof_columns);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn Lat_Long_geolocalisation_4_master new_Lat_Long_geolocalisation_4_master(int nof_lines, int *nof_columns);
 * \brief Function used in order to allocate memory for a Lat_Long_geolocalisation_4_master structure
 * \param nof_lines is the number of lines of the future matrix
 * \param nof_columns is a pointer to a vector which specifies the number of columns for each line in the resulting matrix
 * \return a pointer to a structure which defines the latitude/longitude/altitude in the master SAR image.
 */
Lat_Long_geolocalisation_4_master new_Lat_Long_geolocalisation_4_master(int nof_lines, int *nof_columns);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void free_SAR_geolocalisation_matrix_MS(int nof_lines, SAR_geolocalisation_matrix_MS info_matrix);
 * \param nof_lines is the number of lines of the matrix represented by info_matrix
 * \param info_matrix is the matrix builded considering the master azimuth which should be freed
 * \brief Function used in order to deallocate the memory for an SAR_geolocalisation_matrix_MS structure
 */
void free_SAR_geolocalisation_matrix_MS(int nof_lines, SAR_geolocalisation_matrix_MS info_matrix);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void free_Lat_Long_geolocalisation_4_master(int nof_lines, Lat_Long_geolocalisation_4_master info_matrix);
 * \param nof_lines is the number of lines of the matrix represented by info_matrix
 * \param info_matrix is the matrix builded considering the master azimuth which should be freed
 * \brief Function used in order to deallocate the memory for an Lat_Long_geolocalisation_4_master structure
 */
void free_Lat_Long_geolocalisation_4_master(int nof_lines, Lat_Long_geolocalisation_4_master info_matrix);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn int get_median_in_vector(float *vector, int vector_size);
 * \brief Function used in order to compute the median value of a specific vector conveyed as parameter
 * \param vector is the vector where we should search for the median value
 * \param vector_size is the length of the vector
 * \return a float value which is the median value of the vector.
 */
int get_median_in_vector(float *vector, int vector_size);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn int *get_delta_master_slave_azimuth4matrix(int **slave_azimuth_matrix, Azimuth_image_statistics stats);
 * \brief Function used in order to compute difference between the values of master azimuth and the median value of the slave azimuth for the entire matrix in respect with the "azimuth on slave" field of the matrix. This function should be used to test if there is no repetition of lines for slave resampled SAR image
 * \param slave_azimuth_matrix is the matrix of slave azimuth that was build in a previous step
 * \param stats are the statistics for the master azimuth image
 * \return a pointer to a vector which contains the values for the difference (master_azimuth - median_value4slave_azimuth)
 */
int *get_delta_master_slave_azimuth4matrix(int **slave_azimuth_matrix, Azimuth_image_statistics stats);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn SAR_geolocalisation_matrix_MS build_MS_SAR_geolocalisation_matrix(int descending_track, EFIDIRImage Azim_master, EFIDIRImage Azim_slave, EFIDIRImage Range_master, EFIDIRImage Range_slave, int ignored_value);
 * \brief Function used in order to obtain the matrix of structures holding the values of azimuth slave, range for both master and slave SAR images
 * \param descending_track specifies if the current track is an ascending or a descending one. If descending this parameter should be one otherwise should be zero.
 * \param Range_master is the image of range for the master image
 * \param Range_slave is the image of range for the slave image
 * \param Azim_master is the image of azimuth for the master image
 * \param Azim_slave is the image of azimuth for the slave image
 * \param ignored_value is the value that should be ignored in all images.
 * \return a pointer to a structure contain the sar_geolocalisation matrix.
 */
SAR_geolocalisation_matrix_MS build_MS_SAR_geolocalisation_matrix(int descending_track, EFIDIRImage Azim_master, EFIDIRImage Azim_slave, EFIDIRImage Range_master, EFIDIRImage Range_slave, int ignored_value);
/**
 * \ingroup ground_2_SAR_transformations
 * \fn Lat_Long_geolocalisation_4_master build_master_lat_long_geolocalisation_matrix(int descending_track, EFIDIRImage Azim_master, EFIDIRImage Range_master, EFIDIRImage imDEM, int ignored_value);
 * \brief Function used in order to obtain the matrix of geographic informations for master SAR.
 * \param descending_track specifies if the current track is an ascending or a descending one. If descending this parameter should be one otherwise should be zero.
 * \param Range_master is the image of range for the master image
 * \param Azim_master is the image of azimuth for the master image
 * \param imDEM is the DEM used in order to obtain all images enumerated above
 * \param ignored_value is the value that should be ignored in all images.
 * \return a pointer to a structure contain the lla_geolocalisation matrix.
 */
Lat_Long_geolocalisation_4_master build_master_lat_long_geolocalisation_matrix(int descending_track, EFIDIRImage Azim_master, EFIDIRImage Range_master, EFIDIRImage imDEM, int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn int constant_difference_master_slave_azimuth(int **azimuth_on_slave, Azimuth_image_statistics stats);
 * \brief Function is used in order to see if the list of azimuth slave has a constant difference compared with master azimuth
 * \param azimuth_on_slave is list of slave values on lines
 * \param stats are the statistics of the master slave
 * \return a zero if the difference between the master and slave azimuth is non-constant and a non-zero value which specifies the difference between master and slave azimuth is the difference is constant
 */
int constant_difference_master_slave_azimuth(int **azimuth_on_slave, Azimuth_image_statistics stats);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void gen_full_precision_resampled_slave_SAR_image_on_master_grid(int interpolation_type, int nof_lobs, EFIDIRImage masterSAR, EFIDIRImage slaveSAR, EFIDIRImage imMask, SAR_geolocalisation_matrix_MS ground_on_sar_l, Azimuth_image_statistics stats, EFIDIRImage resampledSlaveSAR, int ignored_value);
 * \brief Function is used in order generate the slave resampled SAR image on master SAR grid in a full precision manner
 * \param interpolation_type specifies which kind of interpolation will be used. The valid values for this parameter are: 1 for nearest neighbor interpolation, 2 for bilinear interpolation and 3 for sinc 2D interpolation.
 * \param nof_lobs is a parameter which is relevant only if the interpolation that will be performed is a sinc interpolation, otherwise it has no meaning.
 * \param masterSAR is the SAR master image
 * \param slaveSAR is the SAR slave image
 * \param imMask is the resulting foldover mask
 * \param ground_on_sar_l is the matrix containing the range and azimuth for both master and slave images
 * \param stats are the statistics obtained for the master azimuth image
 * \param resampledSlaveSAR is the resulting image and is the slave SAR image resampled on master grid
 * \param ignored_value is the value that will be used where there is no DEM
 */
void gen_full_precision_resampled_slave_SAR_image_on_master_grid(int interpolation_type, int nof_lobs, EFIDIRImage masterSAR, EFIDIRImage slaveSAR, EFIDIRImage imMask, SAR_geolocalisation_matrix_MS ground_on_sar_l, Azimuth_image_statistics stats, EFIDIRImage resampledSlaveSAR, int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void gen_approximative_resampled_slave_SAR_image_on_master_grid(int interpolation_type, int nof_lobs, EFIDIRImage masterSAR, EFIDIRImage slaveSAR, EFIDIRImage imMask, SAR_geolocalisation_matrix_MS ground_on_sar_l, Azimuth_image_statistics stats, EFIDIRImage resampledSlaveSAR, int ignored_value);
 * \brief Function is used in order generate the slave resampled SAR image on master SAR grid in an approximative manner
 * \param interpolation_type specifies which kind of interpolation will be used. The valid values for this parameter are: 1 for nearest neighbor interpolation, 2 for linear interpolation and 3 for sinc 1D interpolation.
 * \param nof_lobs is a parameter which is relevant only if the interpolation that will be performed is a sinc interpolation, otherwise it has no meaning.
 * \param masterSAR is the SAR master image
 * \param slaveSAR is the SAR slave image
 * \param imMask is the resulting foldover mask
 * \param ground_on_sar_l is the matrix containing the range and azimuth for both master and slave images
 * \param stats are the statistics obtained for the master azimuth image
 * \param resampledSlaveSAR is the resulting image and is the slave SAR image resampled on master grid
 * \param ignored_value is the value that will be used where there is no DEM
 */
void gen_approximative_resampled_slave_SAR_image_on_master_grid(int interpolation_type, int nof_lobs, EFIDIRImage masterSAR, EFIDIRImage slaveSAR, EFIDIRImage imMask, SAR_geolocalisation_matrix_MS ground_on_sar_l, Azimuth_image_statistics stats, EFIDIRImage resampledSlaveSAR, int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void gen_LUTs_on_master_grid(EFIDIRImage masterSAR, EFIDIRImage imLAT, EFIDIRImage imLONG, EFIDIRImage imALT, EFIDIRImage imMask, Lat_Long_geolocalisation_4_master lla_geolocalisation_matrix, Azimuth_image_statistics stats, int ignored_value);
 * \brief Function is used in order generate the correspondents of LUTs in older sarlut, this is the images of latitude, longitude and altitude in SAR geometry.
 * \param masterSAR is the SAR master image
 * \param imLAT is the output LUT for the latitude
 * \param imLONG is the output LUT for the longitude
 * \param imALT is the output LUT for the altitude
 * \param imMask is the resulting foldover mask
 * \param lla_geolocalisation_matrix is the lat_long_localisation matrix of the master image containing the latitude/longitude and altitude on the DEM
 * \param stats are the statistics obtained for the master azimuth image
 * \param ignored_value is the value that will be used where there is no DEM
 */
void gen_LUTs_on_master_grid(EFIDIRImage masterSAR, EFIDIRImage imLAT, EFIDIRImage imLONG, EFIDIRImage imALT, EFIDIRImage imMask, Lat_Long_geolocalisation_4_master lla_geolocalisation_matrix, Azimuth_image_statistics stats, int ignored_value);

void gen_delta_azimuth_and_range_master_slave(EFIDIRImage masterSAR, SAR_geolocalisation_matrix_MS ground_on_sar_l, Azimuth_image_statistics stats, EFIDIRImage deltaAzimuthMasterResampledSlave, EFIDIRImage deltaRangeMasterResampledSlave, int ignored_value);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void crop_SAR_image(EFIDIRImage imSAR, EFIDIRImage imSAR_cropped, cropping_info crop_SAR_image_guidelines);
 * \brief Function used in order to crop a SAR image for a specific DEM. So the user dispose either of range and azimuth images for this or of some information which should be conveyed to the function as crop_SAR_image_guidelines parameter.
 * \param imSAR is the SAR image to be cropped
 * \param imSAR_cropped is the SAR image cropped
 * \param crop_SAR_image_guidelines is a structure which holds the guidelines for the cropping process
 */
void crop_SAR_image(EFIDIRImage imSAR, EFIDIRImage imSAR_cropped, cropping_info crop_SAR_image_guidelines);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn double bilinear_approximation(EFIDIRImage input_img, double column, double line, int real_part);double bilinear_approximation(EFIDIRImage input_img, double column, double line, int real_part);v
 * \brief Function used in order to obtain a value in an image using a bilinear approximation approach. This is a refactorization of the function approx_bilin which resides in /sarlut/slt_geom2D.c. It supports not just EFIDIRImageFloat but almost all types of images. Also if one of the origins designates a certain pixel (integer value) this means that an linear interpolation will be performed.
 * \param input_img is the image where we should perform the interpolation
 * \param column is the origin where to compute on columns
 * \param line is the origin where to compute on lines
 * \param real_part specifies if the returned value is for the real part (if this is 1) or for the imaginary part if it is zero. This is applied only if the datatype of the image is complex (4, 8 or 16), for the other datatypes this parameter is ignored.
 * \return a double value which is the interpolated value in a certain pixel
 */
double bilinear_approximation(EFIDIRImage input_img, double column, double line, int real_part);
/**
 * \ingroup ground_2_SAR_transformations
 * \fn double sinc_interpolation_1D(EFIDIRImage imgInput, double column, int line, int real_part, int nof_lobs);
 * \brief Function used in order to obtain a value in an image using a sinc interpolation mono-dimensional on the line of azimuth. This should be applied considering the type of the input image and the number of lobs that we want to include in our interpolation.
 * \param imgInput is the image where we should perform the interpolation
 * \param column is the column in the image where to compute the value. It is double because it can be used in a sub-pixel approach.
 * \param line is the line on which we perform the interpolation.
 * \param real_part is the parameter which becomes significant if the image where to compute is a complex one. If this parameter is a non-zero one then the interpolation is realized on the real part of the complex image otherwise on the imaginary part. If the provided image is a float one then this parameter is irrelevant.
 * \param nof_lobs is the parameter which specifies the number of lobs that will be considered in the interpolation process.
 * \return a double value which is the interpolated value in a certain pixel
 */
double sinc_interpolation_1D(EFIDIRImage imgInput, double column, int line, int real_part, int nof_lobs);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn double sinc_interpolation_2D(EFIDIRImage imgInput, double column, double line, int real_part, int nof_lobs);
 * \brief Function used in order to obtain a value in an image using a sinc interpolation in a bi-dimensional manner (both on lines and columns). This should be applied considering the type of the input image and the number of lobs that we want to include in our interpolation.
 * \param imgInput is the image where we should perform the interpolation
 * \param column is the column in the image where to compute the value. It is double because it can be used in a sub-pixel approach.
 * \param line is the line on which we perform the interpolation.
 * \param real_part is the parameter which becomes significant if the image where to compute is a complex one. If this parameter is a non-zero one then the interpolation is realized on the real part of the complex image otherwise on the imaginary part. If the provided image is a float one then this parameter is irrelevant.
 * \param nof_lobs is the parameter which specifies the number of lobs that will be considered in the interpolation process.
 * \return a double value which is the interpolated value in a certain pixel
 */
double sinc_interpolation_2D(EFIDIRImage imgInput, double column, double line, int real_part, int nof_lobs);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void interpol_lut_georef_calc_multi_band(Simu_param sim, Interpolation_lat_long_sar2ground des, EFIDIRImage iminit, EFIDIRImage imLong, EFIDIRImage imLat, EFIDIRImage imres);
 * \brief Function used in order to bring in radar geometry an image which is represented in ground geometry. The input image may be either the DEM, the difference of distance image, the range image, the azimuth image or an optical image.
 * \param sim is the simulation parameter
 * \param des is the parameter which specifies which are the ignored values in the input and output images and also the interpolation type
 * \param iminit is the image that should be brought in SAR geometry
 * \param imLong is the longitude image in SAR geometry
 * \param imLat is the latitude image in SAR geometry
 * \param imres is the output image in SAR geometry
 */
void interpol_lut_georef_calc_multi_band(Simu_param sim, Interpolation_lat_long_sar2ground des, EFIDIRImage iminit, EFIDIRImage imLong, EFIDIRImage imLat, EFIDIRImage imres);

/**
 * \ingroup ground_2_SAR_transformations
 * \fn void interpol_lut_2_bring_sar_on_ground(Interpolation_range_azimuth_ground2sar ground2SAR_param, EFIDIRImage imAmplitude, EFIDIRImage imRange, EFIDIRImage imAzimuth, EFIDIRImage Amplitude_on_ground);
 * \brief Function used in order to bring in ground geometry an image represented in SAR geometry. We use to utilize this function in order to bring the amplitude image in ground geometry.
 * \param ground2SAR_param is the parameter which specifies which are the ignored values in the input and output images and also the interpolation type
 * \param imAmplitude is the image that should be brought in ground geometry
 * \param imRange is the image of range (in ground geometry)
 * \param imAzimuth is the image of azimuth (in ground geometry)
 * \param Amplitude_on_ground is the output image in ground geometry
 */
void interpol_lut_2_bring_sar_on_ground(Interpolation_range_azimuth_ground2sar ground2SAR_param, EFIDIRImage imAmplitude, EFIDIRImage imRange, EFIDIRImage imAzimuth, EFIDIRImage Amplitude_on_ground);

#endif /* SLT_GROUND_2_SAR_TRANSFORM_H */
