/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup sarlut_operators
 * \defgroup geom2D 2D geometric transformation
 */
/**
 * \ingroup geom2D
 * \file slt_geom2D.h
 * \author Ivan Petillot and Pascal Bellemain
 * 
 * \brief operator of geometric transformation in 2D
 * 
 * file includes (stucture, prototype de fonction) of the operator geom2D
 * 
 * 
 */

#ifndef SLT_GEOM2D_H
#define  SLT_GEOM2D_H

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "efidir_image.h"

#define MaxDimEq 16


/* *******************************  TYPES ***********************************/
/**
 * \ingroup geom2D
 * \struct point_2d
 *
 * \brief structure that contains the localisation of a point in 2D
 *
 */
typedef struct // structure de 2 champs de 8 octets chacun
{
	double coordI;
	double coordJ;
} point_2d;

/**
 * \ingroup geom2D
 * \struct coord_2d
 *
 * \brief structure that contains the 2D coordonates of a point
 *
 */
typedef struct // structure de 2 champs de 8 octets chacun
{
	double coordX;
	double coordZ;
} coord_2d;

/**
 * \ingroup geom2D
 * \struct coord_3d
 *
 * \brief structure that contains the 3D coordonates of a point
 *
 */
typedef struct // structure de 3 champs de 8 octets chacun
{
	double coordX;
	double coordY;
	double coordZ;
} coord_3d;

/**
 * \ingroup geom2D
 * \struct coord_4d
 *
 * \brief structure that contains the 4D coordonates of a point (3D + time)
 *
 */
typedef struct // structure de 3 champs de 8 octets chacun
{
	double coordX;
	double coordY;
	double coordZ;
	double temps;
} coord_4d;

/**
 * \ingroup geom2D
 * \struct droite_reglin
 *
 * \brief structure that contains the parameters for linear regression
 *
 */
typedef struct // resultat du calcul de la regression linéaire, sert également pour une équation de droite
{
	double a; // coefficient directeur
	double b; // ordonnée à l'origine
	double r;
	double varianceX;
	double varianceY;
	double covariance;
} droite_reglin; // en 2D

/**
 * \ingroup geom2D
 * \struct droite_par_points_3D
 *
 * \brief structure that contains the parameters for 3D linear regression
 *
 */
typedef struct // resultat du calcul de la regression linéaire, sert également pour une équation de droite
{
	coord_3d P0;
	coord_3d P1;
} droite_par_points_3D;


typedef float matrice_3x3[3][3];
typedef float matrice_2x2[2][2];
typedef float vecteur_3[3];
typedef float vecteur_2[2];



/* *******************************  FONCTIONS DIVERSES ***********************************/

/**
 * \ingroup geom2D
 * \fn  void regressionXY(coord_3d *points,int N,droite_reglin *result);
 * \brief function of linear regression
 * Function allowing realising a linear regression
 *
 * \param points is the coordonates 3D of point
 * \param N is the number of points 
 * \param result is the result of linear regression
 *
 */
void regressionXY(coord_3d *points,int N,droite_reglin *result);
// void regressionXY4d(coord_4d *points,int N,droite_reglin *result);


/**
 * \ingroup geom2D
 * \fn EFIDIRImageFloat init_sobel();
 * \brief function of initialisation of Sobel filter
 *
 */
EFIDIRImageFloat init_sobel();

/**
 * \ingroup geom2D
 * \fn EFIDIRImageFloat init_sobel_derivx();
 * \brief function of initialisation of Sobel filter in direction Ox
 *
 */
EFIDIRImageFloat init_sobel_derivx();

/**
 * \ingroup geom2D
 * \fn EFIDIRImageFloat init_sobel_derivy();
 * \brief function of initialisation of Sobel filter in direction Oy
 *
 */
EFIDIRImageFloat init_sobel_derivy();


/* *************************  TRANSFORMATIONS GEOMETRIQUES *******************************/
/**
 * \ingroup geom2D
 * \fn  void rotation_ppv(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot) ;
 * \brief function of application of roration to an image
 * Function allowing rotating an image
 *
 * \param angle is the rotation angle to apply
 * \param imOrig is the original image
 * \param imRot is the image after rotation
 *
 */
void rotation_ppv(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot);

/**
 * \ingroup geom2D
 * \fn  void rotation_decoupe_ppv(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot) ;
 * \brief function of application of roration to an image and cutting
 * Function allowing rotating and cutting an image
 *
 * \param angle is the rotation angle to apply
 * \param imOrig is the original image
 * \param imRot is the image after rotation
 *
 */
void rotation_decoupe_ppv(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot);

/**
 * \ingroup geom2D
 * \fn  void rotation_decoupe_bilineaire(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot) ;
 * \brief function of application of roration to an image and cutting
 * Function allowing rotating and cutting an image
 *
 * \param angle is the rotation angle to apply
 * \param imOrig is the original image
 * \param imRot is the image after rotation
 *
 */
void rotation_decoupe_bilineaire(double angle, EFIDIRImageFloat imOrig, EFIDIRImageFloat imRot);

/**
 * \ingroup geom2D
 * \fn  void flipcpyfl_DG(EFIDIRImageFloat imOrig, EFIDIRImageFloat imFlip);
 * \brief function of application of flip to a float image
 * Function allowing flipping a float image (left/right)
 *
 * \param imOrig is the original image
 * \param imFlip is the image after flip
 *
 */
void flipcpyfl_DG(EFIDIRImageFloat imOrig, EFIDIRImageFloat imFlip);

/**
 * \ingroup geom2D
 * \fn void flipfl_DG(EFIDIRImageFloat im);
 * \brief function of application of flip to a float image
 * Function allowing flipping a float image (left/right) and rewriting on top
 *
 * \param im is the float image to flip
 *
 */
void flipfl_DG(EFIDIRImageFloat im);

/**
 * \ingroup geom2D
 * \fn void flipu1_DG(EFIDIRImageByte im);
 * \brief function of application of flip to a 8bytes image
 * Function allowing flipping a 8bytes image (left/right) and rewriting on top
 *
 * \param im is the 8bytes image to flip
 *
 */
void flipu1_DG(EFIDIRImageByte im);


void imaflcpy(EFIDIRImageFloat imOrig, EFIDIRImageFloat imRes);
void imau1cpy(EFIDIRImageByte imOrig, EFIDIRImageByte imRes);


/* **************************  CALCULS en 3D  *******************************/
/**
 * \ingroup geom2D
 * \fn double produit_scalaire_3d(coord_3d V1,coord_3d V2);
 * \brief function of calculating the scalar product of 2 vectors
 *
 * \param V1 is the first vector
 * \param V2 is the second vector
 * \return the result of the scalar product
 *
 */
double produit_scalaire_3d(coord_3d V1,coord_3d V2);

/**
 * \ingroup geom2D
 * \fn coord_3d soustraction_points_3d(coord_3d P0, coord_3d P1);
 * \brief function of subtraction of 2 points in 3D
 *
 * \param P0 is the first point
 * \param P1 is the second point
 * \return the result of P0-P1
 *
 */
coord_3d soustraction_points_3d(coord_3d P0, coord_3d P1);

/**
 * \ingroup geom2D
 * \fn coord_3d addition_points_3d(coord_3d P0, coord_3d P1);
 * \brief function of addition of 2 points in 3D
 *
 * \param P0 is the first point
 * \param P1 is the second point
 * \return the result of P0+P1
 *
 */
coord_3d addition_points_3d(coord_3d P0, coord_3d P1);

/**
 * \ingroup geom2D
 * \fn coord_3d multiplication_scalaire_vecteur_3d(double a, coord_3d V);
 * \brief function of calculation the multiplication of a scalar and a vector
 *
 * \param a is the scalar
 * \param V is the vector
 * \return the result of a*V
 *
 */
coord_3d multiplication_scalaire_vecteur_3d(double a, coord_3d V);

/**
 * \ingroup geom2D
 * \fn coord_3d projection_point_sur_droite_3d(coord_3d P, droite_par_points_3D L);
 * \brief function of projection of a point on a line
 *
 * \param P is the point
 * \param L is the line
 * \return the projected point
 *
 */
coord_3d projection_point_sur_droite_3d(coord_3d P, droite_par_points_3D L);

/**
 * \ingroup geom2D
 * \fn double distance_points_3d(coord_3d P0, coord_3d P1);
 * \brief function of calculation of distance between 2 points 3D
 *
 * \param P0 is the first point
 * \param P1 is the second point
 * \return the distance between these two points
 *
 */
double distance_points_3d(coord_3d P0, coord_3d P1);



/* *************************  CALCULS DIVERS  *******************************/
/**
 * \ingroup geom2D
 * \fn double approx_bilin(EFIDIRImageFloat im, double Xorig, double Yorig);
 * \brief function of simple bilinear approximation on a float image
 *
 * \param im is the original image
 * \param Xorig is the precise abscissa of pixel
 * \param Yorig is the precise ordonate of pixel
 * \return bilinear interpolation value of between pixels of image
 *
 */

double approx_bilin(EFIDIRImageFloat im, double Xorig, double Yorig);

/**
 * \ingroup geom2D
 * \fn double approx_bilin_robuste(EFIDIRImageFloat im, double Xorig, double Yorig, float inut_read, float inut_write);
 * \brief function of simple bilinear approximation on a float image with verification
 * Function of simple bilinear approximation on a float image with verification of limit
 * of images and invalid value
 *
 * \param im is the original image
 * \param Xorig is the precise abscissa of pixel
 * \param Yorig is the precise ordonate of pixel
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 * \return bilinear interpolation value of between pixels of image
 *
 */
double approx_bilin_robuste(EFIDIRImageFloat im, double Xorig, double Yorig, float inut_read, float inut_write);

/**
 * \ingroup geom2D
 * \fn double approx_bilin_u1(EFIDIRImageByte im, double Xorig, double Yorig);
 * \brief function of simple bilinear approximation on a 8bytes image
 *
 * \param im is the original image
 * \param Xorig is the precise abscissa of pixel
 * \param Yorig is the precise ordonate of pixel
 * \return bilinear interpolation value of between pixels of image
 *
 */
double approx_bilin_u1(EFIDIRImageByte im, double Xorig, double Yorig);

/**
 * \ingroup geom2D
 * \fn double approx_baryc(EFIDIRImageFloat im, double Xorig, double Yorig);
 * \brief function of barycentric approximation on a float image
 *
 * \param im is the original image
 * \param Xorig is the precise abscissa of pixel
 * \param Yorig is the precise ordonate of pixel
 * \return  barycentric interpolation value of between pixels of image
 *
 */
double approx_baryc(EFIDIRImageFloat im, double Xorig, double Yorig);

/* *************************  TRANSFORMATIONS MATRICIELLES  *******************************/
/**
 * \ingroup geom2D
 * \fn float det(float a[MaxDimEq][MaxDimEq],int n);
 * \brief function of calculation of determinant of a matrix
 *
 * \param a is the matrix
 * \param n is the dimension of matrix
 * \return  the determinant of matrix a
 *
 */
float det(float a[MaxDimEq][MaxDimEq],int n);

/**
 * \ingroup geom2D
 * \fn void inversion(float a[MaxDimEq][MaxDimEq],float a_1[MaxDimEq][MaxDimEq],int n);
 * \brief function of calculation of inverse of a matrix
 *
 * \param a is the matrix
 * \param a_1 is the inverse matrix of a
 * \param n is the dimension of matrix a
 * \return  the determinant of matrix a
 *
 */
void inversion(float a[MaxDimEq][MaxDimEq],float a_1[MaxDimEq][MaxDimEq],int n);

/**
 * \ingroup geom2D
 * \fn void affiche_matrice3x3(matrice_3x3 matrice);
 * \brief function of display of a matrix 3x3
 *
 * \param matrice is the matrix to display
 *
 */
void affiche_matrice3x3(matrice_3x3 matrice);

/**
 * \ingroup geom2D
 * \fn void RST_param2matrice(float rotation, vecteur_2 scale, vecteur_2 translation, matrice_3x3 matrice_RST);
 * \brief function of calculation of coefficients matrix of RST transformation
 *
 * \param rotation is the rotation of the RST in radians in the clockwise
 * \param scale is the scale of RST depending on X and Y
 * \param translation is the translation of RST depending on X and Y
 * \param matrice_RST is the RST matrix
 *
 */
void RST_param2matrice(float rotation, vecteur_2 scale, vecteur_2 translation, matrice_3x3 matrice_RST);

/**
 * \ingroup geom2D
 * \fn void RST_transform_inverse_ppv_u1(matrice_3x3 matrice_RST, EFIDIRImageByte imInit, EFIDIRImageByte imRes, int inut_read, int inut_write);
 * \brief function of calculation of RST matrix by a inverse approach (8 bytes image)
 *
 * \param matrice_RST is the RST matrix
 * \param imInit is the initial 8 bytes image to transform
 * \param imRes is the transformed 8 bytes image
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 *
 */
void RST_transform_inverse_ppv_u1(matrice_3x3 matrice_RST, EFIDIRImageByte imInit, EFIDIRImageByte imRes, int inut_read, int inut_write);

/**
 * \ingroup geom2D
 * \fn void RST_transform_inverse_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes, float inut_read, float inut_write);
 * \brief function of calculation of RST matrix by a inverse approach (float image)
 *
 * \param matrice_RST is the RST matrix
 * \param imInit is the initial float image to transform
 * \param imRes is the transformed float image
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 *
 */
void RST_transform_inverse_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes, float inut_read, float inut_write);

/**
 * \ingroup geom2D
 * \fn void RST_transform_invBIS_ppv_u1(matrice_3x3 matrice_RST, EFIDIRImageByte imInit, EFIDIRImageByte imRes, int inut_read, int inut_write);
 * \brief another function of calculation of RST matrix by a inverse approach (8 bytes image)
 *
 * \param matrice_RST is the RST matrix
 * \param imInit is the initial 8 bytes image to transform
 * \param imRes is the transformed 8 bytes image
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 *
 */
void RST_transform_invBIS_ppv_u1(matrice_3x3 matrice_RST, EFIDIRImageByte imInit, EFIDIRImageByte imRes, int inut_read, int inut_write);

/**
 * \ingroup geom2D
 * \fn void RST_transform_invBIS_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes, float inut_read, float inut_write);
 * \brief another function of calculation of RST matrix by a inverse approach (float image)
 *
 * \param matrice_RST is the RST matrix
 * \param imInit is the initial float image to transform
 * \param imRes is the transformed float image
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 *
 */
void RST_transform_invBIS_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat imInit, EFIDIRImageFloat imRes, float inut_read, float inut_write);

/**
 * \ingroup geom2D
 * \fn void RST_transform_LUTrs_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat LUTrs_X, EFIDIRImageFloat LUTrs_Y, float inut_read, float inut_write);
 * \brief modification of LUT taking RST into consideration
 *
 * \param matrice_RST is the RST matrix
 * \param LUTrs_X is the coordonate in X of LUT (radar->ground) 
 * \param LUTrs_Y is the coordonate in Y of LUT (radar->ground) 
 * \param inut_read is the invalid value of input image
 * \param inut_write is the invalid value of output image
 *
 */
void RST_transform_LUTrs_fl(matrice_3x3 matrice_RST, EFIDIRImageFloat LUTrs_X, EFIDIRImageFloat LUTrs_Y, float inut_read, float inut_write);

/**
 * \ingroup geom2D
 * \fn void comatrices(float a[MaxDimEq][MaxDimEq],float c[MaxDimEq][MaxDimEq],int i,int j,int n);
 * \brief function of calculation of sub-comatrix of a squared matrix
 *
 * \param a is the original matrix
 * \param c is the sub-comatrix to calculate
 * \param i is the row number of the sub-comatrix
 * \param j is the column number of the sub-comatrix
 * \param n is the dimension of matrix a
 *
 */
void comatrices(float a[MaxDimEq][MaxDimEq],float c[MaxDimEq][MaxDimEq],int i,int j,int n);


/* *************************  FONCTIONS TRÈS ANNEXES... *********************/
/**
 * \ingroup geom2D
 * \fn void erreur(int code, char *msg);
 * \brief function of management of errors of execution
 *
 * \param code is the value to output during shutdown
 * \param msg is the message to display when errors occur
 *
 */

void erreur(int code, char *msg);

// TRIS
/**
 * \ingroup geom2D
 * \fn void quickSort(float numbers[],int ind[], int array_size);
 * \brief function of fast sorting order
 *
 * \param numbers is a number table
 * \param ind index table
 * \param array_size is the size of table to sort 
 *
 */
void quickSort(float numbers[],int ind[], int array_size);

/**
 * \ingroup geom2D
 * \fn void void q_sort(float numbers[],int ind[], int left, int right);
 * \brief recursive function of fast sorting order
 *
 * \param numbers is a number table
 * \param ind index table
 * \param left buffer parameter
 * \param right buffer parameter
 *
 */
void q_sort(float numbers[],int ind[], int left, int right);

/**
 * \ingroup geom2D
 * \fn void quickSort_simple(int numbers[],int array_size);
 * \brief simple function of fast sorting order
 *
 * \param numbers is a number table
 * \param  array_size is the size of table
 *
 */
void quickSort_simple(int numbers[],int array_size);

/**
 * \ingroup geom2D
 * \fn void q_sort_simple(int numbers[], int left, int right);
 * \brief recursive function of fast sorting order
 *
 * \param numbers is a number table
 * \param left is a buffer parameter
 * \param right is a buffer parameter
 *
 */
void q_sort_simple(int numbers[], int left, int right);

/**
 * \ingroup geom2D
 * \fn int mediane_simple(int numbers[], int array_size);
 * \brief function of calculation of median value of a table of numbers
 *
 * \param numbers is a table of numbers
 * \param array_size is the size of number table
 * \return the median value
 *
 */
int mediane_simple(int numbers[], int array_size);

#endif
