/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLT_CALCULATIONS_GEODETIC_H
#define  SLT_CALCULATIONS_GEODETIC_H

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "efidir_image.h"
#include "slt_geom2D.h"

#define RTerrestre (6366000.0)
#define WGSa (6378137.0)
#define WGSb (6356752.3142)
#define WGSf (1/298.257223563)

/**
 * \ingroup sarlut_operators
 * \defgroup geodetic_tools Some geodetic tools
 */

/**
 * \ingroup geodetic_tools
 * \file slt_calculations_geodetic.h
 * \author  Ivan Petillot and Pascal Bellemain
 * 
 * \brief Some geodetic base tools for sarlut package
 * 
 * file includes (stucture, prototype de fonction) of sarlut package
 * 
 * 
 */

/**
 * \ingroup geodetic_tools
 * \struct TLatLong
 *
 * \brief structure that contains the LAT/LON coordonate in radian
 *
 */

typedef struct {
	double lat;
	double lon;
} TLatLong;


/**
 * \ingroup geodetic_tools
 * \struct TLatLong_dms
 *
 * \brief structure that contains the LAT/LON coordonate in degree-minite-second
 *
 */
typedef struct {
	short int lat_d;
	short int lat_m;
	float lat_s;
	short int lon_d;
	short int lon_m;
	float lon_s;
	char orientation[2];
} TLatLong_dms;

/**
 * \ingroup geodetic_tools
 * \struct TMapInfo
 *
 * \brief structure that contains the mapinfo elements in image header
 *
 */
typedef struct {
	double lon;
	double lat;
	double size_lon;
	double size_lat;
	double rot; // en degrés
} TMapInfo;

/**
 * \ingroup geodetic_tools
 * \struct TMapFile
 *
 * \brief structure that contains the mapfile information
 *
 */
typedef struct {
	double pas_lon;
	double rot_lon;
	double rot_lat;
	double pas_lat;
	double lon;
	double lat;
} TMapFile;


void lldeg2llrad(TLatLong *ll);
/**
 * \ingroup geodetic_tools
 * \fn int MapInfo2MapFile(TMapInfo mapinfo, TMapFile *mapfile);
 * \brief function of generation of mapfile from mapinfo
 * Function allowing to generate a mapfile from the coordonates of mapinfo 
 * in image header
 *
 * \param mapinfo is the geocode information in image header
 * \param mapfile is the geomap file used in sarlut package
 * \return 0 if no error occurs
 *
 */
int MapInfo2MapFile(TMapInfo mapinfo, TMapFile *mapfile);

/**
 * \ingroup geodetic_tools
 * \fn int MapFile2MapInfo(TMapFile mapfile, TMapInfo *mapinfo);
 * \brief function of generation of mapinfo from mapfile
 * Function allowing to generate mapinfo in image header from the mapfile
 * generated by sarlut package
 *
 * \param mapfile is the geomap file used in sarlut package
 * \param mapinfo is the geocode information in image header
 * \return 0 if no error occurs
 *
 */
int MapFile2MapInfo(TMapFile mapfile, TMapInfo *mapinfo);

/**
 * \ingroup geodetic_tools
 * \fn coord_3d llrad2llcart(TLatLong ll, float h, float a, float b);
 * \brief function of transformation of coordonate
 * Function allowing to convert LAT/LON coordonates of a point on the ellipsoid 
 * in the cartesian reference. IGN ALG0009 algorithm.
 *
 * \param ll is the  LAT/LON coordonates structure
 * \param h is the height above the ellipsoid
 * \param a is the semi-major axis of the ellipsoid
 * \param b is the first eccentricity of the ellipsoid
 * \return A coordonate 3D of a point in the cartesian reference
 *
 */
coord_3d llrad2llcart(TLatLong ll, float h, float a, float b);

/**
 * \ingroup geodetic_tools
 * \fn TLatLong lldms2llrad(TLatLong_dms lldms);
 * \brief function of transformation of degree-minute-second to radian
 * Function allowing to convert LAT/LON (degree-minute-second) to radian
 *
 * \param lldms is the coordonates in degree-minite-second format
 * \return A coordonate LAT/LON in radian
 *
 */
TLatLong lldms2llrad(TLatLong_dms lldms);



/* *************************  DISTANCES ENTRE DEUX POINTS  *******************************/
/**
 * \ingroup geodetic_tools
 * \fn double dist_vincenty(TLatLong p1, TLatLong p2, float a, float b, float f);
 * \brief function of calculation of the distance between 2 points
 * Calculates the distance between 2 points specified by LAT/LON
 * and located on an ellipsoid semi-major axis a and semi minor axis b,
 * using the Vincenty inverse formula for ellipsoid
 *
 * \param p1 is the LAT/LON coordonate of the first point
 * \param p2 is the LAT/LON coordonate of the second point
 * \param a is the semi-major axis of the ellipsoid
 * \param b is the first eccentricity of the ellipsoid
 * \param f is the buffer parameter
 * \return the distance between p1 and p2
 *
 */
double dist_vincenty(TLatLong p1, TLatLong p2, float a, float b, float f);

/**
 * \ingroup geodetic_tools
 * \fn double dist_orthodromique_metr(float rayon, TLatLong A, TLatLong B);
 * \brief function of calculation of the circle distance between 2 points
 * Calculates the circle distance (m) between 2 points specified by LAT/LON
 * and located on a sphere.
 * ortho(A,B)=Rarccos(cos[lat(A)].cos[lat(B)].cos[long]+sin[lat(A)].sin[lat(B)] )
 *
 * \param rayon is the radius of the earth
 * \param A is the first point in LAT/LON
 * \param B is the second point in LAT/LON
 * \return the circle distance between p1 and p2
 *
 */
double dist_orthodromique_metr(float rayon, TLatLong A, TLatLong B);

/**
 * \ingroup geodetic_tools
 * \fn double dist_orthodromique_angl(TLatLong A, TLatLong B);
 * \brief function of calculation of the circle distance between 2 points
 * Calculates the circle distance (radian) between 2 points specified by LAT/LON
 * and located on a sphere.
 * ortho(A,B)=Rarccos(cos[lat(A)].cos[lat(B)].cos[long]+sin[lat(A)].sin[lat(B)] )
 *
 * \param A is the first point in LAT/LON
 * \param B is the second point in LAT/LON
 * \return the circle distance between p1 and p2
 *
 */
double dist_orthodromique_angl(TLatLong A, TLatLong B);

/**
 * \ingroup geodetic_tools
 * \fn double dist_orthodromique_mill(TLatLong A, TLatLong B);
 * \brief function of calculation of the circle distance between 2 points
 * Calculates the circle distance (nautical miles) 
 * between 2 points specified by LAT/LON and located on a sphere.
 * ortho(A,B)=60arccos(cos[lat(A)].cos[lat(B)].cos[long]+sin[lat(A)].sin[lat(B)] )
 *
 * \param A is the first point in LAT/LON
 * \param B is the second point in LAT/LON
 * \return the circle distance between p1 and p2
 *
 */
double dist_orthodromique_mill(TLatLong A, TLatLong B);

/**
 * \ingroup geodetic_tools
 * \fn double double dist_segment(float rayon, TLatLong A, TLatLong B);
 * \brief function of calculation of the circle distance between 2 points
 * Calculates the length of the segment between 2 points specified by LAT/LON 
 * and located on a sphere.
 * AB = R. [ .(1-cos[ lat(A)].cos [lat(B)].cos [long]-sin [lat(A)] .sin [ lat(B)] ) ] ^ (1/2)
 *
 * \param rayon is the radius of the earth
 * \param A is the first point in LAT/LON
 * \param B is the second point in LAT/LON
 * \return the circle distance between p1 and p2
 *
 */

double dist_segment(float rayon, TLatLong A, TLatLong B);


/* *************************  CALCULS DIVERS  *******************************/
/**
 * \ingroup geodetic_tools
 * \fn double rayon_terrestre(TLatLong A, float a, float b);
 * \brief function of calculation of radius of the earth for a given point
 * Calculates the Earth radius at point A, i.e the radius of the sphere locally 
 * approximating the ellipsoid to point A. 

 * R = Re * (1 - a * sin ^ 2 (q))
 * where q is the latitude of point A, is to flatten the ellipsoid and Re = 6378.137 km
 * equatorial radius.

 * or with Pythagoras: Rt ^ 2 = x ^ 2 + y ^ 2
 * x = 1 / 2 major axis * cos (lata)
 * y = 1 / 2 small ax * sin (lata)

 * if a and b = 0 then by default it takes the a and b of WGS-84
 *
 * \param A is the point in LAT/LON
 * \param a is the semi-major axis
 * \param b is the semi-minor axis
 * \return the circle distance between p1 and p2
 *
 */
double rayon_terrestre(TLatLong A, float a, float b);

/**
 * \ingroup geodetic_tools
 * \fn void gen_igroundatitudes(char *lat_orientation, int nblig, int nbcol, EFIDIRImageFloat imLAT, double lat_orig, double pas_lat);
 * \brief function of generation of an image of latitude, constant on the same line
 *
 * \param lat_orientation N for Nord, S for Sud
 * \param nblig is the number of lines
 * \param nbcol is the number of columns
 * \param imLAT is the image of latitude to output 
 * \param lat_orig is the latitude of the first pixel
 * \param pas_lat is the posting of latitude
 *
 */
void gen_igroundatitudes(char *lat_orientation, int nblig, int nbcol, EFIDIRImageFloat imLAT, double lat_orig, double pas_lat);

/**
 * \ingroup geodetic_tools
 * \fn void gen_igroundongitudes(int nblig, int nbcol, EFIDIRImageFloat imLON, double lon_orig, double pas_lon);
 * \brief function of generation of an image of longitude constant on the same column
 *
 * \param nblig is the number of lines
 * \param nbcol is the number of columns
 * \param imLON is the image of longitude to output 
 * \param lon_orig is the latitude of the first pixel
 * \param pas_lon is the posting of latitude
 *
 */
void gen_igroundongitudes(int nblig, int nbcol, EFIDIRImageFloat imLON, double lon_orig, double pas_lon);

/**
 * \ingroup geodetic_tools
 * \fn void gen_latitudes(char *lat_orientation, int nblig, int nbcol, EFIDIRImageFloat imLATrot, double lat_orig, double pas_lat, double rot);
 * \brief function of generation of an image of latitude with a rotation
 *
 * \param lat_orientation N for Nord, S for Sud
 * \param nblig is the number of lines
 * \param nbcol is the number of columns
 * \param imLATrot is the image of latitude to output 
 * \param lat_orig is the latitude of the first pixel
 * \param pas_lat is the posting of latitude
 * \param rot is the rotation angle
 *
 */
void gen_latitudes(char *lat_orientation, int nblig, int nbcol, EFIDIRImageFloat imLATrot, double lat_orig, double pas_lat, double rot);

/**
 * \ingroup geodetic_tools
 * \fn void gen_igroundongitudes(int nblig, int nbcol, EFIDIRImageFloat imLON, double lon_orig, double pas_lon);
 * \brief function of generation of an image of longitude with the rotation
 *
 * \param nblig is the number of lines
 * \param nbcol is the number of columns
 * \param imLON is the image of longitude to output 
 * \param lon_orig is the latitude of the first pixel
 * \param pas_lon is the posting of latitude
 *
 */
void gen_longitudes(int nblig, int nbcol, EFIDIRImageFloat imLONrot, double lon_orig, double pas_lon, double rot);


#endif
