/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLT_BASE_LINE_H
#define SLT_BASE_LINE_H

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <string.h>
#include "efidir_allocation.h"
#include "slt_calculations_geodetic.h"
#include "slt_parser_tools.h"
#include "efidir_math.h"
#include "slt_geom2D.h"
#include "slt_orbit_calculations.h"
#include "efidir_assert.h"

/**
 * \ingroup sarlut_operators
 * \defgroup orbit_computations Some tools for orbit computations
 */

/**
 * \ingroup orbit_computations
 * \file slt_base_line.h
 * \author  Diana Rosu
 *
 * \brief Some tools that should be used for orbit computations and for operations over one or more orbits
 *
 * file includes (stucture, prototype de fonction) of sarlut package
 */

/**
 * \ingroup orbit_computations
 * \struct Baselines_and_incidence_angles_s
 * \brief This structure holds the values of both baseline and orthogonal baseline and also the master and slave incidence angles
 */
typedef struct Baselines_and_incidence_angles_s{

	double baseline; /* The baseline between the master orbit and the slave orbit */
	double orthogonal_bl; /* The orthogonal baseline between the master orbit and the slave orbit */
	double incidence_angle_master; /* The incidence angle for the master orbit */
	double incidence_angle_slave; /* The incidence angle for the slave orbit */

} baselines_and_incidence_angles, *Baselines_and_incidence_angles;

/**
 * \ingroup orbit_computations
 * \fn Baselines_and_incidence_angles new_Baselines_and_incidence_angles();
 * \brief Function used in order to allocate memory for a Baselines_and_incidence_angles_s structure
 * \return a pointer to a structure which specifies both baseline and orthogonal baseline, and the incidence angles for the master and slave orbits
 */
Baselines_and_incidence_angles new_Baselines_and_incidence_angles();

/**
 * \ingroup orbit_computation
 * \fn double compute_area_heron(double a, double b, double c);
 * \brief This function is used in order to compute the area of a triangle using the Heron's formula, where the Heron's formula is A = sqrt(s(s-a)(s-b)(s-c))
 * \param a is the first side of the triangle
 * \param b is the second side of the triangle
 * \param c is the third side of the triangle
 * \return a double value which is the area of the triangle
 */
double compute_area_heron(double a, double b, double c);

/**
 * \ingroup orbit_computation
 * \fn baselines_and_incidence_angles compute_baselines_and_incidence_angles (coord_3d slave, coord_3d mnt, coord_3d master);
 * \brief This function is used in order to compute the orthogonal baseline
 * \param slave is the structure which contains the coordinates for the first vertex of the triangle
 * \param mnt is the structure which contains the coordinates for the second vertex of the triangle
 * \param master is the structure which contains the coordinates for the third vertex of the triangle
 * \return a double value which is the orthogonal baseline between two trajectories
 */
baselines_and_incidence_angles compute_baselines_and_incidence_angles (coord_3d slave, coord_3d mnt, coord_3d master);

/**
 * \ingroup orbit_computation
 * \fn void gen_orthogonal_base_line_img(int ignored_val, int nof_estim_master, int nof_estim_slave, Estimators estim_master, Estimators estim_slave, EFIDIRImage imMNT, EFIDIRImage orth_bl_img);
 * \brief This function is used in order to generate the orthogonal base line image
 * \param ignored_val is the value in the MNT image that should be ignored. The resulting image will have the same value for the correspondent pixels
 * \param nof_estim_master is the number of estimators for the master orbit
 * \param nof_estim_slave is the number of estimators for the slave orbit
 * \param estim_master is a pointer to a structure which contains the estimators for the master orbit
 * \param estim_slave is a pointer to a structure which contains the estimators for the slave orbit
 * \param imMNT is the MNT image
 * \param orth_bl_img is the resulting image which contains the orthogonal baseline for each point of the mnt.
 */
void gen_orthogonal_base_line_img(int ignored_val, int nof_estim_master, int nof_estim_slave, Estimators estim_master, Estimators estim_slave, EFIDIRImage imMNT, EFIDIRImage orth_bl_img);

/**
 * \ingroup orbit_computation
 * \fn double *distance_master_st_vect_2_slave_orb(int nof_master_state_vect, int nof_slave_estim, State_vectors master_st_vect, Estimators slave_estim);
 * \brief This function is used in order to compute the distance between two orbits
 * \param nof_master_state_vect is the number of state vectors of the master orbit
 * \param nof_slave_estim is the number of estimators for the slave orbit
 * \param master_st_vect is a pointer to a structure which contains the state vectors for the master orbit
 * \param slave_estim is a pointer to a structure which contains the estimators for the slave orbit
 * \return a pointer to a double vector which contains the distances from the state vector of the master orbit to the estimators of the slave orbit
 */
double *distance_master_st_vect_2_slave_orb(int nof_master_state_vect, int nof_slave_estim, State_vectors master_st_vect, Estimators slave_estim);

/**
 * \ingroup orbit_computation
 * \fn void gen_base_line_img(int ignored_val, int nof_estim_master, int nof_estim_slave, Estimators estim_master, Estimators estim_slave, EFIDIRImage imMNT, EFIDIRImage bl_img);
 * \brief This function is used in order to generate the base line image
 * \param ignored_val is the value in the MNT image that should be ignored. The resulting image will have the same value for the correspondent pixels
 * \param nof_estim_master is the number of estimators for the master orbit
 * \param nof_estim_slave is the number of estimators for the slave orbit
 * \param estim_master is a pointer to a structure which contains the estimators for the master orbit
 * \param estim_slave is a pointer to a structure which contains the estimators for the slave orbit
 * \param imMNT is the MNT image
 * \param bl_img is the resulting image which contains the baseline for each point of the mnt.
 */
void gen_base_line_img(int ignored_val, int nof_estim_master, int nof_estim_slave, Estimators estim_master, Estimators estim_slave, EFIDIRImage imMNT, EFIDIRImage bl_img);

#endif /* SLT_BASE_LINE_H */
