/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PAM_PATTERN_LOCALIZATION_H
#define PAM_PATTERN_LOCALIZATION_H

#define MAX_LABEL_VALUE 100000
#define MAX_NUMBER_OF_LABELS 256
#define MAX_CHARACTERS 256

#include "efidir_image.h"
#include "efidir_param.h"
#include <stdlib.h>

/**
 * \ingroup spatpam_operators
 * \defgroup pattern_localization Pattern localization operator
 */

/**
 * \ingroup pattern_localization
 * \file pam_pattern_localization.h
 * \author Nicolas Meger
 * \brief pattern localization operator 
 */

/**
 * \ingroup pattern_localization
 * \struct pattern_localization_param
 *
 * \brief Structure that contains the parameters of the operator
 *
 */

typedef struct Pattern_localization_param
{
  char* pattern_to_be_localized;
}pattern_localization_param, *Pattern_localization_param;

/**
 * \ingroup pattern_localization
 * \fn pattern_localization_param new_pattern_localization_param()
 * \brief Create a new pattern_localization_param reference
 * 
 * \return A new reference on an allocated pattern_localization_param structure
 *
 *
 */

static Pattern_localization_param new_pattern_localization_param();

/**
 * \ingroup pattern_localization
 * \fn free_pattern_localization_param(Pattern_localization_param p)
 *
 * \param p A reference on a pattern_localization_param structure
 *
 * \brief Free an allocated pattern_localization_param reference 
 *
 */

void free_pattern_localization_param(Pattern_localization_param p);

/**
 * \ingroup pattern_localization
 * \fn void define_pattern_localization_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/

void define_pattern_localization_param(char *extra_description);

/**
 * \ingroup pattern_localization
 * \fn pattern_localization_param get_pattern_localization_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A pattern_localization_param that contains the parameters.
 */

Pattern_localization_param get_pattern_localization_param();

/**
 * \ingroup pattern_localization
 * \fn void pattern_localization(Pattern_localization_param p, FILE* preprocessed_data_pointer, EFIDIRImage outputImageOcc, EFIDIRImage outputNormalizedImageStart, EFIDIRImage outputNormalizedImageEnd);
 * \brief Pattern localization operator
 * \param p is the structure containing the parameters, i.e. the pattern to be localized (example: 1,2,1)
 * \param preprocessed_data_pointer points to the preprocessed data file (of the input SITS, see SITS_quantization operator)
 * \param outputImageOcc is a spatio-temporal localization of the pattern. Each ith bit of the 32 bits allocated to a pixel value is set to 1 if a symbol of the localized pattern occurs in the ith image (the first occurrence is considered). This temporal representation thus works for 32 images as a maximum.
 * \param outputNormalizedImageStart is a spatio-temporal localization of the pattern. Pixel values give the occurrence date of the image in which the pattern starts.
 * \param outputNormalizedImageEnd is a spatio-temporal localization of the pattern. Pixel values give the occurrence date of the image in which the pattern ends.
 */

void pattern_localization(Pattern_localization_param p, FILE* preprocessed_data_pointer, EFIDIRImage outputImageOcc, EFIDIRImage outputNormalizedImageStart, EFIDIRImage outputNormalizedImageEnd);

#endif /* PAM_PATTERN_LOCALIZATION_H */
