/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

#ifndef NSB_DELAYZ_ERAI_H
#define NSB_DELAYZ_ERAI_H

#include "math.h"
#include "efidir_matrix.h"
#include "efidir_goodies.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_assert.h"
#include "efidir_math.h"
#include <stdlib.h> 
#include <stdio.h>
#include <string.h>
#include <math.h>




/**
 * \ingroup nsbas_operators
 * \defgroup delay_erai Delay as a function of z  from ERAI datas
 */
/**
 * \ingroup delay_erai
 * \typedef delay_erai_param
 * \author Sophie Martz (CNRS-ENS)
 *
 * \brief reference on delay_erai_param structure
 *
 */
/**
 * \ingroup delay_erai
 * \struct delay_erai_param
 *
 * \brief structure that contains the parameters of the operator delay
 *
 */
typedef struct struct_delay_erai_param{
  unsigned int *date;
  float **temp;  /*! temperature */
  float **rh;    /*! relative humidity*/
  float **hgt;  /*! geopotential */
  int d;        /*nber of dates in the input files*/
}delay_erai_param, *Delay_erai_param;
/**
 * \ingroup delay_erai
 * \fn new_delay_erai_param()
 *
 * \return A new reference on allocated delay_erai_param structure
 *
 * \brief Create a new Delay_erai_param reference 
 *
 */
Delay_erai_param new_delay_erai_param();
/**
 * \ingroup delay_erai
 * \fn free_delay_erai_param(Delay_erai_param p)
 *
 * \param p A reference on a delay_erai_param structure
 *
 * \brief Free an allocated Delay_erai_param reference 
 *
 */
void free_delay_erai_param(Delay_erai_param p);

/**
 * \ingroup delay_erai
 * \fn void define_delay_erai_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_delay_erai_param(char *extra_description);
/**
 * \ingroup delay_erai
 * \fn Delay_erai_param get_delay_erai_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Delay_erai_param that contains the parameters.
 */
Delay_erai_param get_delay_erai_param();

/*
 * \ingroup delay_erai
 * \fn void delay_erai(Delay_erai_param p, FILE *dry_delay, FILE *hum_delay, FILE *tot_delay);
 * \brief Delay_erai
 * the delay_erai operator computes the electromagnetic delay using input datas (temperature, pressure and geopotential) at decreasing pressure levels
 *
 * \param p is the structure of the parameters (struct Delay_erai_param) for the operator with 4 tables: one with dates, second with temperature, third with geopotential and fourth with relative humidity and the number of dates
 *
 */
void delay_erai(Delay_erai_param p, FILE *dry_delay, FILE *hum_delay, FILE *tot_delay);

void integr_hum1(float z1, float z2,float e1, float e2, float T1, float T2, float value);

void integr_hum2(float z1, float z2, float e1, float e2,float T1, float T2, float value);

#endif
