/*  Copyright (c) 2008. The EFIDIR team. All right reserved.
* 
*  This file is part of EFIDIR tools.
* 
*  EFIDIR tool(s) is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
* 
*  EFIDIR tool(s) is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
* t
*  You should have received a copy of the GNU General Public License
* along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
* 
*  Any information about the EFIDIR project can be found on www.efidir.fr
*/


#ifndef MGS_PS_LIST_H
#define MGS_PS_LIST_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_math.h"
#include "efidir_matrix.h"
#include "efidir_image_list.h"
#include "efidir_param.h"
#include "com_statistics.h"
#include "com_temporal_stat.h"
#include "com_cal_slc.h"

/**
 * \ingroup magnicos_operators
 * \defgroup ps_list Select PS candidates based on their amplitude dispersion index or based on their log moment of order 2
 */
/**
 * \ingroup ps_list
 * \author Sophie Martz (CNRS)
 *
 */
/**
 * \ingroup ps_list
 * \struct Ps_list_param
 *
 * \brief structure that contains the parameters of the operator ps_list 
 *
 */
typedef struct Ps_list_param{
  int op;
  float threshold_da; /*!< threshold parameter on amplitude dispersion index to produce the PS list */
  float threshold_ln; /*!< threshold parameter on the log moment of order 2 to produce the PS list */
  float threshold_amp; /*!< threshold parameter on amplitude, to eliminate lines in SLCs without data */
  float *coeff_calib;
  //FILE *fps;   /*! file with ps coordinates */
}ps_list_param, *Ps_list_param;

/**
 * \ingroup ps_list
 * \fn new_ps_list_param()
 *
 * \return A new reference on allocated ps_list_param structure
 *
 * \brief Create a new ps_list_param reference 
 *
 */
Ps_list_param new_ps_list_param();

/**
 * \ingroup ps_list
 * \fn free_ps_list_param(Ps_list_param p)
 *
 * \param p A reference on a ps_list_param structure
 *
 * \brief Free an allocated ps_list_param reference 
 *
 */
void free_ps_list_param(Ps_list_param p);

/**
 * \ingroup ps_list
 * \fn void define_ps_list_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_ps_list_param(char *extra_description);

/**
 * \ingroup ps_list
 * \fn Ps_list_param get_ps_list_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A ps_list_param that contains the parameters.
 */
Ps_list_param get_ps_list_param();

/**
 * \ingroup ps_list
 * \fn void ps_list(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage);
 * \brief Ps_list analyse a list of SLC images to produce a list of PS candidates
 * 
 * Ps_list reads a list of SLC images (with full path), compute their average amplitude,
 * and define for each pixel the amplitude dispersion index and the log moment of order 2
 *
 * \param p is the structure of the parameters for operators
 * \param inputListImage images list on which the PS are selected
 * \param outputImage is a four band float image containing (a) the average amplitude, (b) the pixel mask, (c) the Da, (d) the log moment of order 2
 *
 */
//void ps_list(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage, EFIDIRStat *imagesStat);

//void ps_list(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage);
void ps_list(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage);

//void amp_Da_map(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage);

//void amp_Ln_map(Ps_list_param p, EFIDIRListImage inputListImage, EFIDIRImage outputImage);



#endif
